'use strict';

Object.defineProperty(exports, "__esModule", {
    value: true
});

var _di = require('@glimmer/di');

var _debug = require('./utils/debug');

function _classCallCheck(instance, Constructor) {
    if (!(instance instanceof Constructor)) {
        throw new TypeError("Cannot call a class as a function");
    }
}

var Resolver = function () {
    function Resolver(config, registry) {
        _classCallCheck(this, Resolver);

        this.config = config;
        this.registry = registry;
    }

    Resolver.prototype.identify = function identify(specifier, referrer) {
        if ((0, _di.isSpecifierStringAbsolute)(specifier)) {
            return specifier;
        }
        var s = (0, _di.deserializeSpecifier)(specifier);
        var result = void 0;
        if (referrer) {
            var r = (0, _di.deserializeSpecifier)(referrer);
            if ((0, _di.isSpecifierObjectAbsolute)(r)) {
                (0, _debug.assert)('Specifier must not include a rootName, collection, or namespace when combined with an absolute referrer', s.rootName === undefined && s.collection === undefined && s.namespace === undefined);
                // Look locally in the referrer's namespace
                s.rootName = r.rootName;
                s.collection = r.collection;
                if (s.name) {
                    s.namespace = r.namespace ? r.namespace + '/' + r.name : r.name;
                } else {
                    s.namespace = r.namespace;
                    s.name = r.name;
                }
                if (result = this._serializeAndVerify(s)) {
                    return result;
                }
                // Look for a private collection in the referrer's namespace
                var privateCollection = this._definitiveCollection(s.type);
                if (privateCollection) {
                    s.namespace += '/-' + privateCollection;
                    if (result = this._serializeAndVerify(s)) {
                        return result;
                    }
                }
                // Because local and private resolution has failed, clear all but `name` and `type`
                // to proceed with top-level resolution
                s.rootName = s.collection = s.namespace = undefined;
            } else {
                (0, _debug.assert)('Referrer must either be "absolute" or include a `type` to determine the associated type', r.type);
                // Look in the definitive collection for the associated type
                s.collection = this._definitiveCollection(r.type);
                (0, _debug.assert)('\'' + r.type + '\' does not have a definitive collection', s.collection);
            }
        }
        // If the collection is unspecified, use the definitive collection for the `type`
        if (!s.collection) {
            s.collection = this._definitiveCollection(s.type);
            (0, _debug.assert)('\'' + s.type + '\' does not have a definitive collection', s.collection);
        }
        if (!s.rootName) {
            // If the root name is unspecified, try the app's `rootName` first
            s.rootName = this.config.app.rootName || 'app';
            if (result = this._serializeAndVerify(s)) {
                return result;
            }
            // Then look for an addon with a matching `rootName`
            var addonDef = void 0;
            if (s.namespace) {
                addonDef = this.config.addons && this.config.addons[s.namespace];
                s.rootName = s.namespace;
                s.namespace = undefined;
            } else {
                addonDef = this.config.addons && this.config.addons[s.name];
                s.rootName = s.name;
                s.name = 'main';
            }
        }
        if (result = this._serializeAndVerify(s)) {
            return result;
        }
    };

    Resolver.prototype.retrieve = function retrieve(specifier) {
        return this.registry.get(specifier);
    };

    Resolver.prototype.resolve = function resolve(specifier, referrer) {
        var id = this.identify(specifier, referrer);
        if (id) {
            return this.retrieve(id);
        }
    };

    Resolver.prototype._definitiveCollection = function _definitiveCollection(type) {
        var typeDef = this.config.types[type];
        (0, _debug.assert)('\'' + type + '\' is not a recognized type', typeDef);
        return typeDef.definitiveCollection;
    };

    Resolver.prototype._serializeAndVerify = function _serializeAndVerify(specifier) {
        var serialized = (0, _di.serializeSpecifier)(specifier);
        if (this.registry.has(serialized)) {
            return serialized;
        }
    };

    return Resolver;
}();

exports.default = Resolver;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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