import { isSpecifierStringAbsolute, isSpecifierObjectAbsolute, deserializeSpecifier, serializeSpecifier } from '@glimmer/di';
import { assert } from './utils/debug';
export default class Resolver {
    constructor(config, registry) {
        this.config = config;
        this.registry = registry;
    }
    identify(specifier, referrer) {
        if (isSpecifierStringAbsolute(specifier)) {
            return specifier;
        }
        let s = deserializeSpecifier(specifier);
        let result;
        if (referrer) {
            let r = deserializeSpecifier(referrer);
            if (isSpecifierObjectAbsolute(r)) {
                assert('Specifier must not include a rootName, collection, or namespace when combined with an absolute referrer', s.rootName === undefined && s.collection === undefined && s.namespace === undefined);
                // Look locally in the referrer's namespace
                s.rootName = r.rootName;
                s.collection = r.collection;
                if (s.name) {
                    s.namespace = r.namespace ? r.namespace + '/' + r.name : r.name;
                }
                else {
                    s.namespace = r.namespace;
                    s.name = r.name;
                }
                if (result = this._serializeAndVerify(s)) {
                    return result;
                }
                // Look for a private collection in the referrer's namespace
                let privateCollection = this._definitiveCollection(s.type);
                if (privateCollection) {
                    s.namespace += '/-' + privateCollection;
                    if (result = this._serializeAndVerify(s)) {
                        return result;
                    }
                }
                // Because local and private resolution has failed, clear all but `name` and `type`
                // to proceed with top-level resolution
                s.rootName = s.collection = s.namespace = undefined;
            }
            else {
                assert('Referrer must either be "absolute" or include a `type` to determine the associated type', r.type);
                // Look in the definitive collection for the associated type
                s.collection = this._definitiveCollection(r.type);
                assert(`'${r.type}' does not have a definitive collection`, s.collection);
            }
        }
        // If the collection is unspecified, use the definitive collection for the `type`
        if (!s.collection) {
            s.collection = this._definitiveCollection(s.type);
            assert(`'${s.type}' does not have a definitive collection`, s.collection);
        }
        if (!s.rootName) {
            // If the root name is unspecified, try the app's `rootName` first
            s.rootName = this.config.app.rootName || 'app';
            if (result = this._serializeAndVerify(s)) {
                return result;
            }
            // Then look for an addon with a matching `rootName`
            let addonDef;
            if (s.namespace) {
                addonDef = this.config.addons && this.config.addons[s.namespace];
                s.rootName = s.namespace;
                s.namespace = undefined;
            }
            else {
                addonDef = this.config.addons && this.config.addons[s.name];
                s.rootName = s.name;
                s.name = 'main';
            }
        }
        if (result = this._serializeAndVerify(s)) {
            return result;
        }
    }
    retrieve(specifier) {
        return this.registry.get(specifier);
    }
    resolve(specifier, referrer) {
        let id = this.identify(specifier, referrer);
        if (id) {
            return this.retrieve(id);
        }
    }
    _definitiveCollection(type) {
        let typeDef = this.config.types[type];
        assert(`'${type}' is not a recognized type`, typeDef);
        return typeDef.definitiveCollection;
    }
    _serializeAndVerify(specifier) {
        let serialized = serializeSpecifier(specifier);
        if (this.registry.has(serialized)) {
            return serialized;
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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