function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

import { isSpecifierStringAbsolute, isSpecifierObjectAbsolute, deserializeSpecifier, serializeSpecifier } from '@glimmer/di';
import { assert } from './utils/debug';

var Resolver = function () {
    function Resolver(config, registry) {
        _classCallCheck(this, Resolver);

        this.config = config;
        this.registry = registry;
    }

    Resolver.prototype.identify = function identify(specifier, referrer) {
        if (isSpecifierStringAbsolute(specifier)) {
            return specifier;
        }
        var s = deserializeSpecifier(specifier);
        var result = void 0;
        if (referrer) {
            var r = deserializeSpecifier(referrer);
            if (isSpecifierObjectAbsolute(r)) {
                assert('Specifier must not include a rootName, collection, or namespace when combined with an absolute referrer', s.rootName === undefined && s.collection === undefined && s.namespace === undefined);
                // Look locally in the referrer's namespace
                s.rootName = r.rootName;
                s.collection = r.collection;
                if (s.name) {
                    s.namespace = r.namespace ? r.namespace + '/' + r.name : r.name;
                } else {
                    s.namespace = r.namespace;
                    s.name = r.name;
                }
                if (result = this._serializeAndVerify(s)) {
                    return result;
                }
                // Look for a private collection in the referrer's namespace
                var privateCollection = this._definitiveCollection(s.type);
                if (privateCollection) {
                    s.namespace += '/-' + privateCollection;
                    if (result = this._serializeAndVerify(s)) {
                        return result;
                    }
                }
                // Because local and private resolution has failed, clear all but `name` and `type`
                // to proceed with top-level resolution
                s.rootName = s.collection = s.namespace = undefined;
            } else {
                assert('Referrer must either be "absolute" or include a `type` to determine the associated type', r.type);
                // Look in the definitive collection for the associated type
                s.collection = this._definitiveCollection(r.type);
                assert('\'' + r.type + '\' does not have a definitive collection', s.collection);
            }
        }
        // If the collection is unspecified, use the definitive collection for the `type`
        if (!s.collection) {
            s.collection = this._definitiveCollection(s.type);
            assert('\'' + s.type + '\' does not have a definitive collection', s.collection);
        }
        if (!s.rootName) {
            // If the root name is unspecified, try the app's `rootName` first
            s.rootName = this.config.app.rootName || 'app';
            if (result = this._serializeAndVerify(s)) {
                return result;
            }
            // Then look for an addon with a matching `rootName`
            var addonDef = void 0;
            if (s.namespace) {
                addonDef = this.config.addons && this.config.addons[s.namespace];
                s.rootName = s.namespace;
                s.namespace = undefined;
            } else {
                addonDef = this.config.addons && this.config.addons[s.name];
                s.rootName = s.name;
                s.name = 'main';
            }
        }
        if (result = this._serializeAndVerify(s)) {
            return result;
        }
    };

    Resolver.prototype.retrieve = function retrieve(specifier) {
        return this.registry.get(specifier);
    };

    Resolver.prototype.resolve = function resolve(specifier, referrer) {
        var id = this.identify(specifier, referrer);
        if (id) {
            return this.retrieve(id);
        }
    };

    Resolver.prototype._definitiveCollection = function _definitiveCollection(type) {
        var typeDef = this.config.types[type];
        assert('\'' + type + '\' is not a recognized type', typeDef);
        return typeDef.definitiveCollection;
    };

    Resolver.prototype._serializeAndVerify = function _serializeAndVerify(specifier) {
        var serialized = serializeSpecifier(specifier);
        if (this.registry.has(serialized)) {
            return serialized;
        }
    };

    return Resolver;
}();

export default Resolver;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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