"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const bounds_1 = require("../bounds");
const helper_1 = require("../dom/helper");
const util_1 = require("@glimmer/util");
exports.SVG_NAMESPACE = 'http://www.w3.org/2000/svg';
// Patch:    insertAdjacentHTML on SVG Fix
// Browsers: Safari, IE, Edge, Firefox ~33-34
// Reason:   insertAdjacentHTML does not exist on SVG elements in Safari. It is
//           present but throws an exception on IE and Edge. Old versions of
//           Firefox create nodes in the incorrect namespace.
// Fix:      Since IE and Edge silently fail to create SVG nodes using
//           innerHTML, and because Firefox may create nodes in the incorrect
//           namespace using innerHTML on SVG elements, an HTML-string wrapping
//           approach is used. A pre/post SVG tag is added to the string, then
//           that whole string is added to a div. The created nodes are plucked
//           out and applied to the target location on DOM.
function domChanges(document, DOMChangesClass, svgNamespace) {
    if (!document)
        return DOMChangesClass;
    if (!shouldApplyFix(document, svgNamespace)) {
        return DOMChangesClass;
    }
    let div = document.createElement('div');
    return class DOMChangesWithSVGInnerHTMLFix extends DOMChangesClass {
        insertHTMLBefore(parent, nextSibling, html) {
            if (html === null || html === '') {
                return super.insertHTMLBefore(parent, nextSibling, html);
            }
            if (parent.namespaceURI !== svgNamespace) {
                return super.insertHTMLBefore(parent, nextSibling, html);
            }
            return fixSVG(parent, div, html, nextSibling);
        }
    };
}
exports.domChanges = domChanges;
function treeConstruction(document, TreeConstructionClass, svgNamespace) {
    if (!document)
        return TreeConstructionClass;
    if (!shouldApplyFix(document, svgNamespace)) {
        return TreeConstructionClass;
    }
    let div = document.createElement('div');
    return class TreeConstructionWithSVGInnerHTMLFix extends TreeConstructionClass {
        insertHTMLBefore(parent, html, reference) {
            if (html === null || html === '') {
                return super.insertHTMLBefore(parent, html, reference);
            }
            if (parent.namespaceURI !== svgNamespace) {
                return super.insertHTMLBefore(parent, html, reference);
            }
            return fixSVG(parent, div, html, reference);
        }
    };
}
exports.treeConstruction = treeConstruction;
function fixSVG(parent, div, html, reference) {
    // IE, Edge: also do not correctly support using `innerHTML` on SVG
    // namespaced elements. So here a wrapper is used.
    let wrappedHtml = '<svg>' + html + '</svg>';
    div.innerHTML = wrappedHtml;
    let [first, last] = helper_1.moveNodesBefore(div.firstChild, parent, reference);
    return new bounds_1.ConcreteBounds(parent, first, last);
}
function shouldApplyFix(document, svgNamespace) {
    let svg = document.createElementNS(svgNamespace, 'svg');
    try {
        svg['insertAdjacentHTML']('beforeEnd', '<circle></circle>');
    }
    catch (e) {
        // IE, Edge: Will throw, insertAdjacentHTML is unsupported on SVG
        // Safari: Will throw, insertAdjacentHTML is not present on SVG
    }
    finally {
        // FF: Old versions will create a node in the wrong namespace
        if (svg.childNodes.length === 1 && util_1.unwrap(svg.firstChild).namespaceURI === exports.SVG_NAMESPACE) {
            // The test worked as expected, no fix required
            return false;
        }
        return true;
    }
}
//# sourceMappingURL=data:application/json;base64,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