"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const references_1 = require("../../references");
const utils_1 = require("../../utils");
const reference_1 = require("@glimmer/reference");
const util_1 = require("@glimmer/util");
class CompiledNamedArgs {
    constructor(keys, values) {
        this.keys = keys;
        this.values = values;
        this.length = keys.length;
        util_1.assert(keys.length === values.length, 'Keys and values do not have the same length');
    }
    static empty() {
        return exports.COMPILED_EMPTY_NAMED_ARGS;
    }
    static create(map) {
        let keys = Object.keys(map);
        let length = keys.length;
        if (length > 0) {
            let values = [];
            for (let i = 0; i < length; i++) {
                values[i] = map[keys[i]];
            }
            return new this(keys, values);
        }
        else {
            return exports.COMPILED_EMPTY_NAMED_ARGS;
        }
    }
    evaluate(vm) {
        let { keys, values, length } = this;
        let evaluated = new Array(length);
        for (let i = 0; i < length; i++) {
            evaluated[i] = values[i].evaluate(vm);
        }
        return new EvaluatedNamedArgs(keys, evaluated);
    }
    toJSON() {
        let { keys, values } = this;
        let inner = keys.map((key, i) => `${key}: ${values[i].toJSON()}`).join(", ");
        return `{${inner}}`;
    }
}
exports.CompiledNamedArgs = CompiledNamedArgs;
exports.COMPILED_EMPTY_NAMED_ARGS = new (class extends CompiledNamedArgs {
    constructor() {
        super(utils_1.EMPTY_ARRAY, utils_1.EMPTY_ARRAY);
    }
    evaluate(_vm) {
        return exports.EVALUATED_EMPTY_NAMED_ARGS;
    }
    toJSON() {
        return `<EMPTY>`;
    }
});
class EvaluatedNamedArgs {
    constructor(keys, values, _map = null) {
        this.keys = keys;
        this.values = values;
        this._map = _map;
        this.tag = reference_1.combineTagged(values);
        this.length = keys.length;
        util_1.assert(keys.length === values.length, 'Keys and values do not have the same length');
    }
    static create(map) {
        let keys = Object.keys(map);
        let length = keys.length;
        if (length > 0) {
            let values = new Array(length);
            for (let i = 0; i < length; i++) {
                values[i] = map[keys[i]];
            }
            return new this(keys, values, map);
        }
        else {
            return exports.EVALUATED_EMPTY_NAMED_ARGS;
        }
    }
    static empty() {
        return exports.EVALUATED_EMPTY_NAMED_ARGS;
    }
    get map() {
        let { _map: map } = this;
        if (map) {
            return map;
        }
        map = this._map = util_1.dict();
        let { keys, values, length } = this;
        for (let i = 0; i < length; i++) {
            map[keys[i]] = values[i];
        }
        return map;
    }
    get(key) {
        let { keys, values } = this;
        let index = keys.indexOf(key);
        return (index === -1) ? references_1.UNDEFINED_REFERENCE : values[index];
    }
    has(key) {
        return this.keys.indexOf(key) !== -1;
    }
    value() {
        let { keys, values } = this;
        let out = util_1.dict();
        for (let i = 0; i < keys.length; i++) {
            let key = keys[i];
            let ref = values[i];
            out[key] = ref.value();
        }
        return out;
    }
}
exports.EvaluatedNamedArgs = EvaluatedNamedArgs;
exports.EVALUATED_EMPTY_NAMED_ARGS = new (class extends EvaluatedNamedArgs {
    constructor() {
        super(utils_1.EMPTY_ARRAY, utils_1.EMPTY_ARRAY, utils_1.EMPTY_DICT);
    }
    get() {
        return references_1.UNDEFINED_REFERENCE;
    }
    has(_key) {
        return false;
    }
    value() {
        return utils_1.EMPTY_DICT;
    }
});
//# sourceMappingURL=data:application/json;base64,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