"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const content = require("./content");
const vm = require("./vm");
const has_block_1 = require("../../compiled/expressions/has-block");
const util_1 = require("@glimmer/util");
const functions_1 = require("../../syntax/functions");
const expressions_1 = require("../expressions");
const compiler_1 = require("../../compiler");
class Labels {
    constructor() {
        this.labels = util_1.dict();
        this.jumps = [];
        this.ranges = [];
    }
    label(name, index) {
        this.labels[name] = index;
    }
    jump(at, Target, target) {
        this.jumps.push({ at, target, Target });
    }
    range(at, Range, start, end) {
        this.ranges.push({ at, start, end, Range });
    }
    patch(opcodes) {
        for (let i = 0; i < this.jumps.length; i++) {
            let { at, target, Target } = this.jumps[i];
            opcodes.set(at, Target, this.labels[target]);
        }
        for (let i = 0; i < this.ranges.length; i++) {
            let { at, start, end, Range } = this.ranges[i];
            opcodes.set(at, Range, this.labels[start], this.labels[end] - 1);
        }
    }
}
class BasicOpcodeBuilder {
    constructor(symbolTable, env, program) {
        this.symbolTable = symbolTable;
        this.env = env;
        this.program = program;
        this.labelsStack = new util_1.Stack();
        this.constants = env.constants;
        this.start = program.next;
    }
    get end() {
        return this.program.next;
    }
    get pos() {
        return this.program.current;
    }
    get nextPos() {
        return this.program.next;
    }
    opcode(name, op1, op2, op3) {
        this.push(name, op1, op2, op3);
    }
    push(type, op1 = 0, op2 = 0, op3 = 0) {
        this.program.push(type, op1, op2, op3);
    }
    // helpers
    get labels() {
        return util_1.expect(this.labelsStack.current, 'bug: not in a label stack');
    }
    startLabels() {
        this.labelsStack.push(new Labels());
    }
    stopLabels() {
        let label = util_1.expect(this.labelsStack.pop(), 'unbalanced push and pop labels');
        label.patch(this.program);
    }
    // partials
    putPartialDefinition(_definition) {
        let definition = this.constants.other(_definition);
        this.opcode(50 /* PutPartial */, definition);
    }
    putDynamicPartialDefinition() {
        this.opcode(49 /* PutDynamicPartial */, this.constants.other(this.symbolTable));
    }
    evaluatePartial() {
        this.opcode(51 /* EvaluatePartial */, this.constants.other(this.symbolTable), this.constants.other(util_1.dict()));
    }
    // components
    putComponentDefinition(definition) {
        this.opcode(23 /* PutComponent */, this.other(definition));
    }
    putDynamicComponentDefinition() {
        this.opcode(22 /* PutDynamicComponent */);
    }
    openComponent(args, shadow) {
        this.opcode(24 /* OpenComponent */, this.args(args), shadow ? this.block(shadow) : 0);
    }
    didCreateElement() {
        this.opcode(25 /* DidCreateElement */);
    }
    shadowAttributes() {
        this.opcode(26 /* ShadowAttributes */);
        this.opcode(21 /* CloseBlock */);
    }
    didRenderLayout() {
        this.opcode(27 /* DidRenderLayout */);
    }
    closeComponent() {
        this.opcode(28 /* CloseComponent */);
    }
    // content
    dynamicContent(Opcode) {
        this.opcode(31 /* DynamicContent */, this.other(Opcode));
    }
    cautiousAppend() {
        this.dynamicContent(new content.OptimizedCautiousAppendOpcode());
    }
    trustingAppend() {
        this.dynamicContent(new content.OptimizedTrustingAppendOpcode());
    }
    guardedCautiousAppend(expression) {
        this.dynamicContent(new content.GuardedCautiousAppendOpcode(this.compileExpression(expression), this.symbolTable));
    }
    guardedTrustingAppend(expression) {
        this.dynamicContent(new content.GuardedTrustingAppendOpcode(this.compileExpression(expression), this.symbolTable));
    }
    // dom
    text(text) {
        this.opcode(29 /* Text */, this.constants.string(text));
    }
    openPrimitiveElement(tag) {
        this.opcode(32 /* OpenElement */, this.constants.string(tag));
    }
    openComponentElement(tag) {
        this.opcode(35 /* OpenComponentElement */, this.constants.string(tag));
    }
    openDynamicPrimitiveElement() {
        this.opcode(36 /* OpenDynamicElement */);
    }
    flushElement() {
        this.opcode(37 /* FlushElement */);
    }
    closeElement() {
        this.opcode(38 /* CloseElement */);
    }
    staticAttr(_name, _namespace, _value) {
        let name = this.constants.string(_name);
        let namespace = _namespace ? this.constants.string(_namespace) : 0;
        let value = this.constants.string(_value);
        this.opcode(40 /* StaticAttr */, name, value, namespace);
    }
    dynamicAttrNS(_name, _namespace, trusting) {
        let name = this.constants.string(_name);
        let namespace = this.constants.string(_namespace);
        this.opcode(42 /* DynamicAttrNS */, name, namespace, trusting | 0);
    }
    dynamicAttr(_name, trusting) {
        let name = this.constants.string(_name);
        this.opcode(43 /* DynamicAttr */, name, trusting | 0);
    }
    comment(_comment) {
        let comment = this.constants.string(_comment);
        this.opcode(30 /* Comment */, comment);
    }
    modifier(_name, _args) {
        let args = this.constants.expression(this.compile(_args));
        let _modifierManager = this.env.lookupModifier(_name, this.symbolTable);
        let modifierManager = this.constants.other(_modifierManager);
        let name = this.constants.string(_name);
        this.opcode(41 /* Modifier */, name, modifierManager, args);
    }
    // lists
    putIterator() {
        this.opcode(44 /* PutIterator */);
    }
    enterList(start, end) {
        this.push(45 /* EnterList */);
        this.labels.range(this.pos, 45 /* EnterList */, start, end);
    }
    exitList() {
        this.opcode(46 /* ExitList */);
    }
    enterWithKey(start, end) {
        this.push(47 /* EnterWithKey */);
        this.labels.range(this.pos, 47 /* EnterWithKey */, start, end);
    }
    nextIter(end) {
        this.push(48 /* NextIter */);
        this.labels.jump(this.pos, 48 /* NextIter */, end);
    }
    // vm
    openBlock(_args, _inner) {
        let args = this.constants.expression(this.compile(_args));
        let inner = this.constants.other(_inner);
        this.opcode(20 /* OpenBlock */, inner, args);
    }
    closeBlock() {
        this.opcode(21 /* CloseBlock */);
    }
    pushRemoteElement() {
        this.opcode(33 /* PushRemoteElement */);
    }
    popRemoteElement() {
        this.opcode(34 /* PopRemoteElement */);
    }
    popElement() {
        this.opcode(39 /* PopElement */);
    }
    label(name) {
        this.labels.label(name, this.nextPos);
    }
    pushChildScope() {
        this.opcode(0 /* PushChildScope */);
    }
    popScope() {
        this.opcode(1 /* PopScope */);
    }
    pushDynamicScope() {
        this.opcode(2 /* PushDynamicScope */);
    }
    popDynamicScope() {
        this.opcode(3 /* PopDynamicScope */);
    }
    putNull() {
        this.opcode(4 /* Put */, this.constants.NULL_REFERENCE);
    }
    putValue(_expression) {
        let expr = this.constants.expression(this.compileExpression(_expression));
        this.opcode(5 /* EvaluatePut */, expr);
    }
    putArgs(_args) {
        let args = this.constants.expression(this.compile(_args));
        this.opcode(6 /* PutArgs */, args);
    }
    bindDynamicScope(_names) {
        this.opcode(12 /* BindDynamicScope */, this.names(_names));
    }
    bindPositionalArgs(_names, _symbols) {
        this.opcode(7 /* BindPositionalArgs */, this.names(_names), this.symbols(_symbols));
    }
    bindNamedArgs(_names, _symbols) {
        this.opcode(8 /* BindNamedArgs */, this.names(_names), this.symbols(_symbols));
    }
    bindBlocks(_names, _symbols) {
        this.opcode(9 /* BindBlocks */, this.names(_names), this.symbols(_symbols));
    }
    enter(enter, exit) {
        this.push(13 /* Enter */);
        this.labels.range(this.pos, 13 /* Enter */, enter, exit);
    }
    exit() {
        this.opcode(14 /* Exit */);
    }
    evaluate(_block) {
        let block = this.constants.block(_block);
        this.opcode(15 /* Evaluate */, block);
    }
    test(testFunc) {
        let _func;
        if (testFunc === 'const') {
            _func = vm.ConstTest;
        }
        else if (testFunc === 'simple') {
            _func = vm.SimpleTest;
        }
        else if (testFunc === 'environment') {
            _func = vm.EnvironmentTest;
        }
        else if (typeof testFunc === 'function') {
            _func = testFunc;
        }
        else {
            throw new Error('unreachable');
        }
        let func = this.constants.function(_func);
        this.opcode(19 /* Test */, func);
    }
    jump(target) {
        this.push(16 /* Jump */);
        this.labels.jump(this.pos, 16 /* Jump */, target);
    }
    jumpIf(target) {
        this.push(17 /* JumpIf */);
        this.labels.jump(this.pos, 17 /* JumpIf */, target);
    }
    jumpUnless(target) {
        this.push(18 /* JumpUnless */);
        this.labels.jump(this.pos, 18 /* JumpUnless */, target);
    }
    names(_names) {
        let names = _names.map(n => this.constants.string(n));
        return this.constants.array(names);
    }
    symbols(symbols) {
        return this.constants.array(symbols);
    }
    other(value) {
        return this.constants.other(value);
    }
    args(args) {
        return this.constants.expression(this.compile(args));
    }
    block(block) {
        return this.constants.block(block);
    }
}
exports.BasicOpcodeBuilder = BasicOpcodeBuilder;
function isCompilableExpression(expr) {
    return expr && typeof expr['compile'] === 'function';
}
class OpcodeBuilder extends BasicOpcodeBuilder {
    constructor(symbolTable, env, program = env.program) {
        super(symbolTable, env, program);
        this.component = new compiler_1.ComponentBuilder(this);
    }
    compile(expr) {
        if (isCompilableExpression(expr)) {
            return expr.compile(this);
        }
        else {
            return expr;
        }
    }
    compileExpression(expression) {
        if (expression instanceof expressions_1.CompiledExpression) {
            return expression;
        }
        else {
            return functions_1.expr(expression, this);
        }
    }
    bindPositionalArgsForLocals(locals) {
        let names = Object.keys(locals);
        let symbols = new Array(names.length); //Object.keys(locals).map(name => locals[name]);
        for (let i = 0; i < names.length; i++) {
            symbols[i] = locals[names[i]];
        }
        this.opcode(7 /* BindPositionalArgs */, this.symbols(symbols));
    }
    preludeForLayout(layout) {
        let symbols = layout.symbolTable.getSymbols();
        if (symbols.named) {
            let named = symbols.named;
            let namedNames = Object.keys(named);
            let namedSymbols = namedNames.map(n => named[n]);
            this.opcode(8 /* BindNamedArgs */, this.names(namedNames), this.symbols(namedSymbols));
        }
        this.opcode(11 /* BindCallerScope */);
        if (symbols.yields) {
            let yields = symbols.yields;
            let yieldNames = Object.keys(yields);
            let yieldSymbols = yieldNames.map(n => yields[n]);
            this.opcode(9 /* BindBlocks */, this.names(yieldNames), this.symbols(yieldSymbols));
        }
        if (symbols.partialArgs) {
            this.opcode(10 /* BindPartialArgs */, symbols.partialArgs);
        }
    }
    yield(args, to) {
        let yields, partial;
        let inner;
        if (yields = this.symbolTable.getSymbol('yields', to)) {
            inner = new has_block_1.CompiledGetBlockBySymbol(yields, to);
        }
        else if (partial = this.symbolTable.getPartialArgs()) {
            inner = new has_block_1.CompiledInPartialGetBlock(partial, to);
        }
        else {
            throw new Error('[BUG] ${to} is not a valid block name.');
        }
        this.openBlock(args, inner);
        this.closeBlock();
    }
    // TODO
    // come back to this
    labelled(args, callback) {
        if (args)
            this.putArgs(args);
        this.startLabels();
        this.enter('BEGIN', 'END');
        this.label('BEGIN');
        callback(this, 'BEGIN', 'END');
        this.label('END');
        this.exit();
        this.stopLabels();
    }
    // TODO
    // come back to this
    iter(callback) {
        this.startLabels();
        this.enterList('BEGIN', 'END');
        this.label('ITER');
        this.nextIter('BREAK');
        this.enterWithKey('BEGIN', 'END');
        this.label('BEGIN');
        callback(this, 'BEGIN', 'END');
        this.label('END');
        this.exit();
        this.jump('ITER');
        this.label('BREAK');
        this.exitList();
        this.stopLabels();
    }
    // TODO
    // come back to this
    unit(callback) {
        this.startLabels();
        callback(this);
        this.stopLabels();
    }
}
exports.default = OpcodeBuilder;
//# sourceMappingURL=data:application/json;base64,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