"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const upsert_1 = require("../../upsert");
const interfaces_1 = require("../../component/interfaces");
const opcodes_1 = require("../../opcodes");
const expressions_1 = require("../expressions");
const update_1 = require("../../vm/update");
const reference_1 = require("@glimmer/reference");
const util_1 = require("@glimmer/util");
const bounds_1 = require("../../bounds");
const builder_1 = require("../../builder");
const builder_2 = require("./builder");
const references_1 = require("../../references");
const builder_3 = require("../../builder");
const opcodes_2 = require("../../opcodes");
opcodes_2.APPEND_OPCODES.add(31 /* DynamicContent */, (vm, { op1: append }) => {
    let opcode = vm.constants.getOther(append);
    opcode.evaluate(vm);
});
function isEmpty(value) {
    return value === null || value === undefined || typeof value['toString'] !== 'function';
}
function normalizeTextValue(value) {
    if (isEmpty(value)) {
        return '';
    }
    return String(value);
}
exports.normalizeTextValue = normalizeTextValue;
function normalizeTrustedValue(value) {
    if (isEmpty(value)) {
        return '';
    }
    if (upsert_1.isString(value)) {
        return value;
    }
    if (upsert_1.isSafeString(value)) {
        return value.toHTML();
    }
    if (upsert_1.isNode(value)) {
        return value;
    }
    return String(value);
}
function normalizeValue(value) {
    if (isEmpty(value)) {
        return '';
    }
    if (upsert_1.isString(value)) {
        return value;
    }
    if (upsert_1.isSafeString(value) || upsert_1.isNode(value)) {
        return value;
    }
    return String(value);
}
class AppendDynamicOpcode {
    evaluate(vm) {
        let reference = vm.frame.getOperand();
        let normalized = this.normalize(reference);
        let value, cache;
        if (reference_1.isConst(reference)) {
            value = normalized.value();
        }
        else {
            cache = new reference_1.ReferenceCache(normalized);
            value = cache.peek();
        }
        let stack = vm.stack();
        let upsert = this.insert(vm.env.getAppendOperations(), stack, value);
        let bounds = new builder_1.Fragment(upsert.bounds);
        stack.newBounds(bounds);
        if (cache /* i.e. !isConst(reference) */) {
            vm.updateWith(this.updateWith(vm, reference, cache, bounds, upsert));
        }
    }
}
exports.AppendDynamicOpcode = AppendDynamicOpcode;
class GuardedAppendOpcode extends AppendDynamicOpcode {
    constructor(expression, symbolTable) {
        super();
        this.expression = expression;
        this.symbolTable = symbolTable;
        this.start = -1;
        this.end = -1;
    }
    evaluate(vm) {
        if (this.start === -1) {
            vm.evaluateOperand(this.expression);
            let value = vm.frame.getOperand().value();
            if (interfaces_1.isComponentDefinition(value)) {
                this.deopt(vm.env);
                vm.pushEvalFrame(this.start, this.end);
            }
            else {
                super.evaluate(vm);
            }
        }
        else {
            vm.pushEvalFrame(this.start, this.end);
        }
    }
    deopt(env) {
        // At compile time, we determined that this append callsite might refer
        // to a local variable/property lookup that resolves to a component
        // definition at runtime.
        //
        // We could have eagerly compiled this callsite into something like this:
        //
        //   {{#if (is-component-definition foo)}}
        //     {{component foo}}
        //   {{else}}
        //     {{foo}}
        //   {{/if}}
        //
        // However, in practice, there might be a large amout of these callsites
        // and most of them would resolve to a simple value lookup. Therefore, we
        // tried to be optimistic and assumed that the callsite will resolve to
        // appending a simple value.
        //
        // However, we have reached here because at runtime, the guard conditional
        // have detected that this callsite is indeed referring to a component
        // definition object. Since this is likely going to be true for other
        // instances of the same callsite, it is now appropiate to deopt into the
        // expanded version that handles both cases. The compilation would look
        // like this:
        //
        //               PutValue(expression)
        //               Test(is-component-definition)
        //               Enter(BEGIN, END)
        //   BEGIN:      Noop
        //               JumpUnless(VALUE)
        //               PutDynamicComponentDefinitionOpcode
        //               OpenComponent
        //               CloseComponent
        //               Jump(END)
        //   VALUE:      Noop
        //               OptimizedAppend
        //   END:        Noop
        //               Exit
        //
        // Keep in mind that even if we *don't* reach here at initial render time,
        // it is still possible (although quite rare) that the simple value we
        // encounter during initial render could later change into a component
        // definition object at update time. That is handled by the "lazy deopt"
        // code on the update side (scroll down for the next big block of comment).
        let dsl = new builder_2.default(this.symbolTable, env);
        dsl.putValue(this.expression);
        dsl.test(IsComponentDefinitionReference.create);
        dsl.labelled(null, (dsl, _BEGIN, END) => {
            dsl.jumpUnless('VALUE');
            dsl.putDynamicComponentDefinition();
            dsl.openComponent(expressions_1.CompiledArgs.empty());
            dsl.closeComponent();
            dsl.jump(END);
            dsl.label('VALUE');
            dsl.dynamicContent(new this.AppendOpcode());
        });
        this.start = dsl.start;
        this.end = dsl.end;
        // From this point on, we have essentially replaced ourselves with a new set
        // of opcodes. Since we will always be executing the new/deopted code, it's
        // a good idea (as a pattern) to null out any unneeded fields here to avoid
        // holding on to unneeded/stale objects:
        // QUESTION: Shouldn't this whole object be GCed? If not, why not?
        this.expression = null;
        return dsl.start;
    }
}
exports.GuardedAppendOpcode = GuardedAppendOpcode;
class IsComponentDefinitionReference extends references_1.ConditionalReference {
    static create(inner) {
        return new IsComponentDefinitionReference(inner);
    }
    toBool(value) {
        return interfaces_1.isComponentDefinition(value);
    }
}
class UpdateOpcode extends opcodes_1.UpdatingOpcode {
    constructor(cache, bounds, upsert) {
        super();
        this.cache = cache;
        this.bounds = bounds;
        this.upsert = upsert;
        this.tag = cache.tag;
    }
    evaluate(vm) {
        let value = this.cache.revalidate();
        if (reference_1.isModified(value)) {
            let { bounds, upsert } = this;
            let { dom } = vm;
            if (!this.upsert.update(dom, value)) {
                let cursor = new bounds_1.Cursor(bounds.parentElement(), bounds_1.clear(bounds));
                upsert = this.upsert = this.insert(vm.env.getAppendOperations(), cursor, value);
            }
            bounds.update(upsert.bounds);
        }
    }
    toJSON() {
        let { _guid: guid, type, cache } = this;
        return {
            guid,
            type,
            details: { lastValue: JSON.stringify(cache.peek()) }
        };
    }
}
class GuardedUpdateOpcode extends UpdateOpcode {
    constructor(reference, cache, bounds, upsert, appendOpcode, state) {
        super(cache, bounds, upsert);
        this.reference = reference;
        this.appendOpcode = appendOpcode;
        this.state = state;
        this.deopted = null;
        this.tag = this._tag = new reference_1.UpdatableTag(this.tag);
    }
    evaluate(vm) {
        if (this.deopted) {
            vm.evaluateOpcode(this.deopted);
        }
        else {
            if (interfaces_1.isComponentDefinition(this.reference.value())) {
                this.lazyDeopt(vm);
            }
            else {
                super.evaluate(vm);
            }
        }
    }
    lazyDeopt(vm) {
        // Durign initial render, we know that the reference does not contain a
        // component definition, so we optimistically assumed that this append
        // is just a normal append. However, at update time, we discovered that
        // the reference has switched into containing a component definition, so
        // we need to do a "lazy deopt", simulating what would have happened if
        // we had decided to perform the deopt in the first place during initial
        // render.
        //
        // More concretely, we would have expanded the curly into a if/else, and
        // based on whether the value is a component definition or not, we would
        // have entered either the dynamic component branch or the simple value
        // branch.
        //
        // Since we rendered a simple value during initial render (and all the
        // updates up until this point), we need to pretend that the result is
        // produced by the "VALUE" branch of the deopted append opcode:
        //
        //   Try(BEGIN, END)
        //     Assert(IsComponentDefinition, expected=false)
        //     OptimizedUpdate
        //
        // In this case, because the reference has switched from being a simple
        // value into a component definition, what would have happened is that
        // the assert would throw, causing the Try opcode to teardown the bounds
        // and rerun the original append opcode.
        //
        // Since the Try opcode would have nuked the updating opcodes anyway, we
        // wouldn't have to worry about simulating those. All we have to do is to
        // execute the Try opcode and immediately throw.
        let { bounds, appendOpcode, state } = this;
        let env = vm.env;
        let deoptStart = appendOpcode.deopt(env);
        let enter = util_1.expect(env.program.opcode(deoptStart + 8), 'hardcoded deopt location');
        let { op1: start, op2: end } = enter;
        let tracker = new builder_3.UpdatableBlockTracker(bounds.parentElement());
        tracker.newBounds(this.bounds);
        let children = new util_1.LinkedList();
        state.frame.condition = IsComponentDefinitionReference.create(util_1.expect(state.frame['operand'], 'operand should be populated'));
        let deopted = this.deopted = new update_1.TryOpcode(start, end, state, tracker, children);
        this._tag.update(deopted.tag);
        vm.evaluateOpcode(deopted);
        vm.throw();
        // From this point on, we have essentially replaced ourselve with a new
        // opcode. Since we will always be executing the new/deopted code, it's a
        // good idea (as a pattern) to null out any unneeded fields here to avoid
        // holding on to unneeded/stale objects:
        // QUESTION: Shouldn't this whole object be GCed? If not, why not?
        this._tag = null;
        this.reference = null;
        this.cache = null;
        this.bounds = null;
        this.upsert = null;
        this.appendOpcode = null;
        this.state = null;
    }
    toJSON() {
        let { _guid: guid, type, deopted } = this;
        if (deopted) {
            return {
                guid,
                type,
                deopted: true,
                children: [deopted.toJSON()]
            };
        }
        else {
            return super.toJSON();
        }
    }
}
class OptimizedCautiousAppendOpcode extends AppendDynamicOpcode {
    constructor() {
        super(...arguments);
        this.type = 'optimized-cautious-append';
    }
    normalize(reference) {
        return reference_1.map(reference, normalizeValue);
    }
    insert(dom, cursor, value) {
        return upsert_1.cautiousInsert(dom, cursor, value);
    }
    updateWith(_vm, _reference, cache, bounds, upsert) {
        return new OptimizedCautiousUpdateOpcode(cache, bounds, upsert);
    }
}
exports.OptimizedCautiousAppendOpcode = OptimizedCautiousAppendOpcode;
class OptimizedCautiousUpdateOpcode extends UpdateOpcode {
    constructor() {
        super(...arguments);
        this.type = 'optimized-cautious-update';
    }
    insert(dom, cursor, value) {
        return upsert_1.cautiousInsert(dom, cursor, value);
    }
}
class GuardedCautiousAppendOpcode extends GuardedAppendOpcode {
    constructor() {
        super(...arguments);
        this.type = 'guarded-cautious-append';
        this.AppendOpcode = OptimizedCautiousAppendOpcode;
    }
    normalize(reference) {
        return reference_1.map(reference, normalizeValue);
    }
    insert(dom, cursor, value) {
        return upsert_1.cautiousInsert(dom, cursor, value);
    }
    updateWith(vm, reference, cache, bounds, upsert) {
        return new GuardedCautiousUpdateOpcode(reference, cache, bounds, upsert, this, vm.capture());
    }
}
exports.GuardedCautiousAppendOpcode = GuardedCautiousAppendOpcode;
class GuardedCautiousUpdateOpcode extends GuardedUpdateOpcode {
    constructor() {
        super(...arguments);
        this.type = 'guarded-cautious-update';
    }
    insert(dom, cursor, value) {
        return upsert_1.cautiousInsert(dom, cursor, value);
    }
}
class OptimizedTrustingAppendOpcode extends AppendDynamicOpcode {
    constructor() {
        super(...arguments);
        this.type = 'optimized-trusting-append';
    }
    normalize(reference) {
        return reference_1.map(reference, normalizeTrustedValue);
    }
    insert(dom, cursor, value) {
        return upsert_1.trustingInsert(dom, cursor, value);
    }
    updateWith(_vm, _reference, cache, bounds, upsert) {
        return new OptimizedTrustingUpdateOpcode(cache, bounds, upsert);
    }
}
exports.OptimizedTrustingAppendOpcode = OptimizedTrustingAppendOpcode;
class OptimizedTrustingUpdateOpcode extends UpdateOpcode {
    constructor() {
        super(...arguments);
        this.type = 'optimized-trusting-update';
    }
    insert(dom, cursor, value) {
        return upsert_1.trustingInsert(dom, cursor, value);
    }
}
class GuardedTrustingAppendOpcode extends GuardedAppendOpcode {
    constructor() {
        super(...arguments);
        this.type = 'guarded-trusting-append';
        this.AppendOpcode = OptimizedTrustingAppendOpcode;
    }
    normalize(reference) {
        return reference_1.map(reference, normalizeTrustedValue);
    }
    insert(dom, cursor, value) {
        return upsert_1.trustingInsert(dom, cursor, value);
    }
    updateWith(vm, reference, cache, bounds, upsert) {
        return new GuardedTrustingUpdateOpcode(reference, cache, bounds, upsert, this, vm.capture());
    }
}
exports.GuardedTrustingAppendOpcode = GuardedTrustingAppendOpcode;
class GuardedTrustingUpdateOpcode extends GuardedUpdateOpcode {
    constructor() {
        super(...arguments);
        this.type = 'trusting-update';
    }
    insert(dom, cursor, value) {
        return upsert_1.trustingInsert(dom, cursor, value);
    }
}
//# sourceMappingURL=data:application/json;base64,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