"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const opcodes_1 = require("../../opcodes");
const util_1 = require("@glimmer/util");
const reference_1 = require("@glimmer/reference");
const references_1 = require("../../references");
const vm_1 = require("./vm");
const opcodes_2 = require("../../opcodes");
opcodes_2.APPEND_OPCODES.add(29 /* Text */, (vm, { op1: text }) => {
    vm.stack().appendText(vm.constants.getString(text));
});
opcodes_2.APPEND_OPCODES.add(30 /* Comment */, (vm, { op1: text }) => {
    vm.stack().appendComment(vm.constants.getString(text));
});
opcodes_2.APPEND_OPCODES.add(32 /* OpenElement */, (vm, { op1: tag }) => {
    vm.stack().openElement(vm.constants.getString(tag));
});
opcodes_2.APPEND_OPCODES.add(33 /* PushRemoteElement */, vm => {
    let reference = vm.frame.getOperand();
    let cache = reference_1.isConst(reference) ? undefined : new reference_1.ReferenceCache(reference);
    let element = cache ? cache.peek() : reference.value();
    vm.stack().pushRemoteElement(element);
    if (cache) {
        vm.updateWith(new vm_1.Assert(cache));
    }
});
opcodes_2.APPEND_OPCODES.add(34 /* PopRemoteElement */, vm => vm.stack().popRemoteElement());
opcodes_2.APPEND_OPCODES.add(35 /* OpenComponentElement */, (vm, { op1: _tag }) => {
    let tag = vm.constants.getString(_tag);
    vm.stack().openElement(tag, new ComponentElementOperations(vm.env));
});
opcodes_2.APPEND_OPCODES.add(36 /* OpenDynamicElement */, vm => {
    let tagName = vm.frame.getOperand().value();
    vm.stack().openElement(tagName);
});
class ClassList {
    constructor() {
        this.list = null;
        this.isConst = true;
    }
    append(reference) {
        let { list, isConst } = this;
        if (list === null)
            list = this.list = [];
        list.push(reference);
        this.isConst = isConst && reference_1.isConst(reference);
    }
    toReference() {
        let { list, isConst } = this;
        if (!list)
            return references_1.NULL_REFERENCE;
        if (isConst)
            return references_1.PrimitiveReference.create(toClassName(list));
        return new ClassListReference(list);
    }
}
class ClassListReference extends reference_1.CachedReference {
    constructor(list) {
        super();
        this.list = [];
        this.tag = reference_1.combineTagged(list);
        this.list = list;
    }
    compute() {
        return toClassName(this.list);
    }
}
function toClassName(list) {
    let ret = [];
    for (let i = 0; i < list.length; i++) {
        let value = list[i].value();
        if (value !== false && value !== null && value !== undefined)
            ret.push(value);
    }
    return (ret.length === 0) ? null : ret.join(' ');
}
class SimpleElementOperations {
    constructor(env) {
        this.env = env;
        this.opcodes = null;
        this.classList = null;
    }
    addStaticAttribute(element, name, value) {
        if (name === 'class') {
            this.addClass(references_1.PrimitiveReference.create(value));
        }
        else {
            this.env.getAppendOperations().setAttribute(element, name, value);
        }
    }
    addStaticAttributeNS(element, namespace, name, value) {
        this.env.getAppendOperations().setAttribute(element, name, value, namespace);
    }
    addDynamicAttribute(element, name, reference, isTrusting) {
        if (name === 'class') {
            this.addClass(reference);
        }
        else {
            let attributeManager = this.env.attributeFor(element, name, isTrusting);
            let attribute = new DynamicAttribute(element, attributeManager, name, reference);
            this.addAttribute(attribute);
        }
    }
    addDynamicAttributeNS(element, namespace, name, reference, isTrusting) {
        let attributeManager = this.env.attributeFor(element, name, isTrusting, namespace);
        let nsAttribute = new DynamicAttribute(element, attributeManager, name, reference, namespace);
        this.addAttribute(nsAttribute);
    }
    flush(element, vm) {
        let { env } = vm;
        let { opcodes, classList } = this;
        for (let i = 0; opcodes && i < opcodes.length; i++) {
            vm.updateWith(opcodes[i]);
        }
        if (classList) {
            let attributeManager = env.attributeFor(element, 'class', false);
            let attribute = new DynamicAttribute(element, attributeManager, 'class', classList.toReference());
            let opcode = attribute.flush(env);
            if (opcode) {
                vm.updateWith(opcode);
            }
        }
        this.opcodes = null;
        this.classList = null;
    }
    addClass(reference) {
        let { classList } = this;
        if (!classList) {
            classList = this.classList = new ClassList();
        }
        classList.append(reference);
    }
    addAttribute(attribute) {
        let opcode = attribute.flush(this.env);
        if (opcode) {
            let { opcodes } = this;
            if (!opcodes) {
                opcodes = this.opcodes = [];
            }
            opcodes.push(opcode);
        }
    }
}
exports.SimpleElementOperations = SimpleElementOperations;
class ComponentElementOperations {
    constructor(env) {
        this.env = env;
        this.attributeNames = null;
        this.attributes = null;
        this.classList = null;
    }
    addStaticAttribute(element, name, value) {
        if (name === 'class') {
            this.addClass(references_1.PrimitiveReference.create(value));
        }
        else if (this.shouldAddAttribute(name)) {
            this.addAttribute(name, new StaticAttribute(element, name, value));
        }
    }
    addStaticAttributeNS(element, namespace, name, value) {
        if (this.shouldAddAttribute(name)) {
            this.addAttribute(name, new StaticAttribute(element, name, value, namespace));
        }
    }
    addDynamicAttribute(element, name, reference, isTrusting) {
        if (name === 'class') {
            this.addClass(reference);
        }
        else if (this.shouldAddAttribute(name)) {
            let attributeManager = this.env.attributeFor(element, name, isTrusting);
            let attribute = new DynamicAttribute(element, attributeManager, name, reference);
            this.addAttribute(name, attribute);
        }
    }
    addDynamicAttributeNS(element, namespace, name, reference, isTrusting) {
        if (this.shouldAddAttribute(name)) {
            let attributeManager = this.env.attributeFor(element, name, isTrusting, namespace);
            let nsAttribute = new DynamicAttribute(element, attributeManager, name, reference, namespace);
            this.addAttribute(name, nsAttribute);
        }
    }
    flush(element, vm) {
        let { env } = this;
        let { attributes, classList } = this;
        for (let i = 0; attributes && i < attributes.length; i++) {
            let opcode = attributes[i].flush(env);
            if (opcode) {
                vm.updateWith(opcode);
            }
        }
        if (classList) {
            let attributeManager = env.attributeFor(element, 'class', false);
            let attribute = new DynamicAttribute(element, attributeManager, 'class', classList.toReference());
            let opcode = attribute.flush(env);
            if (opcode) {
                vm.updateWith(opcode);
            }
        }
    }
    shouldAddAttribute(name) {
        return !this.attributeNames || this.attributeNames.indexOf(name) === -1;
    }
    addClass(reference) {
        let { classList } = this;
        if (!classList) {
            classList = this.classList = new ClassList();
        }
        classList.append(reference);
    }
    addAttribute(name, attribute) {
        let { attributeNames, attributes } = this;
        if (!attributeNames) {
            attributeNames = this.attributeNames = [];
            attributes = this.attributes = [];
        }
        attributeNames.push(name);
        util_1.unwrap(attributes).push(attribute);
    }
}
exports.ComponentElementOperations = ComponentElementOperations;
opcodes_2.APPEND_OPCODES.add(37 /* FlushElement */, vm => {
    let stack = vm.stack();
    let action = 'FlushElementOpcode#evaluate';
    stack.expectOperations(action).flush(stack.expectConstructing(action), vm);
    stack.flushElement();
});
opcodes_2.APPEND_OPCODES.add(38 /* CloseElement */, vm => vm.stack().closeElement());
opcodes_2.APPEND_OPCODES.add(39 /* PopElement */, vm => vm.stack().popElement());
opcodes_2.APPEND_OPCODES.add(40 /* StaticAttr */, (vm, { op1: _name, op2: _value, op3: _namespace }) => {
    let name = vm.constants.getString(_name);
    let value = vm.constants.getString(_value);
    if (_namespace) {
        let namespace = vm.constants.getString(_namespace);
        vm.stack().setStaticAttributeNS(namespace, name, value);
    }
    else {
        vm.stack().setStaticAttribute(name, value);
    }
});
opcodes_2.APPEND_OPCODES.add(41 /* Modifier */, (vm, { op1: _name, op2: _manager, op3: _args }) => {
    let manager = vm.constants.getOther(_manager);
    let rawArgs = vm.constants.getExpression(_args);
    let stack = vm.stack();
    let { constructing: element, updateOperations } = stack;
    let args = rawArgs.evaluate(vm);
    let dynamicScope = vm.dynamicScope();
    let modifier = manager.create(element, args, dynamicScope, updateOperations);
    vm.env.scheduleInstallModifier(modifier, manager);
    let destructor = manager.getDestructor(modifier);
    if (destructor) {
        vm.newDestroyable(destructor);
    }
    vm.updateWith(new UpdateModifierOpcode(manager, modifier, args));
});
class UpdateModifierOpcode extends opcodes_1.UpdatingOpcode {
    constructor(manager, modifier, args) {
        super();
        this.manager = manager;
        this.modifier = modifier;
        this.args = args;
        this.type = "update-modifier";
        this.tag = args.tag;
        this.lastUpdated = args.tag.value();
    }
    evaluate(vm) {
        let { manager, modifier, tag, lastUpdated } = this;
        if (!tag.validate(lastUpdated)) {
            vm.env.scheduleUpdateModifier(modifier, manager);
            this.lastUpdated = tag.value();
        }
    }
    toJSON() {
        return {
            guid: this._guid,
            type: this.type,
            args: [JSON.stringify(this.args)]
        };
    }
}
exports.UpdateModifierOpcode = UpdateModifierOpcode;
class StaticAttribute {
    constructor(element, name, value, namespace) {
        this.element = element;
        this.name = name;
        this.value = value;
        this.namespace = namespace;
    }
    flush(env) {
        env.getAppendOperations().setAttribute(this.element, this.name, this.value, this.namespace);
        return null;
    }
}
exports.StaticAttribute = StaticAttribute;
class DynamicAttribute {
    constructor(element, attributeManager, name, reference, namespace) {
        this.element = element;
        this.attributeManager = attributeManager;
        this.name = name;
        this.reference = reference;
        this.namespace = namespace;
        this.cache = null;
        this.tag = reference.tag;
    }
    patch(env) {
        let { element, cache } = this;
        let value = util_1.expect(cache, 'must patch after flush').revalidate();
        if (reference_1.isModified(value)) {
            this.attributeManager.updateAttribute(env, element, value, this.namespace);
        }
    }
    flush(env) {
        let { reference, element } = this;
        if (reference_1.isConst(reference)) {
            let value = reference.value();
            this.attributeManager.setAttribute(env, element, value, this.namespace);
            return null;
        }
        else {
            let cache = this.cache = new reference_1.ReferenceCache(reference);
            let value = cache.peek();
            this.attributeManager.setAttribute(env, element, value, this.namespace);
            return new PatchElementOpcode(this);
        }
    }
    toJSON() {
        let { element, namespace, name, cache } = this;
        let formattedElement = formatElement(element);
        let lastValue = util_1.expect(cache, 'must serialize after flush').peek();
        if (namespace) {
            return {
                element: formattedElement,
                type: 'attribute',
                namespace,
                name,
                lastValue
            };
        }
        return {
            element: formattedElement,
            type: 'attribute',
            namespace: namespace === undefined ? null : namespace,
            name,
            lastValue
        };
    }
}
exports.DynamicAttribute = DynamicAttribute;
function formatElement(element) {
    return JSON.stringify(`<${element.tagName.toLowerCase()} />`);
}
opcodes_2.APPEND_OPCODES.add(42 /* DynamicAttrNS */, (vm, { op1: _name, op2: _namespace, op3: trusting }) => {
    let name = vm.constants.getString(_name);
    let namespace = vm.constants.getString(_namespace);
    let reference = vm.frame.getOperand();
    vm.stack().setDynamicAttributeNS(namespace, name, reference, !!trusting);
});
opcodes_2.APPEND_OPCODES.add(43 /* DynamicAttr */, (vm, { op1: _name, op2: trusting }) => {
    let name = vm.constants.getString(_name);
    let reference = vm.frame.getOperand();
    vm.stack().setDynamicAttribute(name, reference, !!trusting);
});
class PatchElementOpcode extends opcodes_1.UpdatingOpcode {
    constructor(operation) {
        super();
        this.type = "patch-element";
        this.tag = operation.tag;
        this.operation = operation;
    }
    evaluate(vm) {
        this.operation.patch(vm.env);
    }
    toJSON() {
        let { _guid, type, operation } = this;
        return {
            guid: _guid,
            type,
            details: operation.toJSON()
        };
    }
}
exports.PatchElementOpcode = PatchElementOpcode;
//# sourceMappingURL=data:application/json;base64,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