"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const blocks_1 = require("./compiled/blocks");
const wire_format_1 = require("@glimmer/wire-format");
const scanner_1 = require("./scanner");
const functions_1 = require("./syntax/functions");
const builder_1 = require("./compiled/opcodes/builder");
function compileLayout(compilable, env) {
    let builder = new ComponentLayoutBuilder(env);
    compilable.compile(builder);
    return builder.compile();
}
exports.compileLayout = compileLayout;
class ComponentLayoutBuilder {
    constructor(env) {
        this.env = env;
    }
    wrapLayout(layout) {
        this.inner = new WrappedBuilder(this.env, layout);
    }
    fromLayout(layout) {
        this.inner = new UnwrappedBuilder(this.env, layout);
    }
    compile() {
        return this.inner.compile();
    }
    get tag() {
        return this.inner.tag;
    }
    get attrs() {
        return this.inner.attrs;
    }
}
class WrappedBuilder {
    constructor(env, layout) {
        this.env = env;
        this.layout = layout;
        this.tag = new ComponentTagBuilder();
        this.attrs = new ComponentAttrsBuilder();
    }
    compile() {
        //========DYNAMIC
        //        PutValue(TagExpr)
        //        Test
        //        JumpUnless(BODY)
        //        OpenDynamicPrimitiveElement
        //        DidCreateElement
        //        ...attr statements...
        //        FlushElement
        // BODY:  Noop
        //        ...body statements...
        //        PutValue(TagExpr)
        //        Test
        //        JumpUnless(END)
        //        CloseElement
        // END:   Noop
        //        DidRenderLayout
        //        Exit
        //
        //========STATIC
        //        OpenPrimitiveElementOpcode
        //        DidCreateElement
        //        ...attr statements...
        //        FlushElement
        //        ...body statements...
        //        CloseElement
        //        DidRenderLayout
        //        Exit
        let { env, layout } = this;
        let symbolTable = layout.symbolTable;
        let b = builder(env, layout.symbolTable);
        b.startLabels();
        let dynamicTag = this.tag.getDynamic();
        let staticTag;
        if (dynamicTag) {
            b.putValue(dynamicTag);
            b.test('simple');
            b.jumpUnless('BODY');
            b.openDynamicPrimitiveElement();
            b.didCreateElement();
            this.attrs['buffer'].forEach(statement => scanner_1.compileStatement(statement, b));
            b.flushElement();
            b.label('BODY');
        }
        else if (staticTag = this.tag.getStatic()) {
            b.openPrimitiveElement(staticTag);
            b.didCreateElement();
            this.attrs['buffer'].forEach(statement => scanner_1.compileStatement(statement, b));
            b.flushElement();
        }
        b.preludeForLayout(layout);
        layout.statements.forEach(statement => scanner_1.compileStatement(statement, b));
        if (dynamicTag) {
            b.putValue(dynamicTag);
            b.test('simple');
            b.jumpUnless('END');
            b.closeElement();
            b.label('END');
        }
        else if (staticTag) {
            b.closeElement();
        }
        b.didRenderLayout();
        b.stopLabels();
        return new blocks_1.CompiledProgram(b.start, b.end, symbolTable.size);
    }
}
function isOpenElement(value) {
    let type = value[0];
    return type === wire_format_1.Ops.OpenElement || type === wire_format_1.Ops.OpenPrimitiveElement;
}
class UnwrappedBuilder {
    constructor(env, layout) {
        this.env = env;
        this.layout = layout;
        this.attrs = new ComponentAttrsBuilder();
    }
    get tag() {
        throw new Error('BUG: Cannot call `tag` on an UnwrappedBuilder');
    }
    compile() {
        let { env, layout } = this;
        let b = builder(env, layout.symbolTable);
        b.startLabels();
        b.preludeForLayout(layout);
        let attrs = this.attrs['buffer'];
        let attrsInserted = false;
        for (let i = 0; i < layout.statements.length; i++) {
            let statement = layout.statements[i];
            if (!attrsInserted && isOpenElement(statement)) {
                b.openComponentElement(statement[1]);
                b.didCreateElement();
                b.shadowAttributes();
                attrs.forEach(statement => scanner_1.compileStatement(statement, b));
                attrsInserted = true;
            }
            else {
                scanner_1.compileStatement(statement, b);
            }
        }
        b.didRenderLayout();
        b.stopLabels();
        return new blocks_1.CompiledProgram(b.start, b.end, layout.symbolTable.size);
    }
}
class ComponentTagBuilder {
    constructor() {
        this.isDynamic = null;
        this.isStatic = null;
        this.staticTagName = null;
        this.dynamicTagName = null;
    }
    getDynamic() {
        if (this.isDynamic) {
            return this.dynamicTagName;
        }
    }
    getStatic() {
        if (this.isStatic) {
            return this.staticTagName;
        }
    }
    static(tagName) {
        this.isStatic = true;
        this.staticTagName = tagName;
    }
    dynamic(tagName) {
        this.isDynamic = true;
        this.dynamicTagName = [wire_format_1.Ops.Function, tagName];
    }
}
class ComponentAttrsBuilder {
    constructor() {
        this.buffer = [];
    }
    static(name, value) {
        this.buffer.push([wire_format_1.Ops.StaticAttr, name, value, null]);
    }
    dynamic(name, value) {
        this.buffer.push([wire_format_1.Ops.DynamicAttr, name, [wire_format_1.Ops.Function, value], null]);
    }
}
class ComponentBuilder {
    constructor(builder) {
        this.builder = builder;
        this.env = builder.env;
    }
    static(definition, args, _symbolTable, shadow) {
        this.builder.unit(b => {
            b.putComponentDefinition(definition);
            b.openComponent(functions_1.compileBaselineArgs(args, b), shadow);
            b.closeComponent();
        });
    }
    dynamic(definitionArgs, definition, args, _symbolTable, shadow) {
        this.builder.unit(b => {
            b.putArgs(functions_1.compileArgs(definitionArgs[0], definitionArgs[1], b));
            b.putValue([wire_format_1.Ops.Function, definition]);
            b.test('simple');
            b.enter('BEGIN', 'END');
            b.label('BEGIN');
            b.jumpUnless('END');
            b.putDynamicComponentDefinition();
            b.openComponent(functions_1.compileBaselineArgs(args, b), shadow);
            b.closeComponent();
            b.label('END');
            b.exit();
        });
    }
}
exports.ComponentBuilder = ComponentBuilder;
function builder(env, symbolTable) {
    return new builder_1.default(symbolTable, env);
}
exports.builder = builder;
//# sourceMappingURL=data:application/json;base64,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