"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sanitized_values_1 = require("./sanitized-values");
const props_1 = require("./props");
const helper_1 = require("./helper");
const content_1 = require("../compiled/opcodes/content");
function defaultManagers(element, attr, _isTrusting, _namespace) {
    let tagName = element.tagName;
    let isSVG = element.namespaceURI === helper_1.SVG_NAMESPACE;
    if (isSVG) {
        return defaultAttributeManagers(tagName, attr);
    }
    let { type, normalized } = props_1.normalizeProperty(element, attr);
    if (type === 'attr') {
        return defaultAttributeManagers(tagName, normalized);
    }
    else {
        return defaultPropertyManagers(tagName, normalized);
    }
}
exports.defaultManagers = defaultManagers;
function defaultPropertyManagers(tagName, attr) {
    if (sanitized_values_1.requiresSanitization(tagName, attr)) {
        return new SafePropertyManager(attr);
    }
    if (isUserInputValue(tagName, attr)) {
        return exports.INPUT_VALUE_PROPERTY_MANAGER;
    }
    if (isOptionSelected(tagName, attr)) {
        return exports.OPTION_SELECTED_MANAGER;
    }
    return new PropertyManager(attr);
}
exports.defaultPropertyManagers = defaultPropertyManagers;
function defaultAttributeManagers(tagName, attr) {
    if (sanitized_values_1.requiresSanitization(tagName, attr)) {
        return new SafeAttributeManager(attr);
    }
    return new AttributeManager(attr);
}
exports.defaultAttributeManagers = defaultAttributeManagers;
function readDOMAttr(element, attr) {
    let isSVG = element.namespaceURI === helper_1.SVG_NAMESPACE;
    let { type, normalized } = props_1.normalizeProperty(element, attr);
    if (isSVG) {
        return element.getAttribute(normalized);
    }
    if (type === 'attr') {
        return element.getAttribute(normalized);
    }
    {
        return element[normalized];
    }
}
exports.readDOMAttr = readDOMAttr;
;
class AttributeManager {
    constructor(attr) {
        this.attr = attr;
    }
    setAttribute(env, element, value, namespace) {
        let dom = env.getAppendOperations();
        let normalizedValue = normalizeAttributeValue(value);
        if (!isAttrRemovalValue(normalizedValue)) {
            dom.setAttribute(element, this.attr, normalizedValue, namespace);
        }
    }
    updateAttribute(env, element, value, namespace) {
        if (value === null || value === undefined || value === false) {
            if (namespace) {
                env.getDOM().removeAttributeNS(element, namespace, this.attr);
            }
            else {
                env.getDOM().removeAttribute(element, this.attr);
            }
        }
        else {
            this.setAttribute(env, element, value);
        }
    }
}
exports.AttributeManager = AttributeManager;
;
class PropertyManager extends AttributeManager {
    setAttribute(_env, element, value, _namespace) {
        if (!isAttrRemovalValue(value)) {
            element[this.attr] = value;
        }
    }
    removeAttribute(env, element, namespace) {
        // TODO this sucks but to preserve properties first and to meet current
        // semantics we must do this.
        let { attr } = this;
        if (namespace) {
            env.getDOM().removeAttributeNS(element, namespace, attr);
        }
        else {
            env.getDOM().removeAttribute(element, attr);
        }
    }
    updateAttribute(env, element, value, namespace) {
        // ensure the property is always updated
        element[this.attr] = value;
        if (isAttrRemovalValue(value)) {
            this.removeAttribute(env, element, namespace);
        }
    }
}
exports.PropertyManager = PropertyManager;
;
function normalizeAttributeValue(value) {
    if (value === false || value === undefined || value === null) {
        return null;
    }
    if (value === true) {
        return '';
    }
    // onclick function etc in SSR
    if (typeof value === 'function') {
        return null;
    }
    return String(value);
}
function isAttrRemovalValue(value) {
    return value === null || value === undefined;
}
class SafePropertyManager extends PropertyManager {
    setAttribute(env, element, value) {
        super.setAttribute(env, element, sanitized_values_1.sanitizeAttributeValue(env, element, this.attr, value));
    }
    updateAttribute(env, element, value) {
        super.updateAttribute(env, element, sanitized_values_1.sanitizeAttributeValue(env, element, this.attr, value));
    }
}
function isUserInputValue(tagName, attribute) {
    return (tagName === 'INPUT' || tagName === 'TEXTAREA') && attribute === 'value';
}
class InputValuePropertyManager extends AttributeManager {
    setAttribute(_env, element, value) {
        let input = element;
        input.value = content_1.normalizeTextValue(value);
    }
    updateAttribute(_env, element, value) {
        let input = element;
        let currentValue = input.value;
        let normalizedValue = content_1.normalizeTextValue(value);
        if (currentValue !== normalizedValue) {
            input.value = normalizedValue;
        }
    }
}
exports.INPUT_VALUE_PROPERTY_MANAGER = new InputValuePropertyManager('value');
function isOptionSelected(tagName, attribute) {
    return tagName === 'OPTION' && attribute === 'selected';
}
class OptionSelectedManager extends PropertyManager {
    setAttribute(_env, element, value) {
        if (value !== null && value !== undefined && value !== false) {
            let option = element;
            option.selected = true;
        }
    }
    updateAttribute(_env, element, value) {
        let option = element;
        if (value) {
            option.selected = true;
        }
        else {
            option.selected = false;
        }
    }
}
exports.OPTION_SELECTED_MANAGER = new OptionSelectedManager('selected');
class SafeAttributeManager extends AttributeManager {
    setAttribute(env, element, value) {
        super.setAttribute(env, element, sanitized_values_1.sanitizeAttributeValue(env, element, this.attr, value));
    }
    updateAttribute(env, element, value, _namespace) {
        super.updateAttribute(env, element, sanitized_values_1.sanitizeAttributeValue(env, element, this.attr, value));
    }
}
//# sourceMappingURL=data:application/json;base64,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