"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const bounds_1 = require("../bounds");
const inner_html_fix_1 = require("../compat/inner-html-fix");
const svg_inner_html_fix_1 = require("../compat/svg-inner-html-fix");
const text_node_merging_fix_1 = require("../compat/text-node-merging-fix");
exports.SVG_NAMESPACE = 'http://www.w3.org/2000/svg';
// http://www.w3.org/TR/html/syntax.html#html-integration-point
const SVG_INTEGRATION_POINTS = { foreignObject: 1, desc: 1, title: 1 };
// http://www.w3.org/TR/html/syntax.html#adjust-svg-attributes
// TODO: Adjust SVG attributes
// http://www.w3.org/TR/html/syntax.html#parsing-main-inforeign
// TODO: Adjust SVG elements
// http://www.w3.org/TR/html/syntax.html#parsing-main-inforeign
exports.BLACKLIST_TABLE = Object.create(null);
([
    "b", "big", "blockquote", "body", "br", "center", "code", "dd", "div", "dl", "dt", "em", "embed",
    "h1", "h2", "h3", "h4", "h5", "h6", "head", "hr", "i", "img", "li", "listing", "main", "meta", "nobr",
    "ol", "p", "pre", "ruby", "s", "small", "span", "strong", "strike", "sub", "sup", "table", "tt", "u",
    "ul", "var"
]).forEach(tag => exports.BLACKLIST_TABLE[tag] = 1);
const WHITESPACE = /[\t-\r \xA0\u1680\u180E\u2000-\u200A\u2028\u2029\u202F\u205F\u3000\uFEFF]/;
let doc = typeof document === 'undefined' ? null : document;
function isWhitespace(string) {
    return WHITESPACE.test(string);
}
exports.isWhitespace = isWhitespace;
function moveNodesBefore(source, target, nextSibling) {
    let first = source.firstChild;
    let last = null;
    let current = first;
    while (current) {
        last = current;
        current = current.nextSibling;
        target.insertBefore(last, nextSibling);
    }
    return [first, last];
}
exports.moveNodesBefore = moveNodesBefore;
var DOM;
(function (DOM) {
    class TreeConstruction {
        constructor(document) {
            this.document = document;
            this.setupUselessElement();
        }
        setupUselessElement() {
            this.uselessElement = this.document.createElement('div');
        }
        createElement(tag, context) {
            let isElementInSVGNamespace, isHTMLIntegrationPoint;
            if (context) {
                isElementInSVGNamespace = context.namespaceURI === exports.SVG_NAMESPACE || tag === 'svg';
                isHTMLIntegrationPoint = SVG_INTEGRATION_POINTS[context.tagName];
            }
            else {
                isElementInSVGNamespace = tag === 'svg';
                isHTMLIntegrationPoint = false;
            }
            if (isElementInSVGNamespace && !isHTMLIntegrationPoint) {
                // FIXME: This does not properly handle <font> with color, face, or
                // size attributes, which is also disallowed by the spec. We should fix
                // this.
                if (exports.BLACKLIST_TABLE[tag]) {
                    throw new Error(`Cannot create a ${tag} inside an SVG context`);
                }
                return this.document.createElementNS(exports.SVG_NAMESPACE, tag);
            }
            else {
                return this.document.createElement(tag);
            }
        }
        createElementNS(namespace, tag) {
            return this.document.createElementNS(namespace, tag);
        }
        setAttribute(element, name, value, namespace) {
            if (namespace) {
                element.setAttributeNS(namespace, name, value);
            }
            else {
                element.setAttribute(name, value);
            }
        }
        createTextNode(text) {
            return this.document.createTextNode(text);
        }
        createComment(data) {
            return this.document.createComment(data);
        }
        insertBefore(parent, node, reference) {
            parent.insertBefore(node, reference);
        }
        insertHTMLBefore(parent, html, reference) {
            return insertHTMLBefore(this.uselessElement, parent, reference, html);
        }
        ;
    }
    DOM.TreeConstruction = TreeConstruction;
    let appliedTreeContruction = TreeConstruction;
    appliedTreeContruction = text_node_merging_fix_1.treeConstruction(doc, appliedTreeContruction);
    appliedTreeContruction = inner_html_fix_1.treeConstruction(doc, appliedTreeContruction);
    appliedTreeContruction = svg_inner_html_fix_1.treeConstruction(doc, appliedTreeContruction, exports.SVG_NAMESPACE);
    DOM.DOMTreeConstruction = appliedTreeContruction;
})(DOM = exports.DOM || (exports.DOM = {}));
class DOMChanges {
    constructor(document) {
        this.document = document;
        this.namespace = null;
        this.uselessElement = this.document.createElement('div');
    }
    setAttribute(element, name, value) {
        element.setAttribute(name, value);
    }
    setAttributeNS(element, namespace, name, value) {
        element.setAttributeNS(namespace, name, value);
    }
    removeAttribute(element, name) {
        element.removeAttribute(name);
    }
    removeAttributeNS(element, namespace, name) {
        element.removeAttributeNS(namespace, name);
    }
    createTextNode(text) {
        return this.document.createTextNode(text);
    }
    createComment(data) {
        return this.document.createComment(data);
    }
    createElement(tag, context) {
        let isElementInSVGNamespace, isHTMLIntegrationPoint;
        if (context) {
            isElementInSVGNamespace = context.namespaceURI === exports.SVG_NAMESPACE || tag === 'svg';
            isHTMLIntegrationPoint = SVG_INTEGRATION_POINTS[context.tagName];
        }
        else {
            isElementInSVGNamespace = tag === 'svg';
            isHTMLIntegrationPoint = false;
        }
        if (isElementInSVGNamespace && !isHTMLIntegrationPoint) {
            // FIXME: This does not properly handle <font> with color, face, or
            // size attributes, which is also disallowed by the spec. We should fix
            // this.
            if (exports.BLACKLIST_TABLE[tag]) {
                throw new Error(`Cannot create a ${tag} inside an SVG context`);
            }
            return this.document.createElementNS(exports.SVG_NAMESPACE, tag);
        }
        else {
            return this.document.createElement(tag);
        }
    }
    insertHTMLBefore(_parent, nextSibling, html) {
        return insertHTMLBefore(this.uselessElement, _parent, nextSibling, html);
    }
    insertNodeBefore(parent, node, reference) {
        if (isDocumentFragment(node)) {
            let { firstChild, lastChild } = node;
            this.insertBefore(parent, node, reference);
            return new bounds_1.ConcreteBounds(parent, firstChild, lastChild);
        }
        else {
            this.insertBefore(parent, node, reference);
            return new bounds_1.SingleNodeBounds(parent, node);
        }
    }
    insertTextBefore(parent, nextSibling, text) {
        let textNode = this.createTextNode(text);
        this.insertBefore(parent, textNode, nextSibling);
        return textNode;
    }
    insertBefore(element, node, reference) {
        element.insertBefore(node, reference);
    }
    insertAfter(element, node, reference) {
        this.insertBefore(element, node, reference.nextSibling);
    }
}
exports.DOMChanges = DOMChanges;
function insertHTMLBefore(_useless, _parent, _nextSibling, html) {
    // TypeScript vendored an old version of the DOM spec where `insertAdjacentHTML`
    // only exists on `HTMLElement` but not on `Element`. We actually work with the
    // newer version of the DOM API here (and monkey-patch this method in `./compat`
    // when we detect older browsers). This is a hack to work around this limitation.
    let parent = _parent;
    let useless = _useless;
    let nextSibling = _nextSibling;
    let prev = nextSibling ? nextSibling.previousSibling : parent.lastChild;
    let last;
    if (html === null || html === '') {
        return new bounds_1.ConcreteBounds(parent, null, null);
    }
    if (nextSibling === null) {
        parent.insertAdjacentHTML('beforeEnd', html);
        last = parent.lastChild;
    }
    else if (nextSibling instanceof HTMLElement) {
        nextSibling.insertAdjacentHTML('beforeBegin', html);
        last = nextSibling.previousSibling;
    }
    else {
        // Non-element nodes do not support insertAdjacentHTML, so add an
        // element and call it on that element. Then remove the element.
        //
        // This also protects Edge, IE and Firefox w/o the inspector open
        // from merging adjacent text nodes. See ./compat/text-node-merging-fix.ts
        parent.insertBefore(useless, nextSibling);
        useless.insertAdjacentHTML('beforeBegin', html);
        last = useless.previousSibling;
        parent.removeChild(useless);
    }
    let first = prev ? prev.nextSibling : parent.firstChild;
    return new bounds_1.ConcreteBounds(parent, first, last);
}
exports.insertHTMLBefore = insertHTMLBefore;
function isDocumentFragment(node) {
    return node.nodeType === Node.DOCUMENT_FRAGMENT_NODE;
}
let helper = DOMChanges;
helper = text_node_merging_fix_1.domChanges(doc, helper);
helper = inner_html_fix_1.domChanges(doc, helper);
helper = svg_inner_html_fix_1.domChanges(doc, helper, exports.SVG_NAMESPACE);
exports.default = helper;
exports.DOMTreeConstruction = DOM.DOMTreeConstruction;
//# sourceMappingURL=data:application/json;base64,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