"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const blocks_1 = require("./compiled/blocks");
const compiler_1 = require("./compiler");
const WireFormat = require("@glimmer/wire-format");
const symbol_table_1 = require("./symbol-table");
const functions_1 = require("./syntax/functions");
const specialize_1 = require("./syntax/specialize");
function compileStatement(statement, builder) {
    let refined = specialize_1.SPECIALIZE.specialize(statement, builder.symbolTable);
    functions_1.STATEMENTS.compile(refined, builder);
}
exports.compileStatement = compileStatement;
class Template {
    constructor(statements, symbolTable) {
        this.statements = statements;
        this.symbolTable = symbolTable;
    }
}
exports.Template = Template;
class Layout extends Template {
}
exports.Layout = Layout;
class EntryPoint extends Template {
    constructor() {
        super(...arguments);
        this.compiled = null;
    }
    compile(env) {
        let compiled = this.compiled;
        if (!compiled) {
            let table = this.symbolTable;
            let b = compiler_1.builder(env, table);
            for (let i = 0; i < this.statements.length; i++) {
                let statement = this.statements[i];
                let refined = specialize_1.SPECIALIZE.specialize(statement, table);
                functions_1.STATEMENTS.compile(refined, b);
            }
            compiled = this.compiled = new blocks_1.CompiledProgram(b.start, b.end, this.symbolTable.size);
        }
        return compiled;
    }
}
exports.EntryPoint = EntryPoint;
class InlineBlock extends Template {
    constructor() {
        super(...arguments);
        this.compiled = null;
    }
    splat(builder) {
        let table = builder.symbolTable;
        let locals = table.getSymbols().locals;
        if (locals) {
            builder.pushChildScope();
            builder.bindPositionalArgsForLocals(locals);
        }
        for (let i = 0; i < this.statements.length; i++) {
            let statement = this.statements[i];
            let refined = specialize_1.SPECIALIZE.specialize(statement, table);
            functions_1.STATEMENTS.compile(refined, builder);
        }
        if (locals) {
            builder.popScope();
        }
    }
    compile(env) {
        let compiled = this.compiled;
        if (!compiled) {
            let table = this.symbolTable;
            let b = compiler_1.builder(env, table);
            this.splat(b);
            compiled = this.compiled = new blocks_1.CompiledBlock(b.start, b.end);
        }
        return compiled;
    }
}
exports.InlineBlock = InlineBlock;
class PartialBlock extends Template {
    constructor() {
        super(...arguments);
        this.compiled = null;
    }
    compile(env) {
        let compiled = this.compiled;
        if (!compiled) {
            let table = this.symbolTable;
            let b = compiler_1.builder(env, table);
            for (let i = 0; i < this.statements.length; i++) {
                let statement = this.statements[i];
                let refined = specialize_1.SPECIALIZE.specialize(statement, table);
                functions_1.STATEMENTS.compile(refined, b);
            }
            compiled = this.compiled = new blocks_1.CompiledProgram(b.start, b.end, table.size);
        }
        return compiled;
    }
}
exports.PartialBlock = PartialBlock;
class Scanner {
    constructor(block, meta, env) {
        this.block = block;
        this.meta = meta;
        this.env = env;
    }
    scanEntryPoint() {
        let { block, meta } = this;
        let symbolTable = symbol_table_1.entryPoint(meta);
        let child = scanBlock(block, symbolTable, this.env);
        return new EntryPoint(child.statements, symbolTable);
    }
    scanLayout() {
        let { block, meta } = this;
        let { named, yields, hasPartials } = block;
        let symbolTable = symbol_table_1.layout(meta, named, yields, hasPartials);
        let child = scanBlock(block, symbolTable, this.env);
        return new Layout(child.statements, symbolTable);
    }
    scanPartial(symbolTable) {
        let { block } = this;
        let child = scanBlock(block, symbolTable, this.env);
        return new PartialBlock(child.statements, symbolTable);
    }
}
exports.default = Scanner;
function scanBlock({ statements }, symbolTable, env) {
    return new RawInlineBlock(env, symbolTable, statements).scan();
}
exports.scanBlock = scanBlock;
var BaselineSyntax;
(function (BaselineSyntax) {
    const { Ops } = WireFormat;
    BaselineSyntax.isScannedComponent = WireFormat.is(Ops.ScannedComponent);
    BaselineSyntax.isPrimitiveElement = WireFormat.is(Ops.OpenPrimitiveElement);
    BaselineSyntax.isOptimizedAppend = WireFormat.is(Ops.OptimizedAppend);
    BaselineSyntax.isUnoptimizedAppend = WireFormat.is(Ops.UnoptimizedAppend);
    BaselineSyntax.isAnyAttr = WireFormat.is(Ops.AnyDynamicAttr);
    BaselineSyntax.isStaticPartial = WireFormat.is(Ops.StaticPartial);
    BaselineSyntax.isDynamicPartial = WireFormat.is(Ops.DynamicPartial);
    BaselineSyntax.isFunctionExpression = WireFormat.is(Ops.Function);
    BaselineSyntax.isNestedBlock = WireFormat.is(Ops.NestedBlock);
    BaselineSyntax.isScannedBlock = WireFormat.is(Ops.ScannedBlock);
    BaselineSyntax.isDebugger = WireFormat.is(Ops.Debugger);
    var NestedBlock;
    (function (NestedBlock) {
        function defaultBlock(sexp) {
            return sexp[4];
        }
        NestedBlock.defaultBlock = defaultBlock;
        function inverseBlock(sexp) {
            return sexp[5];
        }
        NestedBlock.inverseBlock = inverseBlock;
        function params(sexp) {
            return sexp[2];
        }
        NestedBlock.params = params;
        function hash(sexp) {
            return sexp[3];
        }
        NestedBlock.hash = hash;
    })(NestedBlock = BaselineSyntax.NestedBlock || (BaselineSyntax.NestedBlock = {}));
})(BaselineSyntax = exports.BaselineSyntax || (exports.BaselineSyntax = {}));
const { Ops } = WireFormat;
class RawInlineBlock {
    constructor(env, table, statements) {
        this.env = env;
        this.table = table;
        this.statements = statements;
    }
    scan() {
        let buffer = [];
        for (let i = 0; i < this.statements.length; i++) {
            let statement = this.statements[i];
            if (WireFormat.Statements.isBlock(statement)) {
                buffer.push(this.specializeBlock(statement));
            }
            else if (WireFormat.Statements.isComponent(statement)) {
                buffer.push(...this.specializeComponent(statement));
            }
            else {
                buffer.push(statement);
            }
        }
        return new InlineBlock(buffer, this.table);
    }
    specializeBlock(block) {
        let [, path, params, hash, template, inverse] = block;
        return [Ops.ScannedBlock, path, params, hash, this.child(template), this.child(inverse)];
    }
    specializeComponent(sexp) {
        let [, tag, component] = sexp;
        if (this.env.hasComponentDefinition(tag, this.table)) {
            let child = this.child(component);
            let attrs = new RawInlineBlock(this.env, this.table, component.attrs);
            return [[Ops.ScannedComponent, tag, attrs, component.args, child]];
        }
        else {
            let buf = [];
            buf.push([Ops.OpenElement, tag, []]);
            buf.push(...component.attrs);
            buf.push([Ops.FlushElement]);
            buf.push(...component.statements);
            buf.push([Ops.CloseElement]);
            return buf;
        }
    }
    child(block) {
        if (!block)
            return null;
        let table = symbol_table_1.block(this.table, block.locals);
        return new RawInlineBlock(this.env, table, block.statements);
    }
}
exports.RawInlineBlock = RawInlineBlock;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2Nhbm5lci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL2xpYi9zY2FubmVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsOENBQW1FO0FBQ25FLHlDQUFxQztBQUtyQyxtREFBbUQ7QUFDbkQsaURBQTJHO0FBRzNHLGtEQUU0QjtBQUU1QixvREFFNkI7QUFJN0IsMEJBQWlDLFNBQXNDLEVBQUUsT0FBc0I7SUFDN0YsSUFBSSxPQUFPLEdBQUcsdUJBQVUsQ0FBQyxVQUFVLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUNwRSxzQkFBVSxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUUsT0FBTyxDQUFDLENBQUM7QUFDdkMsQ0FBQztBQUhELDRDQUdDO0FBRUQ7SUFDRSxZQUFtQixVQUF5QyxFQUFTLFdBQXdCO1FBQTFFLGVBQVUsR0FBVixVQUFVLENBQStCO1FBQVMsZ0JBQVcsR0FBWCxXQUFXLENBQWE7SUFBRyxDQUFDO0NBQ2xHO0FBRkQsNEJBRUM7QUFFRCxZQUFvQixTQUFRLFFBQVE7Q0FFbkM7QUFGRCx3QkFFQztBQUVELGdCQUF3QixTQUFRLFFBQVE7SUFBeEM7O1FBQ1UsYUFBUSxHQUE0QixJQUFJLENBQUM7SUFxQm5ELENBQUM7SUFqQkMsT0FBTyxDQUFDLEdBQWdCO1FBQ3RCLElBQUksUUFBUSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUM7UUFDN0IsRUFBRSxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO1lBQ2QsSUFBSSxLQUFLLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQztZQUU3QixJQUFJLENBQUMsR0FBRyxrQkFBTyxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsQ0FBQztZQUU1QixHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUM7Z0JBQ2hELElBQUksU0FBUyxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ25DLElBQUksT0FBTyxHQUFHLHVCQUFVLENBQUMsVUFBVSxDQUFDLFNBQVMsRUFBRSxLQUFLLENBQUMsQ0FBQztnQkFDdEQsc0JBQVUsQ0FBQyxPQUFPLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ2pDLENBQUM7WUFFRCxRQUFRLEdBQUcsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLHdCQUFlLENBQUMsQ0FBQyxDQUFDLEtBQUssRUFBRSxDQUFDLENBQUMsR0FBRyxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDeEYsQ0FBQztRQUNELE1BQU0sQ0FBQyxRQUFRLENBQUM7SUFDbEIsQ0FBQztDQUNGO0FBdEJELGdDQXNCQztBQUVELGlCQUF5QixTQUFRLFFBQVE7SUFBekM7O1FBQ1UsYUFBUSxHQUEwQixJQUFJLENBQUM7SUFtQ2pELENBQUM7SUFqQ0MsS0FBSyxDQUFDLE9BQXNCO1FBQzFCLElBQUksS0FBSyxHQUFHLE9BQU8sQ0FBQyxXQUFXLENBQUM7UUFFaEMsSUFBSSxNQUFNLEdBQUcsS0FBSyxDQUFDLFVBQVUsRUFBRSxDQUFDLE1BQU0sQ0FBQztRQUV2QyxFQUFFLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO1lBQ1gsT0FBTyxDQUFDLGNBQWMsRUFBRSxDQUFDO1lBQ3pCLE9BQU8sQ0FBQywyQkFBMkIsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUM5QyxDQUFDO1FBRUQsR0FBRyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDO1lBQ2hELElBQUksU0FBUyxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDbkMsSUFBSSxPQUFPLEdBQUcsdUJBQVUsQ0FBQyxVQUFVLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBQ3RELHNCQUFVLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRSxPQUFPLENBQUMsQ0FBQztRQUN2QyxDQUFDO1FBRUQsRUFBRSxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztZQUNYLE9BQU8sQ0FBQyxRQUFRLEVBQUUsQ0FBQztRQUNyQixDQUFDO0lBQ0gsQ0FBQztJQUVELE9BQU8sQ0FBQyxHQUFnQjtRQUN0QixJQUFJLFFBQVEsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDO1FBQzdCLEVBQUUsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztZQUNkLElBQUksS0FBSyxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUM7WUFDN0IsSUFBSSxDQUFDLEdBQUcsa0JBQU8sQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFFNUIsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUVkLFFBQVEsR0FBRyxJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksc0JBQWEsQ0FBQyxDQUFDLENBQUMsS0FBSyxFQUFFLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUMvRCxDQUFDO1FBQ0QsTUFBTSxDQUFDLFFBQVEsQ0FBQztJQUNsQixDQUFDO0NBQ0Y7QUFwQ0Qsa0NBb0NDO0FBRUQsa0JBQTBCLFNBQVEsUUFBUTtJQUExQzs7UUFDVSxhQUFRLEdBQTRCLElBQUksQ0FBQztJQW9CbkQsQ0FBQztJQWhCQyxPQUFPLENBQUMsR0FBZ0I7UUFDdEIsSUFBSSxRQUFRLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQztRQUM3QixFQUFFLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7WUFDZCxJQUFJLEtBQUssR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDO1lBQzdCLElBQUksQ0FBQyxHQUFHLGtCQUFPLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBRTVCLEdBQUcsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQztnQkFDaEQsSUFBSSxTQUFTLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDbkMsSUFBSSxPQUFPLEdBQUcsdUJBQVUsQ0FBQyxVQUFVLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO2dCQUN0RCxzQkFBVSxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDakMsQ0FBQztZQUVELFFBQVEsR0FBRyxJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksd0JBQWUsQ0FBQyxDQUFDLENBQUMsS0FBSyxFQUFFLENBQUMsQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzdFLENBQUM7UUFDRCxNQUFNLENBQUMsUUFBUSxDQUFDO0lBQ2xCLENBQUM7Q0FDRjtBQXJCRCxvQ0FxQkM7QUFFRDtJQUNFLFlBQW9CLEtBQThCLEVBQVUsSUFBa0IsRUFBVSxHQUFnQjtRQUFwRixVQUFLLEdBQUwsS0FBSyxDQUF5QjtRQUFVLFNBQUksR0FBSixJQUFJLENBQWM7UUFBVSxRQUFHLEdBQUgsR0FBRyxDQUFhO0lBQ3hHLENBQUM7SUFFRCxjQUFjO1FBQ1osSUFBSSxFQUFFLEtBQUssRUFBRSxJQUFJLEVBQUUsR0FBRyxJQUFJLENBQUM7UUFFM0IsSUFBSSxXQUFXLEdBQUcseUJBQWUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN4QyxJQUFJLEtBQUssR0FBRyxTQUFTLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRSxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDcEQsTUFBTSxDQUFDLElBQUksVUFBVSxDQUFDLEtBQUssQ0FBQyxVQUFVLEVBQUUsV0FBVyxDQUFDLENBQUM7SUFDdkQsQ0FBQztJQUVELFVBQVU7UUFDUixJQUFJLEVBQUUsS0FBSyxFQUFFLElBQUksRUFBRSxHQUFHLElBQUksQ0FBQztRQUMzQixJQUFJLEVBQUUsS0FBSyxFQUFFLE1BQU0sRUFBRSxXQUFXLEVBQUUsR0FBRyxLQUFLLENBQUM7UUFFM0MsSUFBSSxXQUFXLEdBQUcscUJBQVcsQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFLE1BQU0sRUFBRSxXQUFXLENBQUMsQ0FBQztRQUNoRSxJQUFJLEtBQUssR0FBRyxTQUFTLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRSxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7UUFFcEQsTUFBTSxDQUFDLElBQUksTUFBTSxDQUFDLEtBQUssQ0FBQyxVQUFVLEVBQUUsV0FBVyxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVELFdBQVcsQ0FBQyxXQUF3QjtRQUNsQyxJQUFJLEVBQUUsS0FBSyxFQUFFLEdBQUcsSUFBSSxDQUFDO1FBRXJCLElBQUksS0FBSyxHQUFHLFNBQVMsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUVwRCxNQUFNLENBQUMsSUFBSSxZQUFZLENBQUMsS0FBSyxDQUFDLFVBQVUsRUFBRSxXQUFXLENBQUMsQ0FBQztJQUN6RCxDQUFDO0NBQ0Y7QUE3QkQsMEJBNkJDO0FBRUQsbUJBQTBCLEVBQUUsVUFBVSxFQUFtQixFQUFFLFdBQXdCLEVBQUUsR0FBZ0I7SUFDbkcsTUFBTSxDQUFDLElBQUksY0FBYyxDQUFDLEdBQUcsRUFBRSxXQUFXLEVBQUUsVUFBVSxDQUFDLENBQUMsSUFBSSxFQUFFLENBQUM7QUFDakUsQ0FBQztBQUZELDhCQUVDO0FBS0QsSUFBaUIsY0FBYyxDQWdGOUI7QUFoRkQsV0FBaUIsY0FBYztJQUc3QixNQUFNLEVBQUUsR0FBRyxFQUFFLEdBQUcsVUFBVSxDQUFDO0lBSWQsaUNBQWtCLEdBQUcsVUFBVSxDQUFDLEVBQUUsQ0FBbUIsR0FBRyxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFPM0UsaUNBQWtCLEdBQUcsVUFBVSxDQUFDLEVBQUUsQ0FBdUIsR0FBRyxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFHbkYsZ0NBQWlCLEdBQUcsVUFBVSxDQUFDLEVBQUUsQ0FBa0IsR0FBRyxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBR3hFLGtDQUFtQixHQUFHLFVBQVUsQ0FBQyxFQUFFLENBQW9CLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0lBRzlFLHdCQUFTLEdBQUcsVUFBVSxDQUFDLEVBQUUsQ0FBaUIsR0FBRyxDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBRzlELDhCQUFlLEdBQUcsVUFBVSxDQUFDLEVBQUUsQ0FBZ0IsR0FBRyxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBRWxFLCtCQUFnQixHQUFHLFVBQVUsQ0FBQyxFQUFFLENBQWlCLEdBQUcsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUlyRSxtQ0FBb0IsR0FBRyxVQUFVLENBQUMsRUFBRSxDQUFxQixHQUFHLENBQUMsUUFBUSxDQUFDLENBQUM7SUFHdkUsNEJBQWEsR0FBRyxVQUFVLENBQUMsRUFBRSxDQUFjLEdBQUcsQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUc1RCw2QkFBYyxHQUFHLFVBQVUsQ0FBQyxFQUFFLENBQWUsR0FBRyxDQUFDLFlBQVksQ0FBQyxDQUFDO0lBRy9ELHlCQUFVLEdBQUcsVUFBVSxDQUFDLEVBQUUsQ0FBVyxHQUFHLENBQUMsUUFBUSxDQUFDLENBQUM7SUFJaEUsSUFBaUIsV0FBVyxDQWdCM0I7SUFoQkQsV0FBaUIsV0FBVztRQUMxQixzQkFBNkIsSUFBaUI7WUFDNUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNqQixDQUFDO1FBRmUsd0JBQVksZUFFM0IsQ0FBQTtRQUVELHNCQUE2QixJQUFpQjtZQUM1QyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ2pCLENBQUM7UUFGZSx3QkFBWSxlQUUzQixDQUFBO1FBRUQsZ0JBQXVCLElBQWlCO1lBQ3RDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDakIsQ0FBQztRQUZlLGtCQUFNLFNBRXJCLENBQUE7UUFFRCxjQUFxQixJQUFpQjtZQUNwQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ2pCLENBQUM7UUFGZSxnQkFBSSxPQUVuQixDQUFBO0lBQ0gsQ0FBQyxFQWhCZ0IsV0FBVyxHQUFYLDBCQUFXLEtBQVgsMEJBQVcsUUFnQjNCO0FBbUJILENBQUMsRUFoRmdCLGNBQWMsR0FBZCxzQkFBYyxLQUFkLHNCQUFjLFFBZ0Y5QjtBQUVELE1BQU0sRUFBRSxHQUFHLEVBQUUsR0FBRyxVQUFVLENBQUM7QUFFM0I7SUFDRSxZQUFvQixHQUFnQixFQUFVLEtBQWtCLEVBQVUsVUFBaUM7UUFBdkYsUUFBRyxHQUFILEdBQUcsQ0FBYTtRQUFVLFVBQUssR0FBTCxLQUFLLENBQWE7UUFBVSxlQUFVLEdBQVYsVUFBVSxDQUF1QjtJQUFHLENBQUM7SUFFL0csSUFBSTtRQUNGLElBQUksTUFBTSxHQUFrQyxFQUFFLENBQUM7UUFFL0MsR0FBRyxDQUFBLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDO1lBQy9DLElBQUksU0FBUyxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDbkMsRUFBRSxDQUFDLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUM3QyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQztZQUMvQyxDQUFDO1lBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDeEQsTUFBTSxDQUFDLElBQUksQ0FBQyxHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO1lBQ3RELENBQUM7WUFBQyxJQUFJLENBQUMsQ0FBQztnQkFDTixNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1lBQ3pCLENBQUM7UUFDSCxDQUFDO1FBRUQsTUFBTSxDQUFDLElBQUksV0FBVyxDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVPLGVBQWUsQ0FBQyxLQUFrQztRQUN4RCxJQUFJLENBQUMsRUFBRSxJQUFJLEVBQUUsTUFBTSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsT0FBTyxDQUFDLEdBQUcsS0FBSyxDQUFDO1FBQ3RELE1BQU0sQ0FBQyxDQUFDLEdBQUcsQ0FBQyxZQUFZLEVBQUUsSUFBSSxFQUFFLE1BQU0sRUFBRSxJQUFJLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7SUFDM0YsQ0FBQztJQUVPLG1CQUFtQixDQUFDLElBQXFDO1FBQy9ELElBQUksQ0FBQyxFQUFFLEdBQUcsRUFBRSxTQUFTLENBQUMsR0FBRyxJQUFJLENBQUM7UUFFOUIsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxHQUFHLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNyRCxJQUFJLEtBQUssR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1lBQ2xDLElBQUksS0FBSyxHQUFHLElBQUksY0FBYyxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsSUFBSSxDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDdEUsTUFBTSxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLEVBQUUsR0FBRyxFQUFFLEtBQUssRUFBRSxTQUFTLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFDckUsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sSUFBSSxHQUFHLEdBQWtDLEVBQUUsQ0FBQztZQUM1QyxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUMsR0FBRyxDQUFDLFdBQVcsRUFBRSxHQUFHLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQztZQUNyQyxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQzdCLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQztZQUM3QixHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsU0FBUyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1lBQ2xDLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQztZQUM3QixNQUFNLENBQUMsR0FBRyxDQUFDO1FBQ2IsQ0FBQztJQUNILENBQUM7SUFFRCxLQUFLLENBQUMsS0FBOEI7UUFDbEMsRUFBRSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUM7WUFBQyxNQUFNLENBQUMsSUFBSSxDQUFDO1FBQ3hCLElBQUksS0FBSyxHQUFHLG9CQUFVLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDakQsTUFBTSxDQUFDLElBQUksY0FBYyxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsS0FBSyxFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUMvRCxDQUFDO0NBQ0Y7QUFoREQsd0NBZ0RDIn0=