"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const WireFormat = require("@glimmer/wire-format");
const value_1 = require("../compiled/expressions/value");
const has_block_1 = require("../compiled/expressions/has-block");
const scanner_1 = require("../scanner");
const util_1 = require("@glimmer/util");
const lookups_1 = require("../compiled/expressions/lookups");
const helper_1 = require("../compiled/expressions/helper");
const concat_1 = require("../compiled/expressions/concat");
const args_1 = require("../compiled/expressions/args");
const has_block_2 = require("../compiled/expressions/has-block");
const function_1 = require("../compiled/expressions/function");
const { defaultBlock, params, hash } = scanner_1.BaselineSyntax.NestedBlock;
function debugCallback(context, get) {
    console.info('Use `context`, and `get(<path>)` to debug this template.');
    /* tslint:disable */
    debugger;
    /* tslint:enable */
    return { context, get };
}
function getter(vm, builder) {
    return (path) => {
        let parts = path.split('.');
        if (parts[0] === 'this') {
            parts[0] = null;
        }
        return compileRef(parts, builder).evaluate(vm);
    };
}
let callback = debugCallback;
// For testing purposes
function setDebuggerCallback(cb) {
    callback = cb;
}
exports.setDebuggerCallback = setDebuggerCallback;
function resetDebuggerCallback() {
    callback = debugCallback;
}
exports.resetDebuggerCallback = resetDebuggerCallback;
class Compilers {
    constructor() {
        this.names = util_1.dict();
        this.funcs = [];
    }
    add(name, func) {
        this.funcs.push(func);
        this.names[name] = this.funcs.length - 1;
    }
    compile(sexp, builder) {
        let name = sexp[0];
        let index = this.names[name];
        let func = this.funcs[index];
        util_1.assert(!!func, `expected an implementation for ${sexp[0]}`);
        return func(sexp, builder);
    }
}
exports.Compilers = Compilers;
let { Ops } = WireFormat;
exports.STATEMENTS = new Compilers();
exports.STATEMENTS.add(Ops.Text, (sexp, builder) => {
    builder.text(sexp[1]);
});
exports.STATEMENTS.add(Ops.Comment, (sexp, builder) => {
    builder.comment(sexp[1]);
});
exports.STATEMENTS.add(Ops.CloseElement, (_sexp, builder) => {
    util_1.LOGGER.trace('close-element statement');
    builder.closeElement();
});
exports.STATEMENTS.add(Ops.FlushElement, (_sexp, builder) => {
    builder.flushElement();
});
exports.STATEMENTS.add(Ops.Modifier, (sexp, builder) => {
    let [, path, params, hash] = sexp;
    let args = compileArgs(params, hash, builder);
    if (builder.env.hasModifier(path[0], builder.symbolTable)) {
        builder.modifier(path[0], args);
    }
    else {
        throw new Error(`Compile Error ${path.join('.')} is not a modifier: Helpers may not be used in the element form.`);
    }
});
exports.STATEMENTS.add(Ops.StaticAttr, (sexp, builder) => {
    let [, name, value, namespace] = sexp;
    builder.staticAttr(name, namespace, value);
});
exports.STATEMENTS.add(Ops.AnyDynamicAttr, (sexp, builder) => {
    let [, name, value, namespace, trusting] = sexp;
    builder.putValue(value);
    if (namespace) {
        builder.dynamicAttrNS(name, namespace, trusting);
    }
    else {
        builder.dynamicAttr(name, trusting);
    }
});
exports.STATEMENTS.add(Ops.OpenElement, (sexp, builder) => {
    util_1.LOGGER.trace('open-element statement');
    builder.openPrimitiveElement(sexp[1]);
});
exports.STATEMENTS.add(Ops.OptimizedAppend, (sexp, builder) => {
    let [, value, trustingMorph] = sexp;
    let { inlines } = builder.env.macros();
    let returned = inlines.compile(sexp, builder) || value;
    if (returned === true)
        return;
    builder.putValue(returned[1]);
    if (trustingMorph) {
        builder.trustingAppend();
    }
    else {
        builder.cautiousAppend();
    }
});
exports.STATEMENTS.add(Ops.UnoptimizedAppend, (sexp, builder) => {
    let [, value, trustingMorph] = sexp;
    let { inlines } = builder.env.macros();
    let returned = inlines.compile(sexp, builder) || value;
    if (returned === true)
        return;
    if (trustingMorph) {
        builder.guardedTrustingAppend(returned[1]);
    }
    else {
        builder.guardedCautiousAppend(returned[1]);
    }
});
exports.STATEMENTS.add(Ops.NestedBlock, (sexp, builder) => {
    let { blocks } = builder.env.macros();
    blocks.compile(sexp, builder);
});
exports.STATEMENTS.add(Ops.ScannedBlock, (sexp, builder) => {
    let [, path, params, hash, template, inverse] = sexp;
    let templateBlock = template && template.scan();
    let inverseBlock = inverse && inverse.scan();
    let { blocks } = builder.env.macros();
    blocks.compile([Ops.NestedBlock, path, params, hash, templateBlock, inverseBlock], builder);
});
// this fixes an issue with Ember versions using glimmer-vm@0.22 when attempting
// to use nested web components.  This is obviously not correct for angle bracket components
// but since no consumers are currently using them with glimmer@0.22.x we are hard coding
// support to just use the fallback case.
exports.STATEMENTS.add(Ops.Component, (sexp, builder) => {
    let [, tag, component] = sexp;
    let { attrs, statements } = component;
    builder.openPrimitiveElement(tag);
    for (let i = 0; i < attrs.length; i++) {
        exports.STATEMENTS.compile(attrs[i], builder);
    }
    builder.flushElement();
    for (let i = 0; i < statements.length; i++) {
        exports.STATEMENTS.compile(statements[i], builder);
    }
    builder.closeElement();
});
exports.STATEMENTS.add(Ops.ScannedComponent, (sexp, builder) => {
    let [, tag, attrs, rawArgs, rawBlock] = sexp;
    let block = rawBlock && rawBlock.scan();
    let args = compileBlockArgs(null, rawArgs, { default: block, inverse: null }, builder);
    let definition = builder.env.getComponentDefinition(tag, builder.symbolTable);
    builder.putComponentDefinition(definition);
    builder.openComponent(args, attrs.scan());
    builder.closeComponent();
});
exports.STATEMENTS.add(Ops.StaticPartial, (sexp, builder) => {
    let [, name] = sexp;
    if (!builder.env.hasPartial(name, builder.symbolTable)) {
        throw new Error(`Compile Error: Could not find a partial named "${name}"`);
    }
    let definition = builder.env.lookupPartial(name, builder.symbolTable);
    builder.putPartialDefinition(definition);
    builder.evaluatePartial();
});
exports.STATEMENTS.add(Ops.DynamicPartial, (sexp, builder) => {
    let [, name] = sexp;
    builder.startLabels();
    builder.putValue(name);
    builder.test('simple');
    builder.enter('BEGIN', 'END');
    builder.label('BEGIN');
    builder.jumpUnless('END');
    builder.putDynamicPartialDefinition();
    builder.evaluatePartial();
    builder.label('END');
    builder.exit();
    builder.stopLabels();
});
exports.STATEMENTS.add(Ops.Yield, function (sexp, builder) {
    let [, to, params] = sexp;
    let args = compileArgs(params, null, builder);
    builder.yield(args, to);
});
exports.STATEMENTS.add(Ops.Debugger, (sexp, builder) => {
    builder.putValue([Ops.Function, (vm) => {
            let context = vm.getSelf().value();
            let get = (path) => {
                return getter(vm, builder)(path).value();
            };
            callback(context, get);
        }]);
    return sexp;
});
let EXPRESSIONS = new Compilers();
function expr(expression, builder) {
    if (Array.isArray(expression)) {
        return EXPRESSIONS.compile(expression, builder);
    }
    else {
        return new value_1.default(expression);
    }
}
exports.expr = expr;
EXPRESSIONS.add(Ops.Unknown, (sexp, builder) => {
    let path = sexp[1];
    let name = path[0];
    if (builder.env.hasHelper(name, builder.symbolTable)) {
        return new helper_1.default(name, builder.env.lookupHelper(name, builder.symbolTable), args_1.CompiledArgs.empty(), builder.symbolTable);
    }
    else {
        return compileRef(path, builder);
    }
});
EXPRESSIONS.add(Ops.Concat, ((sexp, builder) => {
    let params = sexp[1].map(p => expr(p, builder));
    return new concat_1.default(params);
}));
EXPRESSIONS.add(Ops.Function, (sexp, builder) => {
    return new function_1.CompiledFunctionExpression(sexp[1], builder.symbolTable);
});
EXPRESSIONS.add(Ops.Helper, (sexp, builder) => {
    let { env, symbolTable } = builder;
    let [, [name], params, hash] = sexp;
    if (env.hasHelper(name, symbolTable)) {
        let args = compileArgs(params, hash, builder);
        return new helper_1.default(name, env.lookupHelper(name, symbolTable), args, symbolTable);
    }
    else {
        throw new Error(`Compile Error: ${name} is not a helper`);
    }
});
EXPRESSIONS.add(Ops.Get, (sexp, builder) => {
    return compileRef(sexp[1], builder);
});
EXPRESSIONS.add(Ops.Undefined, (_sexp, _builder) => {
    return new value_1.default(undefined);
});
EXPRESSIONS.add(Ops.Arg, (sexp, builder) => {
    let [, parts] = sexp;
    let head = parts[0];
    let named, partial;
    if (named = builder.symbolTable.getSymbol('named', head)) {
        let path = parts.slice(1);
        let inner = new lookups_1.CompiledSymbol(named, head);
        return lookups_1.default.create(inner, path);
    }
    else if (partial = builder.symbolTable.getPartialArgs()) {
        let path = parts.slice(1);
        let inner = new lookups_1.CompiledInPartialName(partial, head);
        return lookups_1.default.create(inner, path);
    }
    else {
        throw new Error(`[BUG] @${parts.join('.')} is not a valid lookup path.`);
    }
});
EXPRESSIONS.add(Ops.HasBlock, (sexp, builder) => {
    let blockName = sexp[1];
    let yields, partial;
    if (yields = builder.symbolTable.getSymbol('yields', blockName)) {
        let inner = new has_block_2.CompiledGetBlockBySymbol(yields, blockName);
        return new has_block_1.default(inner);
    }
    else if (partial = builder.symbolTable.getPartialArgs()) {
        let inner = new has_block_2.CompiledInPartialGetBlock(partial, blockName);
        return new has_block_1.default(inner);
    }
    else {
        throw new Error('[BUG] ${blockName} is not a valid block name.');
    }
});
EXPRESSIONS.add(Ops.HasBlockParams, (sexp, builder) => {
    let blockName = sexp[1];
    let yields, partial;
    if (yields = builder.symbolTable.getSymbol('yields', blockName)) {
        let inner = new has_block_2.CompiledGetBlockBySymbol(yields, blockName);
        return new has_block_1.CompiledHasBlockParams(inner);
    }
    else if (partial = builder.symbolTable.getPartialArgs()) {
        let inner = new has_block_2.CompiledInPartialGetBlock(partial, blockName);
        return new has_block_1.CompiledHasBlockParams(inner);
    }
    else {
        throw new Error('[BUG] ${blockName} is not a valid block name.');
    }
});
function compileArgs(params, hash, builder) {
    let compiledParams = compileParams(params, builder);
    let compiledHash = compileHash(hash, builder);
    return args_1.CompiledArgs.create(compiledParams, compiledHash, args_1.EMPTY_BLOCKS);
}
exports.compileArgs = compileArgs;
function compileBlockArgs(params, hash, blocks, builder) {
    let compiledParams = compileParams(params, builder);
    let compiledHash = compileHash(hash, builder);
    return args_1.CompiledArgs.create(compiledParams, compiledHash, blocks);
}
exports.compileBlockArgs = compileBlockArgs;
function compileBaselineArgs(args, builder) {
    let [params, hash, _default, inverse] = args;
    return args_1.CompiledArgs.create(compileParams(params, builder), compileHash(hash, builder), { default: _default, inverse });
}
exports.compileBaselineArgs = compileBaselineArgs;
function compileParams(params, builder) {
    if (!params || params.length === 0)
        return args_1.COMPILED_EMPTY_POSITIONAL_ARGS;
    let compiled = new Array(params.length);
    for (let i = 0; i < params.length; i++) {
        compiled[i] = expr(params[i], builder);
    }
    return args_1.CompiledPositionalArgs.create(compiled);
}
function compileHash(hash, builder) {
    if (!hash)
        return args_1.COMPILED_EMPTY_NAMED_ARGS;
    let [keys, values] = hash;
    if (keys.length === 0)
        return args_1.COMPILED_EMPTY_NAMED_ARGS;
    let compiled = new Array(values.length);
    for (let i = 0; i < values.length; i++) {
        compiled[i] = expr(values[i], builder);
    }
    return new args_1.CompiledNamedArgs(keys, compiled);
}
function compileRef(parts, builder) {
    let head = parts[0];
    let local;
    if (head === null) {
        let inner = new lookups_1.CompiledSelf();
        let path = parts.slice(1);
        return lookups_1.default.create(inner, path);
    }
    else if (local = builder.symbolTable.getSymbol('local', head)) {
        let path = parts.slice(1);
        let inner = new lookups_1.CompiledSymbol(local, head);
        return lookups_1.default.create(inner, path);
    }
    else {
        let inner = new lookups_1.CompiledSelf();
        return lookups_1.default.create(inner, parts);
    }
}
class Blocks {
    constructor() {
        this.names = util_1.dict();
        this.funcs = [];
    }
    add(name, func) {
        this.funcs.push(func);
        this.names[name] = this.funcs.length - 1;
    }
    addMissing(func) {
        this.missing = func;
    }
    compile(sexp, builder) {
        // assert(sexp[1].length === 1, 'paths in blocks are not supported');
        let name = sexp[1][0];
        let index = this.names[name];
        if (index === undefined) {
            util_1.assert(!!this.missing, `${name} not found, and no catch-all block handler was registered`);
            let func = this.missing;
            let handled = func(sexp, builder);
            util_1.assert(!!handled, `${name} not found, and the catch-all block handler didn't handle it`);
        }
        else {
            let func = this.funcs[index];
            func(sexp, builder);
        }
    }
}
exports.Blocks = Blocks;
exports.BLOCKS = new Blocks();
class Inlines {
    constructor() {
        this.names = util_1.dict();
        this.funcs = [];
    }
    add(name, func) {
        this.funcs.push(func);
        this.names[name] = this.funcs.length - 1;
    }
    addMissing(func) {
        this.missing = func;
    }
    compile(sexp, builder) {
        let value = sexp[1];
        // TODO: Fix this so that expression macros can return
        // things like components, so that {{component foo}}
        // is the same as {{(component foo)}}
        if (!Array.isArray(value))
            return ['expr', value];
        let path;
        let params;
        let hash;
        if (value[0] === Ops.Helper) {
            path = value[1];
            params = value[2];
            hash = value[3];
        }
        else if (value[0] === Ops.Unknown) {
            path = value[1];
            params = hash = null;
        }
        else {
            return ['expr', value];
        }
        if (path.length > 1 && !params && !hash) {
            return ['expr', value];
        }
        let name = path[0];
        let index = this.names[name];
        if (index === undefined && this.missing) {
            let func = this.missing;
            let returned = func(path, params, hash, builder);
            return returned === false ? ['expr', value] : returned;
        }
        else if (index !== undefined) {
            let func = this.funcs[index];
            let returned = func(path, params, hash, builder);
            return returned === false ? ['expr', value] : returned;
        }
        else {
            return ['expr', value];
        }
    }
}
exports.Inlines = Inlines;
exports.INLINES = new Inlines();
populateBuiltins(exports.BLOCKS, exports.INLINES);
function populateBuiltins(blocks = new Blocks(), inlines = new Inlines()) {
    blocks.add('if', (sexp, builder) => {
        //        PutArgs
        //        Test(Environment)
        //        Enter(BEGIN, END)
        // BEGIN: Noop
        //        JumpUnless(ELSE)
        //        Evaluate(default)
        //        Jump(END)
        // ELSE:  Noop
        //        Evalulate(inverse)
        // END:   Noop
        //        Exit
        let [, , params, hash, _default, inverse] = sexp;
        let args = compileArgs(params, hash, builder);
        builder.putArgs(args);
        builder.test('environment');
        builder.labelled(null, b => {
            if (_default && inverse) {
                b.jumpUnless('ELSE');
                b.evaluate(_default);
                b.jump('END');
                b.label('ELSE');
                b.evaluate(inverse);
            }
            else if (_default) {
                b.jumpUnless('END');
                b.evaluate(_default);
            }
            else {
                throw util_1.unreachable();
            }
        });
    });
    blocks.add('-in-element', (sexp, builder) => {
        let block = defaultBlock(sexp);
        let args = compileArgs(params(sexp), null, builder);
        builder.putArgs(args);
        builder.test('simple');
        builder.labelled(null, b => {
            b.jumpUnless('END');
            b.pushRemoteElement();
            b.evaluate(util_1.unwrap(block));
            b.popRemoteElement();
        });
    });
    blocks.add('-with-dynamic-vars', (sexp, builder) => {
        let block = defaultBlock(sexp);
        let args = compileArgs(params(sexp), hash(sexp), builder);
        builder.unit(b => {
            b.putArgs(args);
            b.pushDynamicScope();
            b.bindDynamicScope(args.named.keys);
            b.evaluate(util_1.unwrap(block));
            b.popDynamicScope();
        });
    });
    blocks.add('unless', (sexp, builder) => {
        //        PutArgs
        //        Test(Environment)
        //        Enter(BEGIN, END)
        // BEGIN: Noop
        //        JumpUnless(ELSE)
        //        Evaluate(default)
        //        Jump(END)
        // ELSE:  Noop
        //        Evalulate(inverse)
        // END:   Noop
        //        Exit
        let [, , params, hash, _default, inverse] = sexp;
        let args = compileArgs(params, hash, builder);
        builder.putArgs(args);
        builder.test('environment');
        builder.labelled(null, b => {
            if (_default && inverse) {
                b.jumpIf('ELSE');
                b.evaluate(_default);
                b.jump('END');
                b.label('ELSE');
                b.evaluate(inverse);
            }
            else if (_default) {
                b.jumpIf('END');
                b.evaluate(_default);
            }
            else {
                throw util_1.unreachable();
            }
        });
    });
    blocks.add('with', (sexp, builder) => {
        //        PutArgs
        //        Test(Environment)
        //        Enter(BEGIN, END)
        // BEGIN: Noop
        //        JumpUnless(ELSE)
        //        Evaluate(default)
        //        Jump(END)
        // ELSE:  Noop
        //        Evalulate(inverse)
        // END:   Noop
        //        Exit
        let [, , params, hash, _default, inverse] = sexp;
        let args = compileArgs(params, hash, builder);
        builder.putArgs(args);
        builder.test('environment');
        builder.labelled(null, b => {
            if (_default && inverse) {
                b.jumpUnless('ELSE');
                b.evaluate(_default);
                b.jump('END');
                b.label('ELSE');
                b.evaluate(inverse);
            }
            else if (_default) {
                b.jumpUnless('END');
                b.evaluate(_default);
            }
            else {
                throw util_1.unreachable();
            }
        });
    });
    blocks.add('each', (sexp, builder) => {
        //         Enter(BEGIN, END)
        // BEGIN:  Noop
        //         PutArgs
        //         PutIterable
        //         JumpUnless(ELSE)
        //         EnterList(BEGIN2, END2)
        // ITER:   Noop
        //         NextIter(BREAK)
        //         EnterWithKey(BEGIN2, END2)
        // BEGIN2: Noop
        //         PushChildScope
        //         Evaluate(default)
        //         PopScope
        // END2:   Noop
        //         Exit
        //         Jump(ITER)
        // BREAK:  Noop
        //         ExitList
        //         Jump(END)
        // ELSE:   Noop
        //         Evalulate(inverse)
        // END:    Noop
        //         Exit
        let [, , params, hash, _default, inverse] = sexp;
        let args = compileArgs(params, hash, builder);
        builder.labelled(args, b => {
            b.putIterator();
            if (inverse) {
                b.jumpUnless('ELSE');
            }
            else {
                b.jumpUnless('END');
            }
            b.iter(b => {
                b.evaluate(util_1.unwrap(_default));
            });
            if (inverse) {
                b.jump('END');
                b.label('ELSE');
                b.evaluate(inverse);
            }
        });
    });
    return { blocks, inlines };
}
exports.populateBuiltins = populateBuiltins;
//# sourceMappingURL=data:application/json;base64,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