"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const util_1 = require("@glimmer/util");
const bounds_1 = require("./bounds");
function isSafeString(value) {
    return !!value && typeof value['toHTML'] === 'function';
}
exports.isSafeString = isSafeString;
function isNode(value) {
    return value !== null && typeof value === 'object' && typeof value['nodeType'] === 'number';
}
exports.isNode = isNode;
function isString(value) {
    return typeof value === 'string';
}
exports.isString = isString;
class Upsert {
    constructor(bounds) {
        this.bounds = bounds;
    }
}
exports.default = Upsert;
function cautiousInsert(dom, cursor, value) {
    if (isString(value)) {
        return TextUpsert.insert(dom, cursor, value);
    }
    if (isSafeString(value)) {
        return SafeStringUpsert.insert(dom, cursor, value);
    }
    if (isNode(value)) {
        return NodeUpsert.insert(dom, cursor, value);
    }
    throw util_1.unreachable();
}
exports.cautiousInsert = cautiousInsert;
function trustingInsert(dom, cursor, value) {
    if (isString(value)) {
        return HTMLUpsert.insert(dom, cursor, value);
    }
    if (isNode(value)) {
        return NodeUpsert.insert(dom, cursor, value);
    }
    throw util_1.unreachable();
}
exports.trustingInsert = trustingInsert;
class TextUpsert extends Upsert {
    static insert(dom, cursor, value) {
        let textNode = dom.createTextNode(value);
        dom.insertBefore(cursor.element, textNode, cursor.nextSibling);
        let bounds = new bounds_1.SingleNodeBounds(cursor.element, textNode);
        return new TextUpsert(bounds, textNode);
    }
    constructor(bounds, textNode) {
        super(bounds);
        this.textNode = textNode;
    }
    update(_dom, value) {
        if (isString(value)) {
            let { textNode } = this;
            textNode.nodeValue = value;
            return true;
        }
        else {
            return false;
        }
    }
}
class HTMLUpsert extends Upsert {
    static insert(dom, cursor, value) {
        let bounds = dom.insertHTMLBefore(cursor.element, value, cursor.nextSibling);
        return new HTMLUpsert(bounds);
    }
    update(dom, value) {
        if (isString(value)) {
            let { bounds } = this;
            let parentElement = bounds.parentElement();
            let nextSibling = bounds_1.clear(bounds);
            this.bounds = dom.insertHTMLBefore(parentElement, nextSibling, value);
            return true;
        }
        else {
            return false;
        }
    }
}
class SafeStringUpsert extends Upsert {
    constructor(bounds, lastStringValue) {
        super(bounds);
        this.lastStringValue = lastStringValue;
    }
    static insert(dom, cursor, value) {
        let stringValue = value.toHTML();
        let bounds = dom.insertHTMLBefore(cursor.element, stringValue, cursor.nextSibling);
        return new SafeStringUpsert(bounds, stringValue);
    }
    update(dom, value) {
        if (isSafeString(value)) {
            let stringValue = value.toHTML();
            if (stringValue !== this.lastStringValue) {
                let { bounds } = this;
                let parentElement = bounds.parentElement();
                let nextSibling = bounds_1.clear(bounds);
                this.bounds = dom.insertHTMLBefore(parentElement, nextSibling, stringValue);
                this.lastStringValue = stringValue;
            }
            return true;
        }
        else {
            return false;
        }
    }
}
class NodeUpsert extends Upsert {
    static insert(dom, cursor, node) {
        dom.insertBefore(cursor.element, node, cursor.nextSibling);
        return new NodeUpsert(bounds_1.single(cursor.element, node));
    }
    update(dom, value) {
        if (isNode(value)) {
            let { bounds } = this;
            let parentElement = bounds.parentElement();
            let nextSibling = bounds_1.clear(bounds);
            this.bounds = dom.insertNodeBefore(parentElement, value, nextSibling);
            return true;
        }
        else {
            return false;
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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