"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const environment_1 = require("../environment");
const util_1 = require("@glimmer/util");
const reference_1 = require("@glimmer/reference");
const vm_1 = require("../compiled/opcodes/vm");
const update_1 = require("./update");
const render_result_1 = require("./render-result");
const frame_1 = require("./frame");
const opcodes_1 = require("../opcodes");
class VM {
    constructor(env, scope, dynamicScope, elementStack) {
        this.env = env;
        this.elementStack = elementStack;
        this.dynamicScopeStack = new util_1.Stack();
        this.scopeStack = new util_1.Stack();
        this.updatingOpcodeStack = new util_1.Stack();
        this.cacheGroups = new util_1.Stack();
        this.listBlockStack = new util_1.Stack();
        this.frame = new frame_1.FrameStack();
        this.env = env;
        this.constants = env.constants;
        this.elementStack = elementStack;
        this.scopeStack.push(scope);
        this.dynamicScopeStack.push(dynamicScope);
    }
    static initial(env, self, dynamicScope, elementStack, compiledProgram) {
        let { symbols: size, start, end } = compiledProgram;
        let scope = environment_1.Scope.root(self, size);
        let vm = new VM(env, scope, dynamicScope, elementStack);
        vm.prepare(start, end);
        return vm;
    }
    capture() {
        return {
            env: this.env,
            scope: this.scope(),
            dynamicScope: this.dynamicScope(),
            frame: this.frame.capture()
        };
    }
    goto(ip) {
        this.frame.goto(ip);
    }
    beginCacheGroup() {
        this.cacheGroups.push(this.updating().tail());
    }
    commitCacheGroup() {
        //        JumpIfNotModified(END)
        //        (head)
        //        (....)
        //        (tail)
        //        DidModify
        // END:   Noop
        let END = new vm_1.LabelOpcode("END");
        let opcodes = this.updating();
        let marker = this.cacheGroups.pop();
        let head = marker ? opcodes.nextNode(marker) : opcodes.head();
        let tail = opcodes.tail();
        let tag = reference_1.combineSlice(new util_1.ListSlice(head, tail));
        let guard = new vm_1.JumpIfNotModifiedOpcode(tag, END);
        opcodes.insertBefore(guard, head);
        opcodes.append(new vm_1.DidModifyOpcode(guard));
        opcodes.append(END);
    }
    enter(start, end) {
        let updating = new util_1.LinkedList();
        let tracker = this.stack().pushUpdatableBlock();
        let state = this.capture();
        let tryOpcode = new update_1.TryOpcode(start, end, state, tracker, updating);
        this.didEnter(tryOpcode, updating);
    }
    enterWithKey(key, start, end) {
        let updating = new util_1.LinkedList();
        let tracker = this.stack().pushUpdatableBlock();
        let state = this.capture();
        let tryOpcode = new update_1.TryOpcode(start, end, state, tracker, updating);
        this.listBlock().map[key] = tryOpcode;
        this.didEnter(tryOpcode, updating);
    }
    enterList(start, end) {
        let updating = new util_1.LinkedList();
        let tracker = this.stack().pushBlockList(updating);
        let state = this.capture();
        let artifacts = this.frame.getIterator().artifacts;
        let opcode = new update_1.ListBlockOpcode(start, end, state, tracker, updating, artifacts);
        this.listBlockStack.push(opcode);
        this.didEnter(opcode, updating);
    }
    didEnter(opcode, updating) {
        this.updateWith(opcode);
        this.updatingOpcodeStack.push(updating);
    }
    exit() {
        this.stack().popBlock();
        this.updatingOpcodeStack.pop();
        let parent = this.updating().tail();
        parent.didInitializeChildren();
    }
    exitList() {
        this.exit();
        this.listBlockStack.pop();
    }
    updateWith(opcode) {
        this.updating().append(opcode);
    }
    listBlock() {
        return util_1.expect(this.listBlockStack.current, 'expected a list block');
    }
    updating() {
        return util_1.expect(this.updatingOpcodeStack.current, 'expected updating opcode on the updating opcode stack');
    }
    stack() {
        return this.elementStack;
    }
    scope() {
        return util_1.expect(this.scopeStack.current, 'expected scope on the scope stack');
    }
    dynamicScope() {
        return util_1.expect(this.dynamicScopeStack.current, 'expected dynamic scope on the dynamic scope stack');
    }
    pushFrame(block, args, callerScope) {
        this.frame.push(block.start, block.end);
        if (args)
            this.frame.setArgs(args);
        if (args && args.blocks)
            this.frame.setBlocks(args.blocks);
        if (callerScope)
            this.frame.setCallerScope(callerScope);
    }
    pushComponentFrame(layout, args, callerScope, component, manager, shadow) {
        this.frame.push(layout.start, layout.end, component, manager, shadow);
        if (args)
            this.frame.setArgs(args);
        if (args && args.blocks)
            this.frame.setBlocks(args.blocks);
        if (callerScope)
            this.frame.setCallerScope(callerScope);
    }
    pushEvalFrame(start, end) {
        this.frame.push(start, end);
    }
    pushChildScope() {
        this.scopeStack.push(this.scope().child());
    }
    pushCallerScope() {
        this.scopeStack.push(util_1.expect(this.scope().getCallerScope(), 'pushCallerScope is called when a caller scope is present'));
    }
    pushDynamicScope() {
        let child = this.dynamicScope().child();
        this.dynamicScopeStack.push(child);
        return child;
    }
    pushRootScope(self, size) {
        let scope = environment_1.Scope.root(self, size);
        this.scopeStack.push(scope);
        return scope;
    }
    popScope() {
        this.scopeStack.pop();
    }
    popDynamicScope() {
        this.dynamicScopeStack.pop();
    }
    newDestroyable(d) {
        this.stack().newDestroyable(d);
    }
    /// SCOPE HELPERS
    getSelf() {
        return this.scope().getSelf();
    }
    referenceForSymbol(symbol) {
        return this.scope().getSymbol(symbol);
    }
    getArgs() {
        return this.frame.getArgs();
    }
    /// EXECUTION
    resume(start, end, frame) {
        return this.execute(start, end, vm => vm.frame.restore(frame));
    }
    execute(start, end, initialize) {
        this.prepare(start, end, initialize);
        let result;
        while (true) {
            result = this.next();
            if (result.done)
                break;
        }
        return result.value;
    }
    prepare(start, end, initialize) {
        let { elementStack, frame, updatingOpcodeStack } = this;
        elementStack.pushSimpleBlock();
        updatingOpcodeStack.push(new util_1.LinkedList());
        frame.push(start, end);
        if (initialize)
            initialize(this);
    }
    next() {
        let { frame, env, updatingOpcodeStack, elementStack } = this;
        let opcode;
        if (opcode = frame.nextStatement(env)) {
            opcodes_1.APPEND_OPCODES.evaluate(this, opcode);
            return { done: false, value: null };
        }
        return {
            done: true,
            value: new render_result_1.default(env, util_1.expect(updatingOpcodeStack.pop(), 'there should be a final updating opcode stack'), elementStack.popBlock())
        };
    }
    evaluateOpcode(opcode) {
        opcodes_1.APPEND_OPCODES.evaluate(this, opcode);
    }
    // Make sure you have opcodes that push and pop a scope around this opcode
    // if you need to change the scope.
    invokeBlock(block, args) {
        let compiled = block.compile(this.env);
        this.pushFrame(compiled, args);
    }
    invokePartial(block) {
        let compiled = block.compile(this.env);
        this.pushFrame(compiled);
    }
    invokeLayout(args, layout, callerScope, component, manager, shadow) {
        this.pushComponentFrame(layout, args, callerScope, component, manager, shadow);
    }
    evaluateOperand(expr) {
        this.frame.setOperand(expr.evaluate(this));
    }
    evaluateArgs(args) {
        let evaledArgs = this.frame.setArgs(args.evaluate(this));
        this.frame.setOperand(evaledArgs.positional.at(0));
    }
    bindPositionalArgs(symbols) {
        let args = util_1.expect(this.frame.getArgs(), 'bindPositionalArgs assumes a previous setArgs');
        let { positional } = args;
        let scope = this.scope();
        for (let i = 0; i < symbols.length; i++) {
            scope.bindSymbol(symbols[i], positional.at(i));
        }
    }
    bindNamedArgs(names, symbols) {
        let args = util_1.expect(this.frame.getArgs(), 'bindNamedArgs assumes a previous setArgs');
        let scope = this.scope();
        let { named } = args;
        for (let i = 0; i < names.length; i++) {
            let name = this.constants.getString(names[i]);
            scope.bindSymbol(symbols[i], named.get(name));
        }
    }
    bindBlocks(names, symbols) {
        let blocks = this.frame.getBlocks();
        let scope = this.scope();
        for (let i = 0; i < names.length; i++) {
            let name = this.constants.getString(names[i]);
            scope.bindBlock(symbols[i], (blocks && blocks[name]) || null);
        }
    }
    bindPartialArgs(symbol) {
        let args = util_1.expect(this.frame.getArgs(), 'bindPartialArgs assumes a previous setArgs');
        let scope = this.scope();
        util_1.assert(args, "Cannot bind named args");
        scope.bindPartialArgs(symbol, args);
    }
    bindCallerScope() {
        let callerScope = this.frame.getCallerScope();
        let scope = this.scope();
        util_1.assert(callerScope, "Cannot bind caller scope");
        scope.bindCallerScope(callerScope);
    }
    bindDynamicScope(names) {
        let args = util_1.expect(this.frame.getArgs(), 'bindDynamicScope assumes a previous setArgs');
        let scope = this.dynamicScope();
        util_1.assert(args, "Cannot bind dynamic scope");
        for (let i = 0; i < names.length; i++) {
            let name = this.constants.getString(names[i]);
            scope.set(name, args.named.get(name));
        }
    }
}
exports.default = VM;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwZW5kLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vbGliL3ZtL2FwcGVuZC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLGdEQUEwRTtBQUUxRSx3Q0FBMEc7QUFDMUcsa0RBQWlFO0FBS2pFLCtDQUErRjtBQUUvRixxQ0FBNEU7QUFDNUUsbURBQTJDO0FBQzNDLG1DQUFvRDtBQUVwRCx3Q0FLb0I7QUFlcEI7SUF1QkUsWUFDUyxHQUFnQixFQUN2QixLQUFZLEVBQ1osWUFBMEIsRUFDbEIsWUFBMEI7UUFIM0IsUUFBRyxHQUFILEdBQUcsQ0FBYTtRQUdmLGlCQUFZLEdBQVosWUFBWSxDQUFjO1FBMUI1QixzQkFBaUIsR0FBRyxJQUFJLFlBQUssRUFBZ0IsQ0FBQztRQUM5QyxlQUFVLEdBQUcsSUFBSSxZQUFLLEVBQVMsQ0FBQztRQUNqQyx3QkFBbUIsR0FBRyxJQUFJLFlBQUssRUFBOEIsQ0FBQztRQUM5RCxnQkFBVyxHQUFHLElBQUksWUFBSyxFQUEwQixDQUFDO1FBQ2xELG1CQUFjLEdBQUcsSUFBSSxZQUFLLEVBQW1CLENBQUM7UUFDOUMsVUFBSyxHQUFHLElBQUksa0JBQVUsRUFBRSxDQUFDO1FBdUI5QixJQUFJLENBQUMsR0FBRyxHQUFHLEdBQUcsQ0FBQztRQUNmLElBQUksQ0FBQyxTQUFTLEdBQUcsR0FBRyxDQUFDLFNBQVMsQ0FBQztRQUMvQixJQUFJLENBQUMsWUFBWSxHQUFHLFlBQVksQ0FBQztRQUNqQyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUM1QixJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUF6QkQsTUFBTSxDQUFDLE9BQU8sQ0FDWixHQUFnQixFQUNoQixJQUEyQixFQUMzQixZQUEwQixFQUMxQixZQUEwQixFQUMxQixlQUFnQztRQUVoQyxJQUFJLEVBQUUsT0FBTyxFQUFFLElBQUksRUFBRSxLQUFLLEVBQUUsR0FBRyxFQUFFLEdBQUksZUFBZSxDQUFDO1FBQ3JELElBQUksS0FBSyxHQUFHLG1CQUFLLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNuQyxJQUFJLEVBQUUsR0FBRyxJQUFJLEVBQUUsQ0FBQyxHQUFHLEVBQUUsS0FBSyxFQUFFLFlBQVksRUFBRSxZQUFZLENBQUMsQ0FBQztRQUN4RCxFQUFFLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxHQUFHLENBQUMsQ0FBQztRQUN2QixNQUFNLENBQUMsRUFBRSxDQUFDO0lBQ1osQ0FBQztJQWVELE9BQU87UUFDTCxNQUFNLENBQUM7WUFDTCxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7WUFDYixLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssRUFBRTtZQUNuQixZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVksRUFBRTtZQUNqQyxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUU7U0FDNUIsQ0FBQztJQUNKLENBQUM7SUFFRCxJQUFJLENBQUMsRUFBVTtRQUNiLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRCxlQUFlO1FBQ2IsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVELGdCQUFnQjtRQUNkLGdDQUFnQztRQUNoQyxnQkFBZ0I7UUFDaEIsZ0JBQWdCO1FBQ2hCLGdCQUFnQjtRQUNoQixtQkFBbUI7UUFDbkIsY0FBYztRQUVkLElBQUksR0FBRyxHQUFHLElBQUksZ0JBQVcsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUVqQyxJQUFJLE9BQU8sR0FBRyxJQUFJLENBQUMsUUFBUSxFQUFFLENBQUM7UUFDOUIsSUFBSSxNQUFNLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUNwQyxJQUFJLElBQUksR0FBRyxNQUFNLEdBQUcsT0FBTyxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsR0FBRyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDOUQsSUFBSSxJQUFJLEdBQUcsT0FBTyxDQUFDLElBQUksRUFBRSxDQUFDO1FBQzFCLElBQUksR0FBRyxHQUFHLHdCQUFZLENBQUMsSUFBSSxnQkFBUyxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDO1FBRWxELElBQUksS0FBSyxHQUFHLElBQUksNEJBQXVCLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBRWxELE9BQU8sQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sQ0FBQyxNQUFNLENBQUMsSUFBSSxvQkFBZSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQsS0FBSyxDQUFDLEtBQWEsRUFBRSxHQUFXO1FBQzlCLElBQUksUUFBUSxHQUFHLElBQUksaUJBQVUsRUFBa0IsQ0FBQztRQUVoRCxJQUFJLE9BQU8sR0FBRyxJQUFJLENBQUMsS0FBSyxFQUFFLENBQUMsa0JBQWtCLEVBQUUsQ0FBQztRQUNoRCxJQUFJLEtBQUssR0FBRyxJQUFJLENBQUMsT0FBTyxFQUFFLENBQUM7UUFFM0IsSUFBSSxTQUFTLEdBQUcsSUFBSSxrQkFBUyxDQUFDLEtBQUssRUFBRSxHQUFHLEVBQUUsS0FBSyxFQUFFLE9BQU8sRUFBRSxRQUFRLENBQUMsQ0FBQztRQUVwRSxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQsWUFBWSxDQUFDLEdBQVcsRUFBRSxLQUFhLEVBQUUsR0FBVztRQUNsRCxJQUFJLFFBQVEsR0FBRyxJQUFJLGlCQUFVLEVBQWtCLENBQUM7UUFFaEQsSUFBSSxPQUFPLEdBQUcsSUFBSSxDQUFDLEtBQUssRUFBRSxDQUFDLGtCQUFrQixFQUFFLENBQUM7UUFDaEQsSUFBSSxLQUFLLEdBQUcsSUFBSSxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBRTNCLElBQUksU0FBUyxHQUFHLElBQUksa0JBQVMsQ0FBQyxLQUFLLEVBQUUsR0FBRyxFQUFFLEtBQUssRUFBRSxPQUFPLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFFcEUsSUFBSSxDQUFDLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsR0FBRyxTQUFTLENBQUM7UUFFdEMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVELFNBQVMsQ0FBQyxLQUFhLEVBQUUsR0FBVztRQUNsQyxJQUFJLFFBQVEsR0FBRyxJQUFJLGlCQUFVLEVBQWUsQ0FBQztRQUU3QyxJQUFJLE9BQU8sR0FBRyxJQUFJLENBQUMsS0FBSyxFQUFFLENBQUMsYUFBYSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ25ELElBQUksS0FBSyxHQUFHLElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQztRQUMzQixJQUFJLFNBQVMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsRUFBRSxDQUFDLFNBQVMsQ0FBQztRQUVuRCxJQUFJLE1BQU0sR0FBRyxJQUFJLHdCQUFlLENBQUMsS0FBSyxFQUFFLEdBQUcsRUFBRSxLQUFLLEVBQUUsT0FBTyxFQUFFLFFBQVEsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUVsRixJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUVqQyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxRQUFRLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRU8sUUFBUSxDQUFDLE1BQW1CLEVBQUUsUUFBb0M7UUFDeEUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUN4QixJQUFJLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRCxJQUFJO1FBQ0YsSUFBSSxDQUFDLEtBQUssRUFBRSxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBQ3hCLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUUvQixJQUFJLE1BQU0sR0FBRyxJQUFJLENBQUMsUUFBUSxFQUFFLENBQUMsSUFBSSxFQUFpQixDQUFDO1FBRW5ELE1BQU0sQ0FBQyxxQkFBcUIsRUFBRSxDQUFDO0lBQ2pDLENBQUM7SUFFRCxRQUFRO1FBQ04sSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ1osSUFBSSxDQUFDLGNBQWMsQ0FBQyxHQUFHLEVBQUUsQ0FBQztJQUM1QixDQUFDO0lBRUQsVUFBVSxDQUFDLE1BQXNCO1FBQy9CLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVELFNBQVM7UUFDUCxNQUFNLENBQUMsYUFBTSxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsT0FBTyxFQUFFLHVCQUF1QixDQUFDLENBQUM7SUFDdEUsQ0FBQztJQUVELFFBQVE7UUFDTixNQUFNLENBQUMsYUFBTSxDQUFDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxPQUFPLEVBQUUsdURBQXVELENBQUMsQ0FBQztJQUMzRyxDQUFDO0lBRUQsS0FBSztRQUNILE1BQU0sQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDO0lBQzNCLENBQUM7SUFFRCxLQUFLO1FBQ0gsTUFBTSxDQUFDLGFBQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQ0FBbUMsQ0FBQyxDQUFDO0lBQzlFLENBQUM7SUFFRCxZQUFZO1FBQ1YsTUFBTSxDQUFDLGFBQU0sQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsT0FBTyxFQUFFLG1EQUFtRCxDQUFDLENBQUM7SUFDckcsQ0FBQztJQUVELFNBQVMsQ0FDUCxLQUFvQixFQUNwQixJQUE0QixFQUM1QixXQUFtQjtRQUVuQixJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUV4QyxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUM7WUFBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNuQyxFQUFFLENBQUMsQ0FBQyxJQUFJLElBQUksSUFBSSxDQUFDLE1BQU0sQ0FBQztZQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUMzRCxFQUFFLENBQUMsQ0FBQyxXQUFXLENBQUM7WUFBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUMxRCxDQUFDO0lBRUQsa0JBQWtCLENBQ2hCLE1BQXFCLEVBQ3JCLElBQW1CLEVBQ25CLFdBQWtCLEVBQ2xCLFNBQW9CLEVBQ3BCLE9BQW9DLEVBQ3BDLE1BQTJCO1FBRTNCLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLEdBQUcsRUFBRSxTQUFTLEVBQUUsT0FBTyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBRXRFLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQztZQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ25DLEVBQUUsQ0FBQyxDQUFDLElBQUksSUFBSSxJQUFJLENBQUMsTUFBTSxDQUFDO1lBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQzNELEVBQUUsQ0FBQyxDQUFDLFdBQVcsQ0FBQztZQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQzFELENBQUM7SUFFRCxhQUFhLENBQUMsS0FBYSxFQUFFLEdBQVc7UUFDdEMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLEdBQUcsQ0FBQyxDQUFDO0lBQzlCLENBQUM7SUFFRCxjQUFjO1FBQ1osSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxDQUFDLEtBQUssRUFBRSxDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVELGVBQWU7UUFDYixJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxhQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxDQUFDLGNBQWMsRUFBRSxFQUFFLDBEQUEwRCxDQUFDLENBQUMsQ0FBQztJQUMxSCxDQUFDO0lBRUQsZ0JBQWdCO1FBQ2QsSUFBSSxLQUFLLEdBQUcsSUFBSSxDQUFDLFlBQVksRUFBRSxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQ3hDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDbkMsTUFBTSxDQUFDLEtBQUssQ0FBQztJQUNmLENBQUM7SUFFRCxhQUFhLENBQUMsSUFBd0IsRUFBRSxJQUFZO1FBQ2xELElBQUksS0FBSyxHQUFHLG1CQUFLLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNuQyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUM1QixNQUFNLENBQUMsS0FBSyxDQUFDO0lBQ2YsQ0FBQztJQUVELFFBQVE7UUFDTixJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsRUFBRSxDQUFDO0lBQ3hCLENBQUM7SUFFRCxlQUFlO1FBQ2IsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEdBQUcsRUFBRSxDQUFDO0lBQy9CLENBQUM7SUFFRCxjQUFjLENBQUMsQ0FBYztRQUMzQixJQUFJLENBQUMsS0FBSyxFQUFFLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFFRCxpQkFBaUI7SUFFakIsT0FBTztRQUNMLE1BQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLENBQUMsT0FBTyxFQUFFLENBQUM7SUFDaEMsQ0FBQztJQUVELGtCQUFrQixDQUFDLE1BQWM7UUFDL0IsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVELE9BQU87UUFDTCxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsQ0FBQztJQUM5QixDQUFDO0lBRUQsYUFBYTtJQUViLE1BQU0sQ0FBQyxLQUFhLEVBQUUsR0FBVyxFQUFFLEtBQW9CO1FBQ3JELE1BQU0sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxHQUFHLEVBQUUsRUFBRSxJQUFJLEVBQUUsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7SUFDakUsQ0FBQztJQUVELE9BQU8sQ0FBQyxLQUFhLEVBQUUsR0FBVyxFQUFFLFVBQTZCO1FBQy9ELElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLEdBQUcsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUNyQyxJQUFJLE1BQW9DLENBQUM7UUFFekMsT0FBTyxJQUFJLEVBQUUsQ0FBQztZQUNaLE1BQU0sR0FBRyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7WUFDckIsRUFBRSxDQUFDLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQztnQkFBQyxLQUFLLENBQUM7UUFDekIsQ0FBQztRQUVELE1BQU0sQ0FBQyxNQUFNLENBQUMsS0FBcUIsQ0FBQztJQUN0QyxDQUFDO0lBRU8sT0FBTyxDQUFDLEtBQWEsRUFBRSxHQUFXLEVBQUUsVUFBNkI7UUFDdkUsSUFBSSxFQUFFLFlBQVksRUFBRSxLQUFLLEVBQUUsbUJBQW1CLEVBQUUsR0FBRyxJQUFJLENBQUM7UUFFeEQsWUFBWSxDQUFDLGVBQWUsRUFBRSxDQUFDO1FBRS9CLG1CQUFtQixDQUFDLElBQUksQ0FBQyxJQUFJLGlCQUFVLEVBQWtCLENBQUMsQ0FBQztRQUUzRCxLQUFLLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxHQUFHLENBQUMsQ0FBQztRQUV2QixFQUFFLENBQUMsQ0FBQyxVQUFVLENBQUM7WUFBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVELElBQUk7UUFDRixJQUFJLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxtQkFBbUIsRUFBRSxZQUFZLEVBQUUsR0FBRyxJQUFJLENBQUM7UUFDN0QsSUFBSSxNQUFzQixDQUFDO1FBRTNCLEVBQUUsQ0FBQyxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUN0Qyx3QkFBYyxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsTUFBTSxDQUFDLENBQUM7WUFDdEMsTUFBTSxDQUFDLEVBQUUsSUFBSSxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUUsSUFBSSxFQUFFLENBQUM7UUFDdEMsQ0FBQztRQUVELE1BQU0sQ0FBQztZQUNMLElBQUksRUFBRSxJQUFJO1lBQ1YsS0FBSyxFQUFFLElBQUksdUJBQVksQ0FDckIsR0FBRyxFQUNILGFBQU0sQ0FBQyxtQkFBbUIsQ0FBQyxHQUFHLEVBQUUsRUFBRSwrQ0FBK0MsQ0FBQyxFQUNsRixZQUFZLENBQUMsUUFBUSxFQUFFLENBQ3hCO1NBQ0YsQ0FBQztJQUNKLENBQUM7SUFFRCxjQUFjLENBQUMsTUFBYztRQUMzQix3QkFBYyxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsTUFBTSxDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVELDBFQUEwRTtJQUMxRSxtQ0FBbUM7SUFDbkMsV0FBVyxDQUFDLEtBQWtCLEVBQUUsSUFBMkI7UUFDekQsSUFBSSxRQUFRLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDdkMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVELGFBQWEsQ0FBQyxLQUFtQjtRQUMvQixJQUFJLFFBQVEsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUN2QyxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQzNCLENBQUM7SUFFRCxZQUFZLENBQ1YsSUFBbUIsRUFDbkIsTUFBcUIsRUFDckIsV0FBa0IsRUFDbEIsU0FBb0IsRUFDcEIsT0FBb0MsRUFDcEMsTUFBMkI7UUFFM0IsSUFBSSxDQUFDLGtCQUFrQixDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsTUFBTSxDQUFDLENBQUM7SUFDakYsQ0FBQztJQUVELGVBQWUsQ0FBQyxJQUE2QjtRQUMzQyxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVELFlBQVksQ0FBQyxJQUFrQjtRQUM3QixJQUFJLFVBQVUsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7UUFDekQsSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUNyRCxDQUFDO0lBRUQsa0JBQWtCLENBQUMsT0FBaUI7UUFDbEMsSUFBSSxJQUFJLEdBQUcsYUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFLEVBQUUsK0NBQStDLENBQUMsQ0FBQztRQUV6RixJQUFJLEVBQUUsVUFBVSxFQUFFLEdBQUcsSUFBSSxDQUFDO1FBRTFCLElBQUksS0FBSyxHQUFHLElBQUksQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUV6QixHQUFHLENBQUEsQ0FBQyxJQUFJLENBQUMsR0FBQyxDQUFDLEVBQUUsQ0FBQyxHQUFHLE9BQU8sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQztZQUNyQyxLQUFLLENBQUMsVUFBVSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxVQUFVLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDakQsQ0FBQztJQUNILENBQUM7SUFFRCxhQUFhLENBQUMsS0FBdUIsRUFBRSxPQUFpQjtRQUN0RCxJQUFJLElBQUksR0FBRyxhQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsRUFBRSwwQ0FBMEMsQ0FBQyxDQUFDO1FBQ3BGLElBQUksS0FBSyxHQUFHLElBQUksQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUV6QixJQUFJLEVBQUUsS0FBSyxFQUFFLEdBQUcsSUFBSSxDQUFDO1FBRXJCLEdBQUcsQ0FBQSxDQUFDLElBQUksQ0FBQyxHQUFDLENBQUMsRUFBRSxDQUFDLEdBQUcsS0FBSyxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDO1lBQ25DLElBQUksSUFBSSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQzlDLEtBQUssQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLEtBQUssQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztRQUNoRCxDQUFDO0lBQ0gsQ0FBQztJQUVELFVBQVUsQ0FBQyxLQUF1QixFQUFFLE9BQWlCO1FBQ25ELElBQUksTUFBTSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLENBQUM7UUFDcEMsSUFBSSxLQUFLLEdBQUcsSUFBSSxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRXpCLEdBQUcsQ0FBQSxDQUFDLElBQUksQ0FBQyxHQUFDLENBQUMsRUFBRSxDQUFDLEdBQUcsS0FBSyxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDO1lBQ25DLElBQUksSUFBSSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQzlDLEtBQUssQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsTUFBTSxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxJQUFJLElBQUksQ0FBQyxDQUFDO1FBQ2hFLENBQUM7SUFDSCxDQUFDO0lBRUQsZUFBZSxDQUFDLE1BQWM7UUFDNUIsSUFBSSxJQUFJLEdBQUcsYUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFLEVBQUUsNENBQTRDLENBQUMsQ0FBQztRQUN0RixJQUFJLEtBQUssR0FBRyxJQUFJLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFekIsYUFBTSxDQUFDLElBQUksRUFBRSx3QkFBd0IsQ0FBQyxDQUFDO1FBRXZDLEtBQUssQ0FBQyxlQUFlLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRCxlQUFlO1FBQ2IsSUFBSSxXQUFXLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLEVBQUUsQ0FBQztRQUM5QyxJQUFJLEtBQUssR0FBRyxJQUFJLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFekIsYUFBTSxDQUFDLFdBQVcsRUFBRSwwQkFBMEIsQ0FBQyxDQUFDO1FBRWhELEtBQUssQ0FBQyxlQUFlLENBQUMsV0FBVyxDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVELGdCQUFnQixDQUFDLEtBQXVCO1FBQ3RDLElBQUksSUFBSSxHQUFHLGFBQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxFQUFFLDZDQUE2QyxDQUFDLENBQUM7UUFDdkYsSUFBSSxLQUFLLEdBQUcsSUFBSSxDQUFDLFlBQVksRUFBRSxDQUFDO1FBRWhDLGFBQU0sQ0FBQyxJQUFJLEVBQUUsMkJBQTJCLENBQUMsQ0FBQztRQUUxQyxHQUFHLENBQUEsQ0FBQyxJQUFJLENBQUMsR0FBQyxDQUFDLEVBQUUsQ0FBQyxHQUFHLEtBQUssQ0FBQyxNQUFNLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQztZQUNuQyxJQUFJLElBQUksR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUM5QyxLQUFLLENBQUMsR0FBRyxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1FBQ3hDLENBQUM7SUFDSCxDQUFDO0NBQ0Y7QUE5WEQscUJBOFhDIn0=