import { UNDEFINED_REFERENCE } from '../../references';
import { EMPTY_ARRAY, EMPTY_DICT } from '../../utils';
import { combineTagged } from '@glimmer/reference';
import { assert, dict } from '@glimmer/util';
export class CompiledNamedArgs {
    constructor(keys, values) {
        this.keys = keys;
        this.values = values;
        this.length = keys.length;
        assert(keys.length === values.length, 'Keys and values do not have the same length');
    }
    static empty() {
        return COMPILED_EMPTY_NAMED_ARGS;
    }
    static create(map) {
        let keys = Object.keys(map);
        let length = keys.length;
        if (length > 0) {
            let values = [];
            for (let i = 0; i < length; i++) {
                values[i] = map[keys[i]];
            }
            return new this(keys, values);
        }
        else {
            return COMPILED_EMPTY_NAMED_ARGS;
        }
    }
    evaluate(vm) {
        let { keys, values, length } = this;
        let evaluated = new Array(length);
        for (let i = 0; i < length; i++) {
            evaluated[i] = values[i].evaluate(vm);
        }
        return new EvaluatedNamedArgs(keys, evaluated);
    }
    toJSON() {
        let { keys, values } = this;
        let inner = keys.map((key, i) => `${key}: ${values[i].toJSON()}`).join(", ");
        return `{${inner}}`;
    }
}
export const COMPILED_EMPTY_NAMED_ARGS = new (class extends CompiledNamedArgs {
    constructor() {
        super(EMPTY_ARRAY, EMPTY_ARRAY);
    }
    evaluate(_vm) {
        return EVALUATED_EMPTY_NAMED_ARGS;
    }
    toJSON() {
        return `<EMPTY>`;
    }
});
export class EvaluatedNamedArgs {
    constructor(keys, values, _map = null) {
        this.keys = keys;
        this.values = values;
        this._map = _map;
        this.tag = combineTagged(values);
        this.length = keys.length;
        assert(keys.length === values.length, 'Keys and values do not have the same length');
    }
    static create(map) {
        let keys = Object.keys(map);
        let length = keys.length;
        if (length > 0) {
            let values = new Array(length);
            for (let i = 0; i < length; i++) {
                values[i] = map[keys[i]];
            }
            return new this(keys, values, map);
        }
        else {
            return EVALUATED_EMPTY_NAMED_ARGS;
        }
    }
    static empty() {
        return EVALUATED_EMPTY_NAMED_ARGS;
    }
    get map() {
        let { _map: map } = this;
        if (map) {
            return map;
        }
        map = this._map = dict();
        let { keys, values, length } = this;
        for (let i = 0; i < length; i++) {
            map[keys[i]] = values[i];
        }
        return map;
    }
    get(key) {
        let { keys, values } = this;
        let index = keys.indexOf(key);
        return (index === -1) ? UNDEFINED_REFERENCE : values[index];
    }
    has(key) {
        return this.keys.indexOf(key) !== -1;
    }
    value() {
        let { keys, values } = this;
        let out = dict();
        for (let i = 0; i < keys.length; i++) {
            let key = keys[i];
            let ref = values[i];
            out[key] = ref.value();
        }
        return out;
    }
}
export const EVALUATED_EMPTY_NAMED_ARGS = new (class extends EvaluatedNamedArgs {
    constructor() {
        super(EMPTY_ARRAY, EMPTY_ARRAY, EMPTY_DICT);
    }
    get() {
        return UNDEFINED_REFERENCE;
    }
    has(_key) {
        return false;
    }
    value() {
        return EMPTY_DICT;
    }
});
//# sourceMappingURL=data:application/json;base64,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