import * as content from './content';
import * as vm from './vm';
import { CompiledGetBlockBySymbol, CompiledInPartialGetBlock } from '../../compiled/expressions/has-block';
import { Stack, dict, expect } from '@glimmer/util';
import { expr } from '../../syntax/functions';
import { CompiledExpression } from '../expressions';
import { ComponentBuilder } from '../../compiler';
class Labels {
    constructor() {
        this.labels = dict();
        this.jumps = [];
        this.ranges = [];
    }
    label(name, index) {
        this.labels[name] = index;
    }
    jump(at, Target, target) {
        this.jumps.push({ at, target, Target });
    }
    range(at, Range, start, end) {
        this.ranges.push({ at, start, end, Range });
    }
    patch(opcodes) {
        for (let i = 0; i < this.jumps.length; i++) {
            let { at, target, Target } = this.jumps[i];
            opcodes.set(at, Target, this.labels[target]);
        }
        for (let i = 0; i < this.ranges.length; i++) {
            let { at, start, end, Range } = this.ranges[i];
            opcodes.set(at, Range, this.labels[start], this.labels[end] - 1);
        }
    }
}
export class BasicOpcodeBuilder {
    constructor(symbolTable, env, program) {
        this.symbolTable = symbolTable;
        this.env = env;
        this.program = program;
        this.labelsStack = new Stack();
        this.constants = env.constants;
        this.start = program.next;
    }
    get end() {
        return this.program.next;
    }
    get pos() {
        return this.program.current;
    }
    get nextPos() {
        return this.program.next;
    }
    opcode(name, op1, op2, op3) {
        this.push(name, op1, op2, op3);
    }
    push(type, op1 = 0, op2 = 0, op3 = 0) {
        this.program.push(type, op1, op2, op3);
    }
    // helpers
    get labels() {
        return expect(this.labelsStack.current, 'bug: not in a label stack');
    }
    startLabels() {
        this.labelsStack.push(new Labels());
    }
    stopLabels() {
        let label = expect(this.labelsStack.pop(), 'unbalanced push and pop labels');
        label.patch(this.program);
    }
    // partials
    putPartialDefinition(_definition) {
        let definition = this.constants.other(_definition);
        this.opcode(50 /* PutPartial */, definition);
    }
    putDynamicPartialDefinition() {
        this.opcode(49 /* PutDynamicPartial */, this.constants.other(this.symbolTable));
    }
    evaluatePartial() {
        this.opcode(51 /* EvaluatePartial */, this.constants.other(this.symbolTable), this.constants.other(dict()));
    }
    // components
    putComponentDefinition(definition) {
        this.opcode(23 /* PutComponent */, this.other(definition));
    }
    putDynamicComponentDefinition() {
        this.opcode(22 /* PutDynamicComponent */);
    }
    openComponent(args, shadow) {
        this.opcode(24 /* OpenComponent */, this.args(args), shadow ? this.block(shadow) : 0);
    }
    didCreateElement() {
        this.opcode(25 /* DidCreateElement */);
    }
    shadowAttributes() {
        this.opcode(26 /* ShadowAttributes */);
        this.opcode(21 /* CloseBlock */);
    }
    didRenderLayout() {
        this.opcode(27 /* DidRenderLayout */);
    }
    closeComponent() {
        this.opcode(28 /* CloseComponent */);
    }
    // content
    dynamicContent(Opcode) {
        this.opcode(31 /* DynamicContent */, this.other(Opcode));
    }
    cautiousAppend() {
        this.dynamicContent(new content.OptimizedCautiousAppendOpcode());
    }
    trustingAppend() {
        this.dynamicContent(new content.OptimizedTrustingAppendOpcode());
    }
    guardedCautiousAppend(expression) {
        this.dynamicContent(new content.GuardedCautiousAppendOpcode(this.compileExpression(expression), this.symbolTable));
    }
    guardedTrustingAppend(expression) {
        this.dynamicContent(new content.GuardedTrustingAppendOpcode(this.compileExpression(expression), this.symbolTable));
    }
    // dom
    text(text) {
        this.opcode(29 /* Text */, this.constants.string(text));
    }
    openPrimitiveElement(tag) {
        this.opcode(32 /* OpenElement */, this.constants.string(tag));
    }
    openComponentElement(tag) {
        this.opcode(35 /* OpenComponentElement */, this.constants.string(tag));
    }
    openDynamicPrimitiveElement() {
        this.opcode(36 /* OpenDynamicElement */);
    }
    flushElement() {
        this.opcode(37 /* FlushElement */);
    }
    closeElement() {
        this.opcode(38 /* CloseElement */);
    }
    staticAttr(_name, _namespace, _value) {
        let name = this.constants.string(_name);
        let namespace = _namespace ? this.constants.string(_namespace) : 0;
        let value = this.constants.string(_value);
        this.opcode(40 /* StaticAttr */, name, value, namespace);
    }
    dynamicAttrNS(_name, _namespace, trusting) {
        let name = this.constants.string(_name);
        let namespace = this.constants.string(_namespace);
        this.opcode(42 /* DynamicAttrNS */, name, namespace, trusting | 0);
    }
    dynamicAttr(_name, trusting) {
        let name = this.constants.string(_name);
        this.opcode(43 /* DynamicAttr */, name, trusting | 0);
    }
    comment(_comment) {
        let comment = this.constants.string(_comment);
        this.opcode(30 /* Comment */, comment);
    }
    modifier(_name, _args) {
        let args = this.constants.expression(this.compile(_args));
        let _modifierManager = this.env.lookupModifier(_name, this.symbolTable);
        let modifierManager = this.constants.other(_modifierManager);
        let name = this.constants.string(_name);
        this.opcode(41 /* Modifier */, name, modifierManager, args);
    }
    // lists
    putIterator() {
        this.opcode(44 /* PutIterator */);
    }
    enterList(start, end) {
        this.push(45 /* EnterList */);
        this.labels.range(this.pos, 45 /* EnterList */, start, end);
    }
    exitList() {
        this.opcode(46 /* ExitList */);
    }
    enterWithKey(start, end) {
        this.push(47 /* EnterWithKey */);
        this.labels.range(this.pos, 47 /* EnterWithKey */, start, end);
    }
    nextIter(end) {
        this.push(48 /* NextIter */);
        this.labels.jump(this.pos, 48 /* NextIter */, end);
    }
    // vm
    openBlock(_args, _inner) {
        let args = this.constants.expression(this.compile(_args));
        let inner = this.constants.other(_inner);
        this.opcode(20 /* OpenBlock */, inner, args);
    }
    closeBlock() {
        this.opcode(21 /* CloseBlock */);
    }
    pushRemoteElement() {
        this.opcode(33 /* PushRemoteElement */);
    }
    popRemoteElement() {
        this.opcode(34 /* PopRemoteElement */);
    }
    popElement() {
        this.opcode(39 /* PopElement */);
    }
    label(name) {
        this.labels.label(name, this.nextPos);
    }
    pushChildScope() {
        this.opcode(0 /* PushChildScope */);
    }
    popScope() {
        this.opcode(1 /* PopScope */);
    }
    pushDynamicScope() {
        this.opcode(2 /* PushDynamicScope */);
    }
    popDynamicScope() {
        this.opcode(3 /* PopDynamicScope */);
    }
    putNull() {
        this.opcode(4 /* Put */, this.constants.NULL_REFERENCE);
    }
    putValue(_expression) {
        let expr = this.constants.expression(this.compileExpression(_expression));
        this.opcode(5 /* EvaluatePut */, expr);
    }
    putArgs(_args) {
        let args = this.constants.expression(this.compile(_args));
        this.opcode(6 /* PutArgs */, args);
    }
    bindDynamicScope(_names) {
        this.opcode(12 /* BindDynamicScope */, this.names(_names));
    }
    bindPositionalArgs(_names, _symbols) {
        this.opcode(7 /* BindPositionalArgs */, this.names(_names), this.symbols(_symbols));
    }
    bindNamedArgs(_names, _symbols) {
        this.opcode(8 /* BindNamedArgs */, this.names(_names), this.symbols(_symbols));
    }
    bindBlocks(_names, _symbols) {
        this.opcode(9 /* BindBlocks */, this.names(_names), this.symbols(_symbols));
    }
    enter(enter, exit) {
        this.push(13 /* Enter */);
        this.labels.range(this.pos, 13 /* Enter */, enter, exit);
    }
    exit() {
        this.opcode(14 /* Exit */);
    }
    evaluate(_block) {
        let block = this.constants.block(_block);
        this.opcode(15 /* Evaluate */, block);
    }
    test(testFunc) {
        let _func;
        if (testFunc === 'const') {
            _func = vm.ConstTest;
        }
        else if (testFunc === 'simple') {
            _func = vm.SimpleTest;
        }
        else if (testFunc === 'environment') {
            _func = vm.EnvironmentTest;
        }
        else if (typeof testFunc === 'function') {
            _func = testFunc;
        }
        else {
            throw new Error('unreachable');
        }
        let func = this.constants.function(_func);
        this.opcode(19 /* Test */, func);
    }
    jump(target) {
        this.push(16 /* Jump */);
        this.labels.jump(this.pos, 16 /* Jump */, target);
    }
    jumpIf(target) {
        this.push(17 /* JumpIf */);
        this.labels.jump(this.pos, 17 /* JumpIf */, target);
    }
    jumpUnless(target) {
        this.push(18 /* JumpUnless */);
        this.labels.jump(this.pos, 18 /* JumpUnless */, target);
    }
    names(_names) {
        let names = _names.map(n => this.constants.string(n));
        return this.constants.array(names);
    }
    symbols(symbols) {
        return this.constants.array(symbols);
    }
    other(value) {
        return this.constants.other(value);
    }
    args(args) {
        return this.constants.expression(this.compile(args));
    }
    block(block) {
        return this.constants.block(block);
    }
}
function isCompilableExpression(expr) {
    return expr && typeof expr['compile'] === 'function';
}
export default class OpcodeBuilder extends BasicOpcodeBuilder {
    constructor(symbolTable, env, program = env.program) {
        super(symbolTable, env, program);
        this.component = new ComponentBuilder(this);
    }
    compile(expr) {
        if (isCompilableExpression(expr)) {
            return expr.compile(this);
        }
        else {
            return expr;
        }
    }
    compileExpression(expression) {
        if (expression instanceof CompiledExpression) {
            return expression;
        }
        else {
            return expr(expression, this);
        }
    }
    bindPositionalArgsForLocals(locals) {
        let names = Object.keys(locals);
        let symbols = new Array(names.length); //Object.keys(locals).map(name => locals[name]);
        for (let i = 0; i < names.length; i++) {
            symbols[i] = locals[names[i]];
        }
        this.opcode(7 /* BindPositionalArgs */, this.symbols(symbols));
    }
    preludeForLayout(layout) {
        let symbols = layout.symbolTable.getSymbols();
        if (symbols.named) {
            let named = symbols.named;
            let namedNames = Object.keys(named);
            let namedSymbols = namedNames.map(n => named[n]);
            this.opcode(8 /* BindNamedArgs */, this.names(namedNames), this.symbols(namedSymbols));
        }
        this.opcode(11 /* BindCallerScope */);
        if (symbols.yields) {
            let yields = symbols.yields;
            let yieldNames = Object.keys(yields);
            let yieldSymbols = yieldNames.map(n => yields[n]);
            this.opcode(9 /* BindBlocks */, this.names(yieldNames), this.symbols(yieldSymbols));
        }
        if (symbols.partialArgs) {
            this.opcode(10 /* BindPartialArgs */, symbols.partialArgs);
        }
    }
    yield(args, to) {
        let yields, partial;
        let inner;
        if (yields = this.symbolTable.getSymbol('yields', to)) {
            inner = new CompiledGetBlockBySymbol(yields, to);
        }
        else if (partial = this.symbolTable.getPartialArgs()) {
            inner = new CompiledInPartialGetBlock(partial, to);
        }
        else {
            throw new Error('[BUG] ${to} is not a valid block name.');
        }
        this.openBlock(args, inner);
        this.closeBlock();
    }
    // TODO
    // come back to this
    labelled(args, callback) {
        if (args)
            this.putArgs(args);
        this.startLabels();
        this.enter('BEGIN', 'END');
        this.label('BEGIN');
        callback(this, 'BEGIN', 'END');
        this.label('END');
        this.exit();
        this.stopLabels();
    }
    // TODO
    // come back to this
    iter(callback) {
        this.startLabels();
        this.enterList('BEGIN', 'END');
        this.label('ITER');
        this.nextIter('BREAK');
        this.enterWithKey('BEGIN', 'END');
        this.label('BEGIN');
        callback(this, 'BEGIN', 'END');
        this.label('END');
        this.exit();
        this.jump('ITER');
        this.label('BREAK');
        this.exitList();
        this.stopLabels();
    }
    // TODO
    // come back to this
    unit(callback) {
        this.startLabels();
        callback(this);
        this.stopLabels();
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVpbGRlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uL2xpYi9jb21waWxlZC9vcGNvZGVzL2J1aWxkZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUEsT0FBTyxLQUFLLE9BQU8sTUFBTSxXQUFXLENBQUM7QUFDckMsT0FBTyxLQUFLLEVBQUUsTUFBTSxNQUFNLENBQUM7QUFJM0IsT0FBTyxFQUVMLHdCQUF3QixFQUN4Qix5QkFBeUIsRUFDMUIsTUFBTSxzQ0FBc0MsQ0FBQztBQUU5QyxPQUFPLEVBQVUsS0FBSyxFQUFnQixJQUFJLEVBQUUsTUFBTSxFQUFFLE1BQU0sZUFBZSxDQUFDO0FBQzFFLE9BQU8sRUFBRSxJQUFJLEVBQUUsTUFBTSx3QkFBd0IsQ0FBQztBQUc5QyxPQUFPLEVBQUUsa0JBQWtCLEVBQUUsTUFBTSxnQkFBZ0IsQ0FBQztBQU1wRCxPQUFPLEVBQUUsZ0JBQWdCLEVBQUUsTUFBTSxnQkFBZ0IsQ0FBQztBQTRCbEQ7SUFBQTtRQUNFLFdBQU0sR0FBRyxJQUFJLEVBQVUsQ0FBQztRQUN4QixVQUFLLEdBQTJELEVBQUUsQ0FBQztRQUNuRSxXQUFNLEdBQXFFLEVBQUUsQ0FBQztJQXlCaEYsQ0FBQztJQXZCQyxLQUFLLENBQUMsSUFBWSxFQUFFLEtBQWE7UUFDL0IsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsR0FBRyxLQUFLLENBQUM7SUFDNUIsQ0FBQztJQUVELElBQUksQ0FBQyxFQUFVLEVBQUUsTUFBb0IsRUFBRSxNQUFjO1FBQ25ELElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEVBQUUsRUFBRSxFQUFFLE1BQU0sRUFBRSxNQUFNLEVBQUUsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRCxLQUFLLENBQUMsRUFBVSxFQUFFLEtBQWtCLEVBQUUsS0FBYSxFQUFFLEdBQVc7UUFDOUQsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRSxFQUFFLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFRCxLQUFLLENBQUMsT0FBZ0I7UUFDcEIsR0FBRyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDO1lBQzNDLElBQUksRUFBRSxFQUFFLEVBQUUsTUFBTSxFQUFFLE1BQU0sRUFBRSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDM0MsT0FBTyxDQUFDLEdBQUcsQ0FBQyxFQUFFLEVBQUUsTUFBTSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztRQUMvQyxDQUFDO1FBRUQsR0FBRyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDO1lBQzVDLElBQUksRUFBRSxFQUFFLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxLQUFLLEVBQUUsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQy9DLE9BQU8sQ0FBQyxHQUFHLENBQUMsRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDbkUsQ0FBQztJQUNILENBQUM7Q0FDRjtBQUVELE1BQU07SUFLSixZQUFtQixXQUF3QixFQUFTLEdBQWdCLEVBQVMsT0FBZ0I7UUFBMUUsZ0JBQVcsR0FBWCxXQUFXLENBQWE7UUFBUyxRQUFHLEdBQUgsR0FBRyxDQUFhO1FBQVMsWUFBTyxHQUFQLE9BQU8sQ0FBUztRQUpyRixnQkFBVyxHQUFHLElBQUksS0FBSyxFQUFVLENBQUM7UUFLeEMsSUFBSSxDQUFDLFNBQVMsR0FBRyxHQUFHLENBQUMsU0FBUyxDQUFDO1FBQy9CLElBQUksQ0FBQyxLQUFLLEdBQUcsT0FBTyxDQUFDLElBQUksQ0FBQztJQUM1QixDQUFDO0lBS0QsSUFBVyxHQUFHO1FBQ1osTUFBTSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDO0lBQzNCLENBQUM7SUFFRCxJQUFZLEdBQUc7UUFDYixNQUFNLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUM7SUFDOUIsQ0FBQztJQUVELElBQVksT0FBTztRQUNqQixNQUFNLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUM7SUFDM0IsQ0FBQztJQUVTLE1BQU0sQ0FBQyxJQUFRLEVBQUUsR0FBWSxFQUFFLEdBQVksRUFBRSxHQUFZO1FBQ2pFLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVELElBQUksQ0FBQyxJQUFZLEVBQUUsR0FBRyxHQUFHLENBQUMsRUFBRSxHQUFHLEdBQUcsQ0FBQyxFQUFFLEdBQUcsR0FBRyxDQUFDO1FBQzFDLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRCxVQUFVO0lBRVYsSUFBWSxNQUFNO1FBQ2hCLE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxPQUFPLEVBQUUsMkJBQTJCLENBQUMsQ0FBQztJQUN2RSxDQUFDO0lBRUQsV0FBVztRQUNULElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLElBQUksTUFBTSxFQUFFLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQsVUFBVTtRQUNSLElBQUksS0FBSyxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLEdBQUcsRUFBRSxFQUFFLGdDQUFnQyxDQUFDLENBQUM7UUFDN0UsS0FBSyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDNUIsQ0FBQztJQUVELFdBQVc7SUFFWCxvQkFBb0IsQ0FBQyxXQUFzQztRQUN6RCxJQUFJLFVBQVUsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUNuRCxJQUFJLENBQUMsTUFBTSxzQkFBZ0IsVUFBVSxDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVELDJCQUEyQjtRQUN6QixJQUFJLENBQUMsTUFBTSw2QkFBdUIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUM7SUFDNUUsQ0FBQztJQUVELGVBQWU7UUFDYixJQUFJLENBQUMsTUFBTSwyQkFBcUIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQztJQUN4RyxDQUFDO0lBRUQsYUFBYTtJQUViLHNCQUFzQixDQUFDLFVBQXVDO1FBQzVELElBQUksQ0FBQyxNQUFNLHdCQUFrQixJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUM7SUFDdkQsQ0FBQztJQUVELDZCQUE2QjtRQUMzQixJQUFJLENBQUMsTUFBTSw4QkFBd0IsQ0FBQztJQUN0QyxDQUFDO0lBRUQsYUFBYSxDQUFDLElBQThCLEVBQUUsTUFBb0I7UUFDaEUsSUFBSSxDQUFDLE1BQU0seUJBQW1CLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsTUFBTSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7SUFDbEYsQ0FBQztJQUVELGdCQUFnQjtRQUNkLElBQUksQ0FBQyxNQUFNLDJCQUFxQixDQUFDO0lBQ25DLENBQUM7SUFFRCxnQkFBZ0I7UUFDZCxJQUFJLENBQUMsTUFBTSwyQkFBcUIsQ0FBQztRQUNqQyxJQUFJLENBQUMsTUFBTSxxQkFBZSxDQUFDO0lBQzdCLENBQUM7SUFFRCxlQUFlO1FBQ2IsSUFBSSxDQUFDLE1BQU0sMEJBQW9CLENBQUM7SUFDbEMsQ0FBQztJQUVELGNBQWM7UUFDWixJQUFJLENBQUMsTUFBTSx5QkFBbUIsQ0FBQztJQUNqQyxDQUFDO0lBRUQsVUFBVTtJQUVWLGNBQWMsQ0FBQyxNQUE4QztRQUMzRCxJQUFJLENBQUMsTUFBTSwwQkFBb0IsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO0lBQ3JELENBQUM7SUFFRCxjQUFjO1FBQ1osSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLE9BQU8sQ0FBQyw2QkFBNkIsRUFBRSxDQUFDLENBQUM7SUFDbkUsQ0FBQztJQUVELGNBQWM7UUFDWixJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksT0FBTyxDQUFDLDZCQUE2QixFQUFFLENBQUMsQ0FBQztJQUNuRSxDQUFDO0lBRUQscUJBQXFCLENBQUMsVUFBZ0M7UUFDcEQsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLE9BQU8sQ0FBQywyQkFBMkIsQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsVUFBVSxDQUFDLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUM7SUFDckgsQ0FBQztJQUVELHFCQUFxQixDQUFDLFVBQWdDO1FBQ3BELElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxPQUFPLENBQUMsMkJBQTJCLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLFVBQVUsQ0FBQyxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDO0lBQ3JILENBQUM7SUFFRCxNQUFNO0lBRU4sSUFBSSxDQUFDLElBQVk7UUFDZixJQUFJLENBQUMsTUFBTSxnQkFBVSxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO0lBQ3BELENBQUM7SUFFRCxvQkFBb0IsQ0FBQyxHQUFXO1FBQzlCLElBQUksQ0FBQyxNQUFNLHVCQUFpQixJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO0lBQzFELENBQUM7SUFFRCxvQkFBb0IsQ0FBQyxHQUFXO1FBQzlCLElBQUksQ0FBQyxNQUFNLGdDQUEwQixJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO0lBQ25FLENBQUM7SUFFRCwyQkFBMkI7UUFDekIsSUFBSSxDQUFDLE1BQU0sNkJBQXVCLENBQUM7SUFDckMsQ0FBQztJQUVELFlBQVk7UUFDVixJQUFJLENBQUMsTUFBTSx1QkFBaUIsQ0FBQztJQUMvQixDQUFDO0lBRUQsWUFBWTtRQUNWLElBQUksQ0FBQyxNQUFNLHVCQUFpQixDQUFDO0lBQy9CLENBQUM7SUFFRCxVQUFVLENBQUMsS0FBYSxFQUFFLFVBQTBCLEVBQUUsTUFBYztRQUNsRSxJQUFJLElBQUksR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUN4QyxJQUFJLFNBQVMsR0FBRyxVQUFVLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ25FLElBQUksS0FBSyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBRTFDLElBQUksQ0FBQyxNQUFNLHNCQUFnQixJQUFJLEVBQUUsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO0lBQ3JELENBQUM7SUFFRCxhQUFhLENBQUMsS0FBYSxFQUFFLFVBQWtCLEVBQUUsUUFBaUI7UUFDaEUsSUFBSSxJQUFJLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDeEMsSUFBSSxTQUFTLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLENBQUM7UUFFbEQsSUFBSSxDQUFDLE1BQU0seUJBQW1CLElBQUksRUFBRSxTQUFTLEVBQUcsUUFBZ0IsR0FBQyxDQUFDLENBQUMsQ0FBQztJQUN0RSxDQUFDO0lBRUQsV0FBVyxDQUFDLEtBQWEsRUFBRSxRQUFpQjtRQUMxQyxJQUFJLElBQUksR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUN4QyxJQUFJLENBQUMsTUFBTSx1QkFBaUIsSUFBSSxFQUFHLFFBQWdCLEdBQUMsQ0FBQyxDQUFDLENBQUM7SUFDekQsQ0FBQztJQUVELE9BQU8sQ0FBQyxRQUFnQjtRQUN0QixJQUFJLE9BQU8sR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUM5QyxJQUFJLENBQUMsTUFBTSxtQkFBYSxPQUFPLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQsUUFBUSxDQUFDLEtBQWEsRUFBRSxLQUErQjtRQUNyRCxJQUFJLElBQUksR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFDMUQsSUFBSSxnQkFBZ0IsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLGNBQWMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBQ3hFLElBQUksZUFBZSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFDN0QsSUFBSSxJQUFJLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDeEMsSUFBSSxDQUFDLE1BQU0sb0JBQWMsSUFBSSxFQUFFLGVBQWUsRUFBRSxJQUFJLENBQUMsQ0FBQztJQUN4RCxDQUFDO0lBRUQsUUFBUTtJQUVSLFdBQVc7UUFDVCxJQUFJLENBQUMsTUFBTSxzQkFBZ0IsQ0FBQztJQUM5QixDQUFDO0lBRUQsU0FBUyxDQUFDLEtBQWEsRUFBRSxHQUFXO1FBQ2xDLElBQUksQ0FBQyxJQUFJLG9CQUFjLENBQUM7UUFDeEIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEdBQUcsc0JBQWdCLEtBQUssRUFBRSxHQUFHLENBQUMsQ0FBQztJQUN4RCxDQUFDO0lBRUQsUUFBUTtRQUNOLElBQUksQ0FBQyxNQUFNLG1CQUFhLENBQUM7SUFDM0IsQ0FBQztJQUVELFlBQVksQ0FBQyxLQUFhLEVBQUUsR0FBVztRQUNyQyxJQUFJLENBQUMsSUFBSSx1QkFBaUIsQ0FBQztRQUMzQixJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsR0FBRyx5QkFBbUIsS0FBSyxFQUFFLEdBQUcsQ0FBQyxDQUFDO0lBQzNELENBQUM7SUFFRCxRQUFRLENBQUMsR0FBVztRQUNsQixJQUFJLENBQUMsSUFBSSxtQkFBYSxDQUFDO1FBQ3ZCLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLHFCQUFlLEdBQUcsQ0FBQyxDQUFDO0lBQy9DLENBQUM7SUFFRCxLQUFLO0lBRUwsU0FBUyxDQUFDLEtBQStCLEVBQUUsTUFBd0I7UUFDakUsSUFBSSxJQUFJLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBQzFELElBQUksS0FBSyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3pDLElBQUksQ0FBQyxNQUFNLHFCQUFlLEtBQUssRUFBRSxJQUFJLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQsVUFBVTtRQUNSLElBQUksQ0FBQyxNQUFNLHFCQUFlLENBQUM7SUFDN0IsQ0FBQztJQUVELGlCQUFpQjtRQUNmLElBQUksQ0FBQyxNQUFNLDRCQUFzQixDQUFDO0lBQ3BDLENBQUM7SUFFRCxnQkFBZ0I7UUFDZCxJQUFJLENBQUMsTUFBTSwyQkFBcUIsQ0FBQztJQUNuQyxDQUFDO0lBRUQsVUFBVTtRQUNSLElBQUksQ0FBQyxNQUFNLHFCQUFlLENBQUM7SUFDN0IsQ0FBQztJQUVELEtBQUssQ0FBQyxJQUFZO1FBQ2hCLElBQUksQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVELGNBQWM7UUFDWixJQUFJLENBQUMsTUFBTSx3QkFBbUIsQ0FBQztJQUNqQyxDQUFDO0lBRUQsUUFBUTtRQUNOLElBQUksQ0FBQyxNQUFNLGtCQUFhLENBQUM7SUFDM0IsQ0FBQztJQUVELGdCQUFnQjtRQUNkLElBQUksQ0FBQyxNQUFNLDBCQUFxQixDQUFDO0lBQ25DLENBQUM7SUFFRCxlQUFlO1FBQ2IsSUFBSSxDQUFDLE1BQU0seUJBQW9CLENBQUM7SUFDbEMsQ0FBQztJQUVELE9BQU87UUFDTCxJQUFJLENBQUMsTUFBTSxjQUFTLElBQUksQ0FBQyxTQUFTLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDckQsQ0FBQztJQUVELFFBQVEsQ0FBQyxXQUFpQztRQUN4QyxJQUFJLElBQUksR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQztRQUMxRSxJQUFJLENBQUMsTUFBTSxzQkFBaUIsSUFBSSxDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVELE9BQU8sQ0FBQyxLQUErQjtRQUNyQyxJQUFJLElBQUksR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFDMUQsSUFBSSxDQUFDLE1BQU0sa0JBQWEsSUFBSSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVELGdCQUFnQixDQUFDLE1BQWdCO1FBQy9CLElBQUksQ0FBQyxNQUFNLDRCQUFzQixJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7SUFDdkQsQ0FBQztJQUVELGtCQUFrQixDQUFDLE1BQWdCLEVBQUUsUUFBa0I7UUFDckQsSUFBSSxDQUFDLE1BQU0sNkJBQXdCLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO0lBQ2pGLENBQUM7SUFFRCxhQUFhLENBQUMsTUFBZ0IsRUFBRSxRQUFrQjtRQUNoRCxJQUFJLENBQUMsTUFBTSx3QkFBbUIsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7SUFDNUUsQ0FBQztJQUVELFVBQVUsQ0FBQyxNQUFnQixFQUFFLFFBQWtCO1FBQzdDLElBQUksQ0FBQyxNQUFNLHFCQUFnQixJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztJQUN6RSxDQUFDO0lBRUQsS0FBSyxDQUFDLEtBQWEsRUFBRSxJQUFZO1FBQy9CLElBQUksQ0FBQyxJQUFJLGdCQUFVLENBQUM7UUFDcEIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEdBQUcsa0JBQVksS0FBSyxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQ3JELENBQUM7SUFFRCxJQUFJO1FBQ0YsSUFBSSxDQUFDLE1BQU0sZUFBUyxDQUFDO0lBQ3ZCLENBQUM7SUFFRCxRQUFRLENBQUMsTUFBbUI7UUFDMUIsSUFBSSxLQUFLLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDekMsSUFBSSxDQUFDLE1BQU0sb0JBQWMsS0FBSyxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVELElBQUksQ0FBQyxRQUE4RDtRQUNqRSxJQUFJLEtBQXNCLENBQUM7UUFFM0IsRUFBRSxDQUFDLENBQUMsUUFBUSxLQUFLLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDekIsS0FBSyxHQUFHLEVBQUUsQ0FBQyxTQUFTLENBQUM7UUFDdkIsQ0FBQztRQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxRQUFRLEtBQUssUUFBUSxDQUFDLENBQUMsQ0FBQztZQUNqQyxLQUFLLEdBQUcsRUFBRSxDQUFDLFVBQVUsQ0FBQztRQUN4QixDQUFDO1FBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLFFBQVEsS0FBSyxhQUFhLENBQUMsQ0FBQyxDQUFDO1lBQ3RDLEtBQUssR0FBRyxFQUFFLENBQUMsZUFBZSxDQUFDO1FBQzdCLENBQUM7UUFBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsT0FBTyxRQUFRLEtBQUssVUFBVSxDQUFDLENBQUMsQ0FBQztZQUMxQyxLQUFLLEdBQUcsUUFBUSxDQUFDO1FBQ25CLENBQUM7UUFBQyxJQUFJLENBQUMsQ0FBQztZQUNOLE1BQU0sSUFBSSxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDakMsQ0FBQztRQUVELElBQUksSUFBSSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzFDLElBQUksQ0FBQyxNQUFNLGdCQUFVLElBQUksQ0FBQyxDQUFDO0lBQzdCLENBQUM7SUFFRCxJQUFJLENBQUMsTUFBYztRQUNqQixJQUFJLENBQUMsSUFBSSxlQUFTLENBQUM7UUFDbkIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsaUJBQVcsTUFBTSxDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUVELE1BQU0sQ0FBQyxNQUFjO1FBQ25CLElBQUksQ0FBQyxJQUFJLGlCQUFXLENBQUM7UUFDckIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsbUJBQWEsTUFBTSxDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVELFVBQVUsQ0FBQyxNQUFjO1FBQ3ZCLElBQUksQ0FBQyxJQUFJLHFCQUFlLENBQUM7UUFDekIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsdUJBQWlCLE1BQU0sQ0FBQyxDQUFDO0lBQ3BELENBQUM7SUFFUyxLQUFLLENBQUMsTUFBZ0I7UUFDOUIsSUFBSSxLQUFLLEdBQUcsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQUksSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN0RCxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVTLE9BQU8sQ0FBQyxPQUFpQjtRQUNqQyxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVTLEtBQUssQ0FBQyxLQUFhO1FBQzNCLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRVMsSUFBSSxDQUFDLElBQThCO1FBQzNDLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7SUFDdkQsQ0FBQztJQUVTLEtBQUssQ0FBQyxLQUFrQjtRQUNoQyxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDckMsQ0FBQztDQUNGO0FBRUQsZ0NBQW1DLElBQW1CO0lBQ3BELE1BQU0sQ0FBQyxJQUFJLElBQUksT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssVUFBVSxDQUFDO0FBQ3ZELENBQUM7QUFFRCxNQUFNLENBQUMsT0FBTyxvQkFBcUIsU0FBUSxrQkFBa0I7SUFHM0QsWUFBWSxXQUF3QixFQUFFLEdBQWdCLEVBQUUsVUFBbUIsR0FBRyxDQUFDLE9BQU87UUFDcEYsS0FBSyxDQUFDLFdBQVcsRUFBRSxHQUFHLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDakMsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLGdCQUFnQixDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFRCxPQUFPLENBQUksSUFBbUI7UUFDNUIsRUFBRSxDQUFDLENBQUMsc0JBQXNCLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ2pDLE1BQU0sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzVCLENBQUM7UUFBQyxJQUFJLENBQUMsQ0FBQztZQUNOLE1BQU0sQ0FBQyxJQUFJLENBQUM7UUFDZCxDQUFDO0lBQ0gsQ0FBQztJQUVELGlCQUFpQixDQUFDLFVBQWdDO1FBQ2hELEVBQUUsQ0FBQyxDQUFDLFVBQVUsWUFBWSxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7WUFDN0MsTUFBTSxDQUFDLFVBQVUsQ0FBQztRQUNwQixDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNoQyxDQUFDO0lBQ0gsQ0FBQztJQUVELDJCQUEyQixDQUFDLE1BQW9CO1FBQzlDLElBQUksS0FBSyxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDaEMsSUFBSSxPQUFPLEdBQWEsSUFBSSxLQUFLLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsZ0RBQWdEO1FBQ2pHLEdBQUcsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsS0FBSyxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDO1lBQ3RDLE9BQU8sQ0FBQyxDQUFDLENBQUMsR0FBRyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDaEMsQ0FBQztRQUNELElBQUksQ0FBQyxNQUFNLDZCQUF3QixJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7SUFDNUQsQ0FBQztJQUVELGdCQUFnQixDQUFDLE1BQWdCO1FBQy9CLElBQUksT0FBTyxHQUFHLE1BQU0sQ0FBQyxXQUFXLENBQUMsVUFBVSxFQUFFLENBQUM7UUFFOUMsRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7WUFDbEIsSUFBSSxLQUFLLEdBQUcsT0FBTyxDQUFDLEtBQUssQ0FBQztZQUMxQixJQUFJLFVBQVUsR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQ3BDLElBQUksWUFBWSxHQUFHLFVBQVUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ2pELElBQUksQ0FBQyxNQUFNLHdCQUFtQixJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQztRQUNwRixDQUFDO1FBRUQsSUFBSSxDQUFDLE1BQU0sMEJBQW9CLENBQUM7UUFFaEMsRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7WUFDbkIsSUFBSSxNQUFNLEdBQUcsT0FBTyxDQUFDLE1BQU0sQ0FBQztZQUM1QixJQUFJLFVBQVUsR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQ3JDLElBQUksWUFBWSxHQUFHLFVBQVUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ2xELElBQUksQ0FBQyxNQUFNLHFCQUFnQixJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQztRQUNqRixDQUFDO1FBRUQsRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUM7WUFDeEIsSUFBSSxDQUFDLE1BQU0sMkJBQXFCLE9BQU8sQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUN2RCxDQUFDO0lBQ0gsQ0FBQztJQUVELEtBQUssQ0FBQyxJQUE4QixFQUFFLEVBQVU7UUFDOUMsSUFBSSxNQUFzQixFQUFFLE9BQXVCLENBQUM7UUFDcEQsSUFBSSxLQUF1QixDQUFDO1FBRTVCLEVBQUUsQ0FBQyxDQUFDLE1BQU0sR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxRQUFRLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3RELEtBQUssR0FBRyxJQUFJLHdCQUF3QixDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNuRCxDQUFDO1FBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLE9BQU8sR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLGNBQWMsRUFBRSxDQUFDLENBQUMsQ0FBQztZQUN2RCxLQUFLLEdBQUcsSUFBSSx5QkFBeUIsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDckQsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sTUFBTSxJQUFJLEtBQUssQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFDO1FBQzVELENBQUM7UUFFRCxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztRQUM1QixJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7SUFDcEIsQ0FBQztJQUVELE9BQU87SUFDUCxvQkFBb0I7SUFDcEIsUUFBUSxDQUFDLElBQXNDLEVBQUUsUUFBdUI7UUFDdEUsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDO1lBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUU3QixJQUFJLENBQUMsV0FBVyxFQUFFLENBQUM7UUFDbkIsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDM0IsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUVwQixRQUFRLENBQUMsSUFBSSxFQUFFLE9BQU8sRUFBRSxLQUFLLENBQUMsQ0FBQztRQUUvQixJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ2xCLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNaLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztJQUNwQixDQUFDO0lBRUQsT0FBTztJQUNQLG9CQUFvQjtJQUNwQixJQUFJLENBQUMsUUFBdUI7UUFDMUIsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDO1FBQ25CLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQy9CLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDbkIsSUFBSSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUN2QixJQUFJLENBQUMsWUFBWSxDQUFDLE9BQU8sRUFBRSxLQUFLLENBQUMsQ0FBQztRQUNsQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBRXBCLFFBQVEsQ0FBQyxJQUFJLEVBQUUsT0FBTyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRS9CLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDbEIsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ1osSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUNsQixJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ3BCLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQztRQUNoQixJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7SUFDcEIsQ0FBQztJQUVELE9BQU87SUFDUCxvQkFBb0I7SUFDcEIsSUFBSSxDQUFDLFFBQTBDO1FBQzdDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQztRQUNuQixRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDZixJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7SUFDcEIsQ0FBQztDQUNGIn0=