import { isSafeString, isNode, isString, cautiousInsert, trustingInsert } from '../../upsert';
import { isComponentDefinition } from '../../component/interfaces';
import { UpdatingOpcode } from '../../opcodes';
import { CompiledArgs } from '../expressions';
import { TryOpcode } from '../../vm/update';
import { ReferenceCache, UpdatableTag, isModified, isConst, map } from '@glimmer/reference';
import { LinkedList, expect } from '@glimmer/util';
import { Cursor, clear } from '../../bounds';
import { Fragment } from '../../builder';
import OpcodeBuilderDSL from './builder';
import { ConditionalReference } from '../../references';
import { UpdatableBlockTracker } from '../../builder';
import { APPEND_OPCODES } from '../../opcodes';
APPEND_OPCODES.add(31 /* DynamicContent */, (vm, { op1: append }) => {
    let opcode = vm.constants.getOther(append);
    opcode.evaluate(vm);
});
function isEmpty(value) {
    return value === null || value === undefined || typeof value['toString'] !== 'function';
}
export function normalizeTextValue(value) {
    if (isEmpty(value)) {
        return '';
    }
    return String(value);
}
function normalizeTrustedValue(value) {
    if (isEmpty(value)) {
        return '';
    }
    if (isString(value)) {
        return value;
    }
    if (isSafeString(value)) {
        return value.toHTML();
    }
    if (isNode(value)) {
        return value;
    }
    return String(value);
}
function normalizeValue(value) {
    if (isEmpty(value)) {
        return '';
    }
    if (isString(value)) {
        return value;
    }
    if (isSafeString(value) || isNode(value)) {
        return value;
    }
    return String(value);
}
export class AppendDynamicOpcode {
    evaluate(vm) {
        let reference = vm.frame.getOperand();
        let normalized = this.normalize(reference);
        let value, cache;
        if (isConst(reference)) {
            value = normalized.value();
        }
        else {
            cache = new ReferenceCache(normalized);
            value = cache.peek();
        }
        let stack = vm.stack();
        let upsert = this.insert(vm.env.getAppendOperations(), stack, value);
        let bounds = new Fragment(upsert.bounds);
        stack.newBounds(bounds);
        if (cache /* i.e. !isConst(reference) */) {
            vm.updateWith(this.updateWith(vm, reference, cache, bounds, upsert));
        }
    }
}
export class GuardedAppendOpcode extends AppendDynamicOpcode {
    constructor(expression, symbolTable) {
        super();
        this.expression = expression;
        this.symbolTable = symbolTable;
        this.start = -1;
        this.end = -1;
    }
    evaluate(vm) {
        if (this.start === -1) {
            vm.evaluateOperand(this.expression);
            let value = vm.frame.getOperand().value();
            if (isComponentDefinition(value)) {
                this.deopt(vm.env);
                vm.pushEvalFrame(this.start, this.end);
            }
            else {
                super.evaluate(vm);
            }
        }
        else {
            vm.pushEvalFrame(this.start, this.end);
        }
    }
    deopt(env) {
        // At compile time, we determined that this append callsite might refer
        // to a local variable/property lookup that resolves to a component
        // definition at runtime.
        //
        // We could have eagerly compiled this callsite into something like this:
        //
        //   {{#if (is-component-definition foo)}}
        //     {{component foo}}
        //   {{else}}
        //     {{foo}}
        //   {{/if}}
        //
        // However, in practice, there might be a large amout of these callsites
        // and most of them would resolve to a simple value lookup. Therefore, we
        // tried to be optimistic and assumed that the callsite will resolve to
        // appending a simple value.
        //
        // However, we have reached here because at runtime, the guard conditional
        // have detected that this callsite is indeed referring to a component
        // definition object. Since this is likely going to be true for other
        // instances of the same callsite, it is now appropiate to deopt into the
        // expanded version that handles both cases. The compilation would look
        // like this:
        //
        //               PutValue(expression)
        //               Test(is-component-definition)
        //               Enter(BEGIN, END)
        //   BEGIN:      Noop
        //               JumpUnless(VALUE)
        //               PutDynamicComponentDefinitionOpcode
        //               OpenComponent
        //               CloseComponent
        //               Jump(END)
        //   VALUE:      Noop
        //               OptimizedAppend
        //   END:        Noop
        //               Exit
        //
        // Keep in mind that even if we *don't* reach here at initial render time,
        // it is still possible (although quite rare) that the simple value we
        // encounter during initial render could later change into a component
        // definition object at update time. That is handled by the "lazy deopt"
        // code on the update side (scroll down for the next big block of comment).
        let dsl = new OpcodeBuilderDSL(this.symbolTable, env);
        dsl.putValue(this.expression);
        dsl.test(IsComponentDefinitionReference.create);
        dsl.labelled(null, (dsl, _BEGIN, END) => {
            dsl.jumpUnless('VALUE');
            dsl.putDynamicComponentDefinition();
            dsl.openComponent(CompiledArgs.empty());
            dsl.closeComponent();
            dsl.jump(END);
            dsl.label('VALUE');
            dsl.dynamicContent(new this.AppendOpcode());
        });
        this.start = dsl.start;
        this.end = dsl.end;
        // From this point on, we have essentially replaced ourselves with a new set
        // of opcodes. Since we will always be executing the new/deopted code, it's
        // a good idea (as a pattern) to null out any unneeded fields here to avoid
        // holding on to unneeded/stale objects:
        // QUESTION: Shouldn't this whole object be GCed? If not, why not?
        this.expression = null;
        return dsl.start;
    }
}
class IsComponentDefinitionReference extends ConditionalReference {
    static create(inner) {
        return new IsComponentDefinitionReference(inner);
    }
    toBool(value) {
        return isComponentDefinition(value);
    }
}
class UpdateOpcode extends UpdatingOpcode {
    constructor(cache, bounds, upsert) {
        super();
        this.cache = cache;
        this.bounds = bounds;
        this.upsert = upsert;
        this.tag = cache.tag;
    }
    evaluate(vm) {
        let value = this.cache.revalidate();
        if (isModified(value)) {
            let { bounds, upsert } = this;
            let { dom } = vm;
            if (!this.upsert.update(dom, value)) {
                let cursor = new Cursor(bounds.parentElement(), clear(bounds));
                upsert = this.upsert = this.insert(vm.env.getAppendOperations(), cursor, value);
            }
            bounds.update(upsert.bounds);
        }
    }
    toJSON() {
        let { _guid: guid, type, cache } = this;
        return {
            guid,
            type,
            details: { lastValue: JSON.stringify(cache.peek()) }
        };
    }
}
class GuardedUpdateOpcode extends UpdateOpcode {
    constructor(reference, cache, bounds, upsert, appendOpcode, state) {
        super(cache, bounds, upsert);
        this.reference = reference;
        this.appendOpcode = appendOpcode;
        this.state = state;
        this.deopted = null;
        this.tag = this._tag = new UpdatableTag(this.tag);
    }
    evaluate(vm) {
        if (this.deopted) {
            vm.evaluateOpcode(this.deopted);
        }
        else {
            if (isComponentDefinition(this.reference.value())) {
                this.lazyDeopt(vm);
            }
            else {
                super.evaluate(vm);
            }
        }
    }
    lazyDeopt(vm) {
        // Durign initial render, we know that the reference does not contain a
        // component definition, so we optimistically assumed that this append
        // is just a normal append. However, at update time, we discovered that
        // the reference has switched into containing a component definition, so
        // we need to do a "lazy deopt", simulating what would have happened if
        // we had decided to perform the deopt in the first place during initial
        // render.
        //
        // More concretely, we would have expanded the curly into a if/else, and
        // based on whether the value is a component definition or not, we would
        // have entered either the dynamic component branch or the simple value
        // branch.
        //
        // Since we rendered a simple value during initial render (and all the
        // updates up until this point), we need to pretend that the result is
        // produced by the "VALUE" branch of the deopted append opcode:
        //
        //   Try(BEGIN, END)
        //     Assert(IsComponentDefinition, expected=false)
        //     OptimizedUpdate
        //
        // In this case, because the reference has switched from being a simple
        // value into a component definition, what would have happened is that
        // the assert would throw, causing the Try opcode to teardown the bounds
        // and rerun the original append opcode.
        //
        // Since the Try opcode would have nuked the updating opcodes anyway, we
        // wouldn't have to worry about simulating those. All we have to do is to
        // execute the Try opcode and immediately throw.
        let { bounds, appendOpcode, state } = this;
        let env = vm.env;
        let deoptStart = appendOpcode.deopt(env);
        let enter = expect(env.program.opcode(deoptStart + 8), 'hardcoded deopt location');
        let { op1: start, op2: end } = enter;
        let tracker = new UpdatableBlockTracker(bounds.parentElement());
        tracker.newBounds(this.bounds);
        let children = new LinkedList();
        state.frame.condition = IsComponentDefinitionReference.create(expect(state.frame['operand'], 'operand should be populated'));
        let deopted = this.deopted = new TryOpcode(start, end, state, tracker, children);
        this._tag.update(deopted.tag);
        vm.evaluateOpcode(deopted);
        vm.throw();
        // From this point on, we have essentially replaced ourselve with a new
        // opcode. Since we will always be executing the new/deopted code, it's a
        // good idea (as a pattern) to null out any unneeded fields here to avoid
        // holding on to unneeded/stale objects:
        // QUESTION: Shouldn't this whole object be GCed? If not, why not?
        this._tag = null;
        this.reference = null;
        this.cache = null;
        this.bounds = null;
        this.upsert = null;
        this.appendOpcode = null;
        this.state = null;
    }
    toJSON() {
        let { _guid: guid, type, deopted } = this;
        if (deopted) {
            return {
                guid,
                type,
                deopted: true,
                children: [deopted.toJSON()]
            };
        }
        else {
            return super.toJSON();
        }
    }
}
export class OptimizedCautiousAppendOpcode extends AppendDynamicOpcode {
    constructor() {
        super(...arguments);
        this.type = 'optimized-cautious-append';
    }
    normalize(reference) {
        return map(reference, normalizeValue);
    }
    insert(dom, cursor, value) {
        return cautiousInsert(dom, cursor, value);
    }
    updateWith(_vm, _reference, cache, bounds, upsert) {
        return new OptimizedCautiousUpdateOpcode(cache, bounds, upsert);
    }
}
class OptimizedCautiousUpdateOpcode extends UpdateOpcode {
    constructor() {
        super(...arguments);
        this.type = 'optimized-cautious-update';
    }
    insert(dom, cursor, value) {
        return cautiousInsert(dom, cursor, value);
    }
}
export class GuardedCautiousAppendOpcode extends GuardedAppendOpcode {
    constructor() {
        super(...arguments);
        this.type = 'guarded-cautious-append';
        this.AppendOpcode = OptimizedCautiousAppendOpcode;
    }
    normalize(reference) {
        return map(reference, normalizeValue);
    }
    insert(dom, cursor, value) {
        return cautiousInsert(dom, cursor, value);
    }
    updateWith(vm, reference, cache, bounds, upsert) {
        return new GuardedCautiousUpdateOpcode(reference, cache, bounds, upsert, this, vm.capture());
    }
}
class GuardedCautiousUpdateOpcode extends GuardedUpdateOpcode {
    constructor() {
        super(...arguments);
        this.type = 'guarded-cautious-update';
    }
    insert(dom, cursor, value) {
        return cautiousInsert(dom, cursor, value);
    }
}
export class OptimizedTrustingAppendOpcode extends AppendDynamicOpcode {
    constructor() {
        super(...arguments);
        this.type = 'optimized-trusting-append';
    }
    normalize(reference) {
        return map(reference, normalizeTrustedValue);
    }
    insert(dom, cursor, value) {
        return trustingInsert(dom, cursor, value);
    }
    updateWith(_vm, _reference, cache, bounds, upsert) {
        return new OptimizedTrustingUpdateOpcode(cache, bounds, upsert);
    }
}
class OptimizedTrustingUpdateOpcode extends UpdateOpcode {
    constructor() {
        super(...arguments);
        this.type = 'optimized-trusting-update';
    }
    insert(dom, cursor, value) {
        return trustingInsert(dom, cursor, value);
    }
}
export class GuardedTrustingAppendOpcode extends GuardedAppendOpcode {
    constructor() {
        super(...arguments);
        this.type = 'guarded-trusting-append';
        this.AppendOpcode = OptimizedTrustingAppendOpcode;
    }
    normalize(reference) {
        return map(reference, normalizeTrustedValue);
    }
    insert(dom, cursor, value) {
        return trustingInsert(dom, cursor, value);
    }
    updateWith(vm, reference, cache, bounds, upsert) {
        return new GuardedTrustingUpdateOpcode(reference, cache, bounds, upsert, this, vm.capture());
    }
}
class GuardedTrustingUpdateOpcode extends GuardedUpdateOpcode {
    constructor() {
        super(...arguments);
        this.type = 'trusting-update';
    }
    insert(dom, cursor, value) {
        return trustingInsert(dom, cursor, value);
    }
}
//# sourceMappingURL=data:application/json;base64,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