import { sanitizeAttributeValue, requiresSanitization } from './sanitized-values';
import { normalizeProperty } from './props';
import { SVG_NAMESPACE } from './helper';
import { normalizeTextValue } from '../compiled/opcodes/content';
export function defaultManagers(element, attr, _isTrusting, _namespace) {
    let tagName = element.tagName;
    let isSVG = element.namespaceURI === SVG_NAMESPACE;
    if (isSVG) {
        return defaultAttributeManagers(tagName, attr);
    }
    let { type, normalized } = normalizeProperty(element, attr);
    if (type === 'attr') {
        return defaultAttributeManagers(tagName, normalized);
    }
    else {
        return defaultPropertyManagers(tagName, normalized);
    }
}
export function defaultPropertyManagers(tagName, attr) {
    if (requiresSanitization(tagName, attr)) {
        return new SafePropertyManager(attr);
    }
    if (isUserInputValue(tagName, attr)) {
        return INPUT_VALUE_PROPERTY_MANAGER;
    }
    if (isOptionSelected(tagName, attr)) {
        return OPTION_SELECTED_MANAGER;
    }
    return new PropertyManager(attr);
}
export function defaultAttributeManagers(tagName, attr) {
    if (requiresSanitization(tagName, attr)) {
        return new SafeAttributeManager(attr);
    }
    return new AttributeManager(attr);
}
export function readDOMAttr(element, attr) {
    let isSVG = element.namespaceURI === SVG_NAMESPACE;
    let { type, normalized } = normalizeProperty(element, attr);
    if (isSVG) {
        return element.getAttribute(normalized);
    }
    if (type === 'attr') {
        return element.getAttribute(normalized);
    }
    {
        return element[normalized];
    }
}
;
export class AttributeManager {
    constructor(attr) {
        this.attr = attr;
    }
    setAttribute(env, element, value, namespace) {
        let dom = env.getAppendOperations();
        let normalizedValue = normalizeAttributeValue(value);
        if (!isAttrRemovalValue(normalizedValue)) {
            dom.setAttribute(element, this.attr, normalizedValue, namespace);
        }
    }
    updateAttribute(env, element, value, namespace) {
        if (value === null || value === undefined || value === false) {
            if (namespace) {
                env.getDOM().removeAttributeNS(element, namespace, this.attr);
            }
            else {
                env.getDOM().removeAttribute(element, this.attr);
            }
        }
        else {
            this.setAttribute(env, element, value);
        }
    }
}
;
export class PropertyManager extends AttributeManager {
    setAttribute(_env, element, value, _namespace) {
        if (!isAttrRemovalValue(value)) {
            element[this.attr] = value;
        }
    }
    removeAttribute(env, element, namespace) {
        // TODO this sucks but to preserve properties first and to meet current
        // semantics we must do this.
        let { attr } = this;
        if (namespace) {
            env.getDOM().removeAttributeNS(element, namespace, attr);
        }
        else {
            env.getDOM().removeAttribute(element, attr);
        }
    }
    updateAttribute(env, element, value, namespace) {
        // ensure the property is always updated
        element[this.attr] = value;
        if (isAttrRemovalValue(value)) {
            this.removeAttribute(env, element, namespace);
        }
    }
}
;
function normalizeAttributeValue(value) {
    if (value === false || value === undefined || value === null) {
        return null;
    }
    if (value === true) {
        return '';
    }
    // onclick function etc in SSR
    if (typeof value === 'function') {
        return null;
    }
    return String(value);
}
function isAttrRemovalValue(value) {
    return value === null || value === undefined;
}
class SafePropertyManager extends PropertyManager {
    setAttribute(env, element, value) {
        super.setAttribute(env, element, sanitizeAttributeValue(env, element, this.attr, value));
    }
    updateAttribute(env, element, value) {
        super.updateAttribute(env, element, sanitizeAttributeValue(env, element, this.attr, value));
    }
}
function isUserInputValue(tagName, attribute) {
    return (tagName === 'INPUT' || tagName === 'TEXTAREA') && attribute === 'value';
}
class InputValuePropertyManager extends AttributeManager {
    setAttribute(_env, element, value) {
        let input = element;
        input.value = normalizeTextValue(value);
    }
    updateAttribute(_env, element, value) {
        let input = element;
        let currentValue = input.value;
        let normalizedValue = normalizeTextValue(value);
        if (currentValue !== normalizedValue) {
            input.value = normalizedValue;
        }
    }
}
export const INPUT_VALUE_PROPERTY_MANAGER = new InputValuePropertyManager('value');
function isOptionSelected(tagName, attribute) {
    return tagName === 'OPTION' && attribute === 'selected';
}
class OptionSelectedManager extends PropertyManager {
    setAttribute(_env, element, value) {
        if (value !== null && value !== undefined && value !== false) {
            let option = element;
            option.selected = true;
        }
    }
    updateAttribute(_env, element, value) {
        let option = element;
        if (value) {
            option.selected = true;
        }
        else {
            option.selected = false;
        }
    }
}
export const OPTION_SELECTED_MANAGER = new OptionSelectedManager('selected');
class SafeAttributeManager extends AttributeManager {
    setAttribute(env, element, value) {
        super.setAttribute(env, element, sanitizeAttributeValue(env, element, this.attr, value));
    }
    updateAttribute(env, element, value, _namespace) {
        super.updateAttribute(env, element, sanitizeAttributeValue(env, element, this.attr, value));
    }
}
//# sourceMappingURL=data:application/json;base64,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