/*
 * @method normalizeProperty
 * @param element {HTMLElement}
 * @param slotName {String}
 * @returns {Object} { name, type }
 */
export function normalizeProperty(element, slotName) {
    let type, normalized;
    if (slotName in element) {
        normalized = slotName;
        type = 'prop';
    }
    else {
        let lower = slotName.toLowerCase();
        if (lower in element) {
            type = 'prop';
            normalized = lower;
        }
        else {
            type = 'attr';
            normalized = slotName;
        }
    }
    if (type === 'prop' &&
        (normalized.toLowerCase() === 'style' ||
            preferAttr(element.tagName, normalized))) {
        type = 'attr';
    }
    return { normalized, type };
}
export function normalizePropertyValue(value) {
    if (value === '') {
        return true;
    }
    return value;
}
// properties that MUST be set as attributes, due to:
// * browser bug
// * strange spec outlier
const ATTR_OVERRIDES = {
    // phantomjs < 2.0 lets you set it as a prop but won't reflect it
    // back to the attribute. button.getAttribute('type') === null
    BUTTON: { type: true, form: true },
    INPUT: {
        // Some version of IE (like IE9) actually throw an exception
        // if you set input.type = 'something-unknown'
        type: true,
        form: true,
        // Chrome 46.0.2464.0: 'autocorrect' in document.createElement('input') === false
        // Safari 8.0.7: 'autocorrect' in document.createElement('input') === false
        // Mobile Safari (iOS 8.4 simulator): 'autocorrect' in document.createElement('input') === true
        autocorrect: true,
        // Chrome 54.0.2840.98: 'list' in document.createElement('input') === true
        // Safari 9.1.3: 'list' in document.createElement('input') === false
        list: true
    },
    // element.form is actually a legitimate readOnly property, that is to be
    // mutated, but must be mutated by setAttribute...
    SELECT: { form: true },
    OPTION: { form: true },
    TEXTAREA: { form: true },
    LABEL: { form: true },
    FIELDSET: { form: true },
    LEGEND: { form: true },
    OBJECT: { form: true }
};
function preferAttr(tagName, propName) {
    let tag = ATTR_OVERRIDES[tagName.toUpperCase()];
    return tag && tag[propName.toLowerCase()] || false;
}
//# sourceMappingURL=data:application/json;base64,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