import { CompiledProgram, CompiledBlock } from './compiled/blocks';
import { builder } from './compiler';
import * as WireFormat from '@glimmer/wire-format';
import { entryPoint as entryPointTable, layout as layoutTable, block as blockTable } from './symbol-table';
import { STATEMENTS } from './syntax/functions';
import { SPECIALIZE } from './syntax/specialize';
export function compileStatement(statement, builder) {
    let refined = SPECIALIZE.specialize(statement, builder.symbolTable);
    STATEMENTS.compile(refined, builder);
}
export class Template {
    constructor(statements, symbolTable) {
        this.statements = statements;
        this.symbolTable = symbolTable;
    }
}
export class Layout extends Template {
}
export class EntryPoint extends Template {
    constructor() {
        super(...arguments);
        this.compiled = null;
    }
    compile(env) {
        let compiled = this.compiled;
        if (!compiled) {
            let table = this.symbolTable;
            let b = builder(env, table);
            for (let i = 0; i < this.statements.length; i++) {
                let statement = this.statements[i];
                let refined = SPECIALIZE.specialize(statement, table);
                STATEMENTS.compile(refined, b);
            }
            compiled = this.compiled = new CompiledProgram(b.start, b.end, this.symbolTable.size);
        }
        return compiled;
    }
}
export class InlineBlock extends Template {
    constructor() {
        super(...arguments);
        this.compiled = null;
    }
    splat(builder) {
        let table = builder.symbolTable;
        let locals = table.getSymbols().locals;
        if (locals) {
            builder.pushChildScope();
            builder.bindPositionalArgsForLocals(locals);
        }
        for (let i = 0; i < this.statements.length; i++) {
            let statement = this.statements[i];
            let refined = SPECIALIZE.specialize(statement, table);
            STATEMENTS.compile(refined, builder);
        }
        if (locals) {
            builder.popScope();
        }
    }
    compile(env) {
        let compiled = this.compiled;
        if (!compiled) {
            let table = this.symbolTable;
            let b = builder(env, table);
            this.splat(b);
            compiled = this.compiled = new CompiledBlock(b.start, b.end);
        }
        return compiled;
    }
}
export class PartialBlock extends Template {
    constructor() {
        super(...arguments);
        this.compiled = null;
    }
    compile(env) {
        let compiled = this.compiled;
        if (!compiled) {
            let table = this.symbolTable;
            let b = builder(env, table);
            for (let i = 0; i < this.statements.length; i++) {
                let statement = this.statements[i];
                let refined = SPECIALIZE.specialize(statement, table);
                STATEMENTS.compile(refined, b);
            }
            compiled = this.compiled = new CompiledProgram(b.start, b.end, table.size);
        }
        return compiled;
    }
}
export default class Scanner {
    constructor(block, meta, env) {
        this.block = block;
        this.meta = meta;
        this.env = env;
    }
    scanEntryPoint() {
        let { block, meta } = this;
        let symbolTable = entryPointTable(meta);
        let child = scanBlock(block, symbolTable, this.env);
        return new EntryPoint(child.statements, symbolTable);
    }
    scanLayout() {
        let { block, meta } = this;
        let { named, yields, hasPartials } = block;
        let symbolTable = layoutTable(meta, named, yields, hasPartials);
        let child = scanBlock(block, symbolTable, this.env);
        return new Layout(child.statements, symbolTable);
    }
    scanPartial(symbolTable) {
        let { block } = this;
        let child = scanBlock(block, symbolTable, this.env);
        return new PartialBlock(child.statements, symbolTable);
    }
}
export function scanBlock({ statements }, symbolTable, env) {
    return new RawInlineBlock(env, symbolTable, statements).scan();
}
export var BaselineSyntax;
(function (BaselineSyntax) {
    const { Ops } = WireFormat;
    BaselineSyntax.isScannedComponent = WireFormat.is(Ops.ScannedComponent);
    BaselineSyntax.isPrimitiveElement = WireFormat.is(Ops.OpenPrimitiveElement);
    BaselineSyntax.isOptimizedAppend = WireFormat.is(Ops.OptimizedAppend);
    BaselineSyntax.isUnoptimizedAppend = WireFormat.is(Ops.UnoptimizedAppend);
    BaselineSyntax.isAnyAttr = WireFormat.is(Ops.AnyDynamicAttr);
    BaselineSyntax.isStaticPartial = WireFormat.is(Ops.StaticPartial);
    BaselineSyntax.isDynamicPartial = WireFormat.is(Ops.DynamicPartial);
    BaselineSyntax.isFunctionExpression = WireFormat.is(Ops.Function);
    BaselineSyntax.isNestedBlock = WireFormat.is(Ops.NestedBlock);
    BaselineSyntax.isScannedBlock = WireFormat.is(Ops.ScannedBlock);
    BaselineSyntax.isDebugger = WireFormat.is(Ops.Debugger);
    var NestedBlock;
    (function (NestedBlock) {
        function defaultBlock(sexp) {
            return sexp[4];
        }
        NestedBlock.defaultBlock = defaultBlock;
        function inverseBlock(sexp) {
            return sexp[5];
        }
        NestedBlock.inverseBlock = inverseBlock;
        function params(sexp) {
            return sexp[2];
        }
        NestedBlock.params = params;
        function hash(sexp) {
            return sexp[3];
        }
        NestedBlock.hash = hash;
    })(NestedBlock = BaselineSyntax.NestedBlock || (BaselineSyntax.NestedBlock = {}));
})(BaselineSyntax || (BaselineSyntax = {}));
const { Ops } = WireFormat;
export class RawInlineBlock {
    constructor(env, table, statements) {
        this.env = env;
        this.table = table;
        this.statements = statements;
    }
    scan() {
        let buffer = [];
        for (let i = 0; i < this.statements.length; i++) {
            let statement = this.statements[i];
            if (WireFormat.Statements.isBlock(statement)) {
                buffer.push(this.specializeBlock(statement));
            }
            else if (WireFormat.Statements.isComponent(statement)) {
                buffer.push(...this.specializeComponent(statement));
            }
            else {
                buffer.push(statement);
            }
        }
        return new InlineBlock(buffer, this.table);
    }
    specializeBlock(block) {
        let [, path, params, hash, template, inverse] = block;
        return [Ops.ScannedBlock, path, params, hash, this.child(template), this.child(inverse)];
    }
    specializeComponent(sexp) {
        let [, tag, component] = sexp;
        if (this.env.hasComponentDefinition(tag, this.table)) {
            let child = this.child(component);
            let attrs = new RawInlineBlock(this.env, this.table, component.attrs);
            return [[Ops.ScannedComponent, tag, attrs, component.args, child]];
        }
        else {
            let buf = [];
            buf.push([Ops.OpenElement, tag, []]);
            buf.push(...component.attrs);
            buf.push([Ops.FlushElement]);
            buf.push(...component.statements);
            buf.push([Ops.CloseElement]);
            return buf;
        }
    }
    child(block) {
        if (!block)
            return null;
        let table = blockTable(this.table, block.locals);
        return new RawInlineBlock(this.env, table, block.statements);
    }
}
//# sourceMappingURL=data:application/json;base64,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