import * as WireFormat from '@glimmer/wire-format';
import CompiledValue from '../compiled/expressions/value';
import CompiledHasBlock, { CompiledHasBlockParams } from '../compiled/expressions/has-block';
import { BaselineSyntax } from '../scanner';
import { LOGGER, dict, assert, unwrap, unreachable } from '@glimmer/util';
import CompiledLookup, { CompiledSelf, CompiledSymbol, CompiledInPartialName } from '../compiled/expressions/lookups';
import CompiledHelper from '../compiled/expressions/helper';
import CompiledConcat from '../compiled/expressions/concat';
import { COMPILED_EMPTY_POSITIONAL_ARGS, COMPILED_EMPTY_NAMED_ARGS, EMPTY_BLOCKS, CompiledArgs, CompiledPositionalArgs, CompiledNamedArgs } from '../compiled/expressions/args';
import { CompiledGetBlockBySymbol, CompiledInPartialGetBlock } from '../compiled/expressions/has-block';
import { CompiledFunctionExpression } from '../compiled/expressions/function';
const { defaultBlock, params, hash } = BaselineSyntax.NestedBlock;
function debugCallback(context, get) {
    console.info('Use `context`, and `get(<path>)` to debug this template.');
    /* tslint:disable */
    debugger;
    /* tslint:enable */
    return { context, get };
}
function getter(vm, builder) {
    return (path) => {
        let parts = path.split('.');
        if (parts[0] === 'this') {
            parts[0] = null;
        }
        return compileRef(parts, builder).evaluate(vm);
    };
}
let callback = debugCallback;
// For testing purposes
export function setDebuggerCallback(cb) {
    callback = cb;
}
export function resetDebuggerCallback() {
    callback = debugCallback;
}
export class Compilers {
    constructor() {
        this.names = dict();
        this.funcs = [];
    }
    add(name, func) {
        this.funcs.push(func);
        this.names[name] = this.funcs.length - 1;
    }
    compile(sexp, builder) {
        let name = sexp[0];
        let index = this.names[name];
        let func = this.funcs[index];
        assert(!!func, `expected an implementation for ${sexp[0]}`);
        return func(sexp, builder);
    }
}
let { Ops } = WireFormat;
export const STATEMENTS = new Compilers();
STATEMENTS.add(Ops.Text, (sexp, builder) => {
    builder.text(sexp[1]);
});
STATEMENTS.add(Ops.Comment, (sexp, builder) => {
    builder.comment(sexp[1]);
});
STATEMENTS.add(Ops.CloseElement, (_sexp, builder) => {
    LOGGER.trace('close-element statement');
    builder.closeElement();
});
STATEMENTS.add(Ops.FlushElement, (_sexp, builder) => {
    builder.flushElement();
});
STATEMENTS.add(Ops.Modifier, (sexp, builder) => {
    let [, path, params, hash] = sexp;
    let args = compileArgs(params, hash, builder);
    if (builder.env.hasModifier(path[0], builder.symbolTable)) {
        builder.modifier(path[0], args);
    }
    else {
        throw new Error(`Compile Error ${path.join('.')} is not a modifier: Helpers may not be used in the element form.`);
    }
});
STATEMENTS.add(Ops.StaticAttr, (sexp, builder) => {
    let [, name, value, namespace] = sexp;
    builder.staticAttr(name, namespace, value);
});
STATEMENTS.add(Ops.AnyDynamicAttr, (sexp, builder) => {
    let [, name, value, namespace, trusting] = sexp;
    builder.putValue(value);
    if (namespace) {
        builder.dynamicAttrNS(name, namespace, trusting);
    }
    else {
        builder.dynamicAttr(name, trusting);
    }
});
STATEMENTS.add(Ops.OpenElement, (sexp, builder) => {
    LOGGER.trace('open-element statement');
    builder.openPrimitiveElement(sexp[1]);
});
STATEMENTS.add(Ops.OptimizedAppend, (sexp, builder) => {
    let [, value, trustingMorph] = sexp;
    let { inlines } = builder.env.macros();
    let returned = inlines.compile(sexp, builder) || value;
    if (returned === true)
        return;
    builder.putValue(returned[1]);
    if (trustingMorph) {
        builder.trustingAppend();
    }
    else {
        builder.cautiousAppend();
    }
});
STATEMENTS.add(Ops.UnoptimizedAppend, (sexp, builder) => {
    let [, value, trustingMorph] = sexp;
    let { inlines } = builder.env.macros();
    let returned = inlines.compile(sexp, builder) || value;
    if (returned === true)
        return;
    if (trustingMorph) {
        builder.guardedTrustingAppend(returned[1]);
    }
    else {
        builder.guardedCautiousAppend(returned[1]);
    }
});
STATEMENTS.add(Ops.NestedBlock, (sexp, builder) => {
    let { blocks } = builder.env.macros();
    blocks.compile(sexp, builder);
});
STATEMENTS.add(Ops.ScannedBlock, (sexp, builder) => {
    let [, path, params, hash, template, inverse] = sexp;
    let templateBlock = template && template.scan();
    let inverseBlock = inverse && inverse.scan();
    let { blocks } = builder.env.macros();
    blocks.compile([Ops.NestedBlock, path, params, hash, templateBlock, inverseBlock], builder);
});
// this fixes an issue with Ember versions using glimmer-vm@0.22 when attempting
// to use nested web components.  This is obviously not correct for angle bracket components
// but since no consumers are currently using them with glimmer@0.22.x we are hard coding
// support to just use the fallback case.
STATEMENTS.add(Ops.Component, (sexp, builder) => {
    let [, tag, component] = sexp;
    let { attrs, statements } = component;
    builder.openPrimitiveElement(tag);
    for (let i = 0; i < attrs.length; i++) {
        STATEMENTS.compile(attrs[i], builder);
    }
    builder.flushElement();
    for (let i = 0; i < statements.length; i++) {
        STATEMENTS.compile(statements[i], builder);
    }
    builder.closeElement();
});
STATEMENTS.add(Ops.ScannedComponent, (sexp, builder) => {
    let [, tag, attrs, rawArgs, rawBlock] = sexp;
    let block = rawBlock && rawBlock.scan();
    let args = compileBlockArgs(null, rawArgs, { default: block, inverse: null }, builder);
    let definition = builder.env.getComponentDefinition(tag, builder.symbolTable);
    builder.putComponentDefinition(definition);
    builder.openComponent(args, attrs.scan());
    builder.closeComponent();
});
STATEMENTS.add(Ops.StaticPartial, (sexp, builder) => {
    let [, name] = sexp;
    if (!builder.env.hasPartial(name, builder.symbolTable)) {
        throw new Error(`Compile Error: Could not find a partial named "${name}"`);
    }
    let definition = builder.env.lookupPartial(name, builder.symbolTable);
    builder.putPartialDefinition(definition);
    builder.evaluatePartial();
});
STATEMENTS.add(Ops.DynamicPartial, (sexp, builder) => {
    let [, name] = sexp;
    builder.startLabels();
    builder.putValue(name);
    builder.test('simple');
    builder.enter('BEGIN', 'END');
    builder.label('BEGIN');
    builder.jumpUnless('END');
    builder.putDynamicPartialDefinition();
    builder.evaluatePartial();
    builder.label('END');
    builder.exit();
    builder.stopLabels();
});
STATEMENTS.add(Ops.Yield, function (sexp, builder) {
    let [, to, params] = sexp;
    let args = compileArgs(params, null, builder);
    builder.yield(args, to);
});
STATEMENTS.add(Ops.Debugger, (sexp, builder) => {
    builder.putValue([Ops.Function, (vm) => {
            let context = vm.getSelf().value();
            let get = (path) => {
                return getter(vm, builder)(path).value();
            };
            callback(context, get);
        }]);
    return sexp;
});
let EXPRESSIONS = new Compilers();
export function expr(expression, builder) {
    if (Array.isArray(expression)) {
        return EXPRESSIONS.compile(expression, builder);
    }
    else {
        return new CompiledValue(expression);
    }
}
EXPRESSIONS.add(Ops.Unknown, (sexp, builder) => {
    let path = sexp[1];
    let name = path[0];
    if (builder.env.hasHelper(name, builder.symbolTable)) {
        return new CompiledHelper(name, builder.env.lookupHelper(name, builder.symbolTable), CompiledArgs.empty(), builder.symbolTable);
    }
    else {
        return compileRef(path, builder);
    }
});
EXPRESSIONS.add(Ops.Concat, ((sexp, builder) => {
    let params = sexp[1].map(p => expr(p, builder));
    return new CompiledConcat(params);
}));
EXPRESSIONS.add(Ops.Function, (sexp, builder) => {
    return new CompiledFunctionExpression(sexp[1], builder.symbolTable);
});
EXPRESSIONS.add(Ops.Helper, (sexp, builder) => {
    let { env, symbolTable } = builder;
    let [, [name], params, hash] = sexp;
    if (env.hasHelper(name, symbolTable)) {
        let args = compileArgs(params, hash, builder);
        return new CompiledHelper(name, env.lookupHelper(name, symbolTable), args, symbolTable);
    }
    else {
        throw new Error(`Compile Error: ${name} is not a helper`);
    }
});
EXPRESSIONS.add(Ops.Get, (sexp, builder) => {
    return compileRef(sexp[1], builder);
});
EXPRESSIONS.add(Ops.Undefined, (_sexp, _builder) => {
    return new CompiledValue(undefined);
});
EXPRESSIONS.add(Ops.Arg, (sexp, builder) => {
    let [, parts] = sexp;
    let head = parts[0];
    let named, partial;
    if (named = builder.symbolTable.getSymbol('named', head)) {
        let path = parts.slice(1);
        let inner = new CompiledSymbol(named, head);
        return CompiledLookup.create(inner, path);
    }
    else if (partial = builder.symbolTable.getPartialArgs()) {
        let path = parts.slice(1);
        let inner = new CompiledInPartialName(partial, head);
        return CompiledLookup.create(inner, path);
    }
    else {
        throw new Error(`[BUG] @${parts.join('.')} is not a valid lookup path.`);
    }
});
EXPRESSIONS.add(Ops.HasBlock, (sexp, builder) => {
    let blockName = sexp[1];
    let yields, partial;
    if (yields = builder.symbolTable.getSymbol('yields', blockName)) {
        let inner = new CompiledGetBlockBySymbol(yields, blockName);
        return new CompiledHasBlock(inner);
    }
    else if (partial = builder.symbolTable.getPartialArgs()) {
        let inner = new CompiledInPartialGetBlock(partial, blockName);
        return new CompiledHasBlock(inner);
    }
    else {
        throw new Error('[BUG] ${blockName} is not a valid block name.');
    }
});
EXPRESSIONS.add(Ops.HasBlockParams, (sexp, builder) => {
    let blockName = sexp[1];
    let yields, partial;
    if (yields = builder.symbolTable.getSymbol('yields', blockName)) {
        let inner = new CompiledGetBlockBySymbol(yields, blockName);
        return new CompiledHasBlockParams(inner);
    }
    else if (partial = builder.symbolTable.getPartialArgs()) {
        let inner = new CompiledInPartialGetBlock(partial, blockName);
        return new CompiledHasBlockParams(inner);
    }
    else {
        throw new Error('[BUG] ${blockName} is not a valid block name.');
    }
});
export function compileArgs(params, hash, builder) {
    let compiledParams = compileParams(params, builder);
    let compiledHash = compileHash(hash, builder);
    return CompiledArgs.create(compiledParams, compiledHash, EMPTY_BLOCKS);
}
export function compileBlockArgs(params, hash, blocks, builder) {
    let compiledParams = compileParams(params, builder);
    let compiledHash = compileHash(hash, builder);
    return CompiledArgs.create(compiledParams, compiledHash, blocks);
}
export function compileBaselineArgs(args, builder) {
    let [params, hash, _default, inverse] = args;
    return CompiledArgs.create(compileParams(params, builder), compileHash(hash, builder), { default: _default, inverse });
}
function compileParams(params, builder) {
    if (!params || params.length === 0)
        return COMPILED_EMPTY_POSITIONAL_ARGS;
    let compiled = new Array(params.length);
    for (let i = 0; i < params.length; i++) {
        compiled[i] = expr(params[i], builder);
    }
    return CompiledPositionalArgs.create(compiled);
}
function compileHash(hash, builder) {
    if (!hash)
        return COMPILED_EMPTY_NAMED_ARGS;
    let [keys, values] = hash;
    if (keys.length === 0)
        return COMPILED_EMPTY_NAMED_ARGS;
    let compiled = new Array(values.length);
    for (let i = 0; i < values.length; i++) {
        compiled[i] = expr(values[i], builder);
    }
    return new CompiledNamedArgs(keys, compiled);
}
function compileRef(parts, builder) {
    let head = parts[0];
    let local;
    if (head === null) {
        let inner = new CompiledSelf();
        let path = parts.slice(1);
        return CompiledLookup.create(inner, path);
    }
    else if (local = builder.symbolTable.getSymbol('local', head)) {
        let path = parts.slice(1);
        let inner = new CompiledSymbol(local, head);
        return CompiledLookup.create(inner, path);
    }
    else {
        let inner = new CompiledSelf();
        return CompiledLookup.create(inner, parts);
    }
}
export class Blocks {
    constructor() {
        this.names = dict();
        this.funcs = [];
    }
    add(name, func) {
        this.funcs.push(func);
        this.names[name] = this.funcs.length - 1;
    }
    addMissing(func) {
        this.missing = func;
    }
    compile(sexp, builder) {
        // assert(sexp[1].length === 1, 'paths in blocks are not supported');
        let name = sexp[1][0];
        let index = this.names[name];
        if (index === undefined) {
            assert(!!this.missing, `${name} not found, and no catch-all block handler was registered`);
            let func = this.missing;
            let handled = func(sexp, builder);
            assert(!!handled, `${name} not found, and the catch-all block handler didn't handle it`);
        }
        else {
            let func = this.funcs[index];
            func(sexp, builder);
        }
    }
}
export const BLOCKS = new Blocks();
export class Inlines {
    constructor() {
        this.names = dict();
        this.funcs = [];
    }
    add(name, func) {
        this.funcs.push(func);
        this.names[name] = this.funcs.length - 1;
    }
    addMissing(func) {
        this.missing = func;
    }
    compile(sexp, builder) {
        let value = sexp[1];
        // TODO: Fix this so that expression macros can return
        // things like components, so that {{component foo}}
        // is the same as {{(component foo)}}
        if (!Array.isArray(value))
            return ['expr', value];
        let path;
        let params;
        let hash;
        if (value[0] === Ops.Helper) {
            path = value[1];
            params = value[2];
            hash = value[3];
        }
        else if (value[0] === Ops.Unknown) {
            path = value[1];
            params = hash = null;
        }
        else {
            return ['expr', value];
        }
        if (path.length > 1 && !params && !hash) {
            return ['expr', value];
        }
        let name = path[0];
        let index = this.names[name];
        if (index === undefined && this.missing) {
            let func = this.missing;
            let returned = func(path, params, hash, builder);
            return returned === false ? ['expr', value] : returned;
        }
        else if (index !== undefined) {
            let func = this.funcs[index];
            let returned = func(path, params, hash, builder);
            return returned === false ? ['expr', value] : returned;
        }
        else {
            return ['expr', value];
        }
    }
}
export const INLINES = new Inlines();
populateBuiltins(BLOCKS, INLINES);
export function populateBuiltins(blocks = new Blocks(), inlines = new Inlines()) {
    blocks.add('if', (sexp, builder) => {
        //        PutArgs
        //        Test(Environment)
        //        Enter(BEGIN, END)
        // BEGIN: Noop
        //        JumpUnless(ELSE)
        //        Evaluate(default)
        //        Jump(END)
        // ELSE:  Noop
        //        Evalulate(inverse)
        // END:   Noop
        //        Exit
        let [, , params, hash, _default, inverse] = sexp;
        let args = compileArgs(params, hash, builder);
        builder.putArgs(args);
        builder.test('environment');
        builder.labelled(null, b => {
            if (_default && inverse) {
                b.jumpUnless('ELSE');
                b.evaluate(_default);
                b.jump('END');
                b.label('ELSE');
                b.evaluate(inverse);
            }
            else if (_default) {
                b.jumpUnless('END');
                b.evaluate(_default);
            }
            else {
                throw unreachable();
            }
        });
    });
    blocks.add('-in-element', (sexp, builder) => {
        let block = defaultBlock(sexp);
        let args = compileArgs(params(sexp), null, builder);
        builder.putArgs(args);
        builder.test('simple');
        builder.labelled(null, b => {
            b.jumpUnless('END');
            b.pushRemoteElement();
            b.evaluate(unwrap(block));
            b.popRemoteElement();
        });
    });
    blocks.add('-with-dynamic-vars', (sexp, builder) => {
        let block = defaultBlock(sexp);
        let args = compileArgs(params(sexp), hash(sexp), builder);
        builder.unit(b => {
            b.putArgs(args);
            b.pushDynamicScope();
            b.bindDynamicScope(args.named.keys);
            b.evaluate(unwrap(block));
            b.popDynamicScope();
        });
    });
    blocks.add('unless', (sexp, builder) => {
        //        PutArgs
        //        Test(Environment)
        //        Enter(BEGIN, END)
        // BEGIN: Noop
        //        JumpUnless(ELSE)
        //        Evaluate(default)
        //        Jump(END)
        // ELSE:  Noop
        //        Evalulate(inverse)
        // END:   Noop
        //        Exit
        let [, , params, hash, _default, inverse] = sexp;
        let args = compileArgs(params, hash, builder);
        builder.putArgs(args);
        builder.test('environment');
        builder.labelled(null, b => {
            if (_default && inverse) {
                b.jumpIf('ELSE');
                b.evaluate(_default);
                b.jump('END');
                b.label('ELSE');
                b.evaluate(inverse);
            }
            else if (_default) {
                b.jumpIf('END');
                b.evaluate(_default);
            }
            else {
                throw unreachable();
            }
        });
    });
    blocks.add('with', (sexp, builder) => {
        //        PutArgs
        //        Test(Environment)
        //        Enter(BEGIN, END)
        // BEGIN: Noop
        //        JumpUnless(ELSE)
        //        Evaluate(default)
        //        Jump(END)
        // ELSE:  Noop
        //        Evalulate(inverse)
        // END:   Noop
        //        Exit
        let [, , params, hash, _default, inverse] = sexp;
        let args = compileArgs(params, hash, builder);
        builder.putArgs(args);
        builder.test('environment');
        builder.labelled(null, b => {
            if (_default && inverse) {
                b.jumpUnless('ELSE');
                b.evaluate(_default);
                b.jump('END');
                b.label('ELSE');
                b.evaluate(inverse);
            }
            else if (_default) {
                b.jumpUnless('END');
                b.evaluate(_default);
            }
            else {
                throw unreachable();
            }
        });
    });
    blocks.add('each', (sexp, builder) => {
        //         Enter(BEGIN, END)
        // BEGIN:  Noop
        //         PutArgs
        //         PutIterable
        //         JumpUnless(ELSE)
        //         EnterList(BEGIN2, END2)
        // ITER:   Noop
        //         NextIter(BREAK)
        //         EnterWithKey(BEGIN2, END2)
        // BEGIN2: Noop
        //         PushChildScope
        //         Evaluate(default)
        //         PopScope
        // END2:   Noop
        //         Exit
        //         Jump(ITER)
        // BREAK:  Noop
        //         ExitList
        //         Jump(END)
        // ELSE:   Noop
        //         Evalulate(inverse)
        // END:    Noop
        //         Exit
        let [, , params, hash, _default, inverse] = sexp;
        let args = compileArgs(params, hash, builder);
        builder.labelled(args, b => {
            b.putIterator();
            if (inverse) {
                b.jumpUnless('ELSE');
            }
            else {
                b.jumpUnless('END');
            }
            b.iter(b => {
                b.evaluate(unwrap(_default));
            });
            if (inverse) {
                b.jump('END');
                b.label('ELSE');
                b.evaluate(inverse);
            }
        });
    });
    return { blocks, inlines };
}
//# sourceMappingURL=data:application/json;base64,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