import { unreachable } from '@glimmer/util';
import { SingleNodeBounds, single, clear } from './bounds';
export function isSafeString(value) {
    return !!value && typeof value['toHTML'] === 'function';
}
export function isNode(value) {
    return value !== null && typeof value === 'object' && typeof value['nodeType'] === 'number';
}
export function isString(value) {
    return typeof value === 'string';
}
class Upsert {
    constructor(bounds) {
        this.bounds = bounds;
    }
}
export default Upsert;
export function cautiousInsert(dom, cursor, value) {
    if (isString(value)) {
        return TextUpsert.insert(dom, cursor, value);
    }
    if (isSafeString(value)) {
        return SafeStringUpsert.insert(dom, cursor, value);
    }
    if (isNode(value)) {
        return NodeUpsert.insert(dom, cursor, value);
    }
    throw unreachable();
}
export function trustingInsert(dom, cursor, value) {
    if (isString(value)) {
        return HTMLUpsert.insert(dom, cursor, value);
    }
    if (isNode(value)) {
        return NodeUpsert.insert(dom, cursor, value);
    }
    throw unreachable();
}
class TextUpsert extends Upsert {
    static insert(dom, cursor, value) {
        let textNode = dom.createTextNode(value);
        dom.insertBefore(cursor.element, textNode, cursor.nextSibling);
        let bounds = new SingleNodeBounds(cursor.element, textNode);
        return new TextUpsert(bounds, textNode);
    }
    constructor(bounds, textNode) {
        super(bounds);
        this.textNode = textNode;
    }
    update(_dom, value) {
        if (isString(value)) {
            let { textNode } = this;
            textNode.nodeValue = value;
            return true;
        }
        else {
            return false;
        }
    }
}
class HTMLUpsert extends Upsert {
    static insert(dom, cursor, value) {
        let bounds = dom.insertHTMLBefore(cursor.element, value, cursor.nextSibling);
        return new HTMLUpsert(bounds);
    }
    update(dom, value) {
        if (isString(value)) {
            let { bounds } = this;
            let parentElement = bounds.parentElement();
            let nextSibling = clear(bounds);
            this.bounds = dom.insertHTMLBefore(parentElement, nextSibling, value);
            return true;
        }
        else {
            return false;
        }
    }
}
class SafeStringUpsert extends Upsert {
    constructor(bounds, lastStringValue) {
        super(bounds);
        this.lastStringValue = lastStringValue;
    }
    static insert(dom, cursor, value) {
        let stringValue = value.toHTML();
        let bounds = dom.insertHTMLBefore(cursor.element, stringValue, cursor.nextSibling);
        return new SafeStringUpsert(bounds, stringValue);
    }
    update(dom, value) {
        if (isSafeString(value)) {
            let stringValue = value.toHTML();
            if (stringValue !== this.lastStringValue) {
                let { bounds } = this;
                let parentElement = bounds.parentElement();
                let nextSibling = clear(bounds);
                this.bounds = dom.insertHTMLBefore(parentElement, nextSibling, stringValue);
                this.lastStringValue = stringValue;
            }
            return true;
        }
        else {
            return false;
        }
    }
}
class NodeUpsert extends Upsert {
    static insert(dom, cursor, node) {
        dom.insertBefore(cursor.element, node, cursor.nextSibling);
        return new NodeUpsert(single(cursor.element, node));
    }
    update(dom, value) {
        if (isNode(value)) {
            let { bounds } = this;
            let parentElement = bounds.parentElement();
            let nextSibling = clear(bounds);
            this.bounds = dom.insertNodeBefore(parentElement, value, nextSibling);
            return true;
        }
        else {
            return false;
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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