import { Scope } from '../environment';
import { Stack, LinkedList, ListSlice, assert, expect } from '@glimmer/util';
import { combineSlice } from '@glimmer/reference';
import { LabelOpcode, JumpIfNotModifiedOpcode, DidModifyOpcode } from '../compiled/opcodes/vm';
import { ListBlockOpcode, TryOpcode } from './update';
import RenderResult from './render-result';
import { FrameStack } from './frame';
import { APPEND_OPCODES, } from '../opcodes';
export default class VM {
    constructor(env, scope, dynamicScope, elementStack) {
        this.env = env;
        this.elementStack = elementStack;
        this.dynamicScopeStack = new Stack();
        this.scopeStack = new Stack();
        this.updatingOpcodeStack = new Stack();
        this.cacheGroups = new Stack();
        this.listBlockStack = new Stack();
        this.frame = new FrameStack();
        this.env = env;
        this.constants = env.constants;
        this.elementStack = elementStack;
        this.scopeStack.push(scope);
        this.dynamicScopeStack.push(dynamicScope);
    }
    static initial(env, self, dynamicScope, elementStack, compiledProgram) {
        let { symbols: size, start, end } = compiledProgram;
        let scope = Scope.root(self, size);
        let vm = new VM(env, scope, dynamicScope, elementStack);
        vm.prepare(start, end);
        return vm;
    }
    capture() {
        return {
            env: this.env,
            scope: this.scope(),
            dynamicScope: this.dynamicScope(),
            frame: this.frame.capture()
        };
    }
    goto(ip) {
        this.frame.goto(ip);
    }
    beginCacheGroup() {
        this.cacheGroups.push(this.updating().tail());
    }
    commitCacheGroup() {
        //        JumpIfNotModified(END)
        //        (head)
        //        (....)
        //        (tail)
        //        DidModify
        // END:   Noop
        let END = new LabelOpcode("END");
        let opcodes = this.updating();
        let marker = this.cacheGroups.pop();
        let head = marker ? opcodes.nextNode(marker) : opcodes.head();
        let tail = opcodes.tail();
        let tag = combineSlice(new ListSlice(head, tail));
        let guard = new JumpIfNotModifiedOpcode(tag, END);
        opcodes.insertBefore(guard, head);
        opcodes.append(new DidModifyOpcode(guard));
        opcodes.append(END);
    }
    enter(start, end) {
        let updating = new LinkedList();
        let tracker = this.stack().pushUpdatableBlock();
        let state = this.capture();
        let tryOpcode = new TryOpcode(start, end, state, tracker, updating);
        this.didEnter(tryOpcode, updating);
    }
    enterWithKey(key, start, end) {
        let updating = new LinkedList();
        let tracker = this.stack().pushUpdatableBlock();
        let state = this.capture();
        let tryOpcode = new TryOpcode(start, end, state, tracker, updating);
        this.listBlock().map[key] = tryOpcode;
        this.didEnter(tryOpcode, updating);
    }
    enterList(start, end) {
        let updating = new LinkedList();
        let tracker = this.stack().pushBlockList(updating);
        let state = this.capture();
        let artifacts = this.frame.getIterator().artifacts;
        let opcode = new ListBlockOpcode(start, end, state, tracker, updating, artifacts);
        this.listBlockStack.push(opcode);
        this.didEnter(opcode, updating);
    }
    didEnter(opcode, updating) {
        this.updateWith(opcode);
        this.updatingOpcodeStack.push(updating);
    }
    exit() {
        this.stack().popBlock();
        this.updatingOpcodeStack.pop();
        let parent = this.updating().tail();
        parent.didInitializeChildren();
    }
    exitList() {
        this.exit();
        this.listBlockStack.pop();
    }
    updateWith(opcode) {
        this.updating().append(opcode);
    }
    listBlock() {
        return expect(this.listBlockStack.current, 'expected a list block');
    }
    updating() {
        return expect(this.updatingOpcodeStack.current, 'expected updating opcode on the updating opcode stack');
    }
    stack() {
        return this.elementStack;
    }
    scope() {
        return expect(this.scopeStack.current, 'expected scope on the scope stack');
    }
    dynamicScope() {
        return expect(this.dynamicScopeStack.current, 'expected dynamic scope on the dynamic scope stack');
    }
    pushFrame(block, args, callerScope) {
        this.frame.push(block.start, block.end);
        if (args)
            this.frame.setArgs(args);
        if (args && args.blocks)
            this.frame.setBlocks(args.blocks);
        if (callerScope)
            this.frame.setCallerScope(callerScope);
    }
    pushComponentFrame(layout, args, callerScope, component, manager, shadow) {
        this.frame.push(layout.start, layout.end, component, manager, shadow);
        if (args)
            this.frame.setArgs(args);
        if (args && args.blocks)
            this.frame.setBlocks(args.blocks);
        if (callerScope)
            this.frame.setCallerScope(callerScope);
    }
    pushEvalFrame(start, end) {
        this.frame.push(start, end);
    }
    pushChildScope() {
        this.scopeStack.push(this.scope().child());
    }
    pushCallerScope() {
        this.scopeStack.push(expect(this.scope().getCallerScope(), 'pushCallerScope is called when a caller scope is present'));
    }
    pushDynamicScope() {
        let child = this.dynamicScope().child();
        this.dynamicScopeStack.push(child);
        return child;
    }
    pushRootScope(self, size) {
        let scope = Scope.root(self, size);
        this.scopeStack.push(scope);
        return scope;
    }
    popScope() {
        this.scopeStack.pop();
    }
    popDynamicScope() {
        this.dynamicScopeStack.pop();
    }
    newDestroyable(d) {
        this.stack().newDestroyable(d);
    }
    /// SCOPE HELPERS
    getSelf() {
        return this.scope().getSelf();
    }
    referenceForSymbol(symbol) {
        return this.scope().getSymbol(symbol);
    }
    getArgs() {
        return this.frame.getArgs();
    }
    /// EXECUTION
    resume(start, end, frame) {
        return this.execute(start, end, vm => vm.frame.restore(frame));
    }
    execute(start, end, initialize) {
        this.prepare(start, end, initialize);
        let result;
        while (true) {
            result = this.next();
            if (result.done)
                break;
        }
        return result.value;
    }
    prepare(start, end, initialize) {
        let { elementStack, frame, updatingOpcodeStack } = this;
        elementStack.pushSimpleBlock();
        updatingOpcodeStack.push(new LinkedList());
        frame.push(start, end);
        if (initialize)
            initialize(this);
    }
    next() {
        let { frame, env, updatingOpcodeStack, elementStack } = this;
        let opcode;
        if (opcode = frame.nextStatement(env)) {
            APPEND_OPCODES.evaluate(this, opcode);
            return { done: false, value: null };
        }
        return {
            done: true,
            value: new RenderResult(env, expect(updatingOpcodeStack.pop(), 'there should be a final updating opcode stack'), elementStack.popBlock())
        };
    }
    evaluateOpcode(opcode) {
        APPEND_OPCODES.evaluate(this, opcode);
    }
    // Make sure you have opcodes that push and pop a scope around this opcode
    // if you need to change the scope.
    invokeBlock(block, args) {
        let compiled = block.compile(this.env);
        this.pushFrame(compiled, args);
    }
    invokePartial(block) {
        let compiled = block.compile(this.env);
        this.pushFrame(compiled);
    }
    invokeLayout(args, layout, callerScope, component, manager, shadow) {
        this.pushComponentFrame(layout, args, callerScope, component, manager, shadow);
    }
    evaluateOperand(expr) {
        this.frame.setOperand(expr.evaluate(this));
    }
    evaluateArgs(args) {
        let evaledArgs = this.frame.setArgs(args.evaluate(this));
        this.frame.setOperand(evaledArgs.positional.at(0));
    }
    bindPositionalArgs(symbols) {
        let args = expect(this.frame.getArgs(), 'bindPositionalArgs assumes a previous setArgs');
        let { positional } = args;
        let scope = this.scope();
        for (let i = 0; i < symbols.length; i++) {
            scope.bindSymbol(symbols[i], positional.at(i));
        }
    }
    bindNamedArgs(names, symbols) {
        let args = expect(this.frame.getArgs(), 'bindNamedArgs assumes a previous setArgs');
        let scope = this.scope();
        let { named } = args;
        for (let i = 0; i < names.length; i++) {
            let name = this.constants.getString(names[i]);
            scope.bindSymbol(symbols[i], named.get(name));
        }
    }
    bindBlocks(names, symbols) {
        let blocks = this.frame.getBlocks();
        let scope = this.scope();
        for (let i = 0; i < names.length; i++) {
            let name = this.constants.getString(names[i]);
            scope.bindBlock(symbols[i], (blocks && blocks[name]) || null);
        }
    }
    bindPartialArgs(symbol) {
        let args = expect(this.frame.getArgs(), 'bindPartialArgs assumes a previous setArgs');
        let scope = this.scope();
        assert(args, "Cannot bind named args");
        scope.bindPartialArgs(symbol, args);
    }
    bindCallerScope() {
        let callerScope = this.frame.getCallerScope();
        let scope = this.scope();
        assert(callerScope, "Cannot bind caller scope");
        scope.bindCallerScope(callerScope);
    }
    bindDynamicScope(names) {
        let args = expect(this.frame.getArgs(), 'bindDynamicScope assumes a previous setArgs');
        let scope = this.dynamicScope();
        assert(args, "Cannot bind dynamic scope");
        for (let i = 0; i < names.length; i++) {
            let name = this.constants.getString(names[i]);
            scope.set(name, args.named.get(name));
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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