"use strict";
// Statements
Object.defineProperty(exports, "__esModule", { value: true });
function buildMustache(path, params, hash, raw, loc) {
    return {
        type: "MustacheStatement",
        path: buildPath(path),
        params: params || [],
        hash: hash || buildHash([]),
        escaped: !raw,
        loc: buildLoc(loc)
    };
}
function buildBlock(path, params, hash, program, inverse, loc) {
    return {
        type: "BlockStatement",
        path: buildPath(path),
        params: params ? params.map(buildPath) : [],
        hash: hash || buildHash([]),
        program: program || null,
        inverse: inverse || null,
        loc: buildLoc(loc)
    };
}
function buildElementModifier(path, params, hash, loc) {
    return {
        type: "ElementModifierStatement",
        path: buildPath(path),
        params: params || [],
        hash: hash || buildHash([]),
        loc: buildLoc(loc)
    };
}
function buildPartial(name, params, hash, indent) {
    return {
        type: "PartialStatement",
        name: name,
        params: params || [],
        hash: hash || buildHash([]),
        indent: indent
    };
}
function buildComment(value, loc) {
    return {
        type: "CommentStatement",
        value: value,
        loc: buildLoc(loc)
    };
}
function buildMustacheComment(value, loc) {
    return {
        type: "MustacheCommentStatement",
        value: value,
        loc: buildLoc(loc)
    };
}
function buildConcat(parts) {
    return {
        type: "ConcatStatement",
        parts: parts || []
    };
}
// Nodes
function buildElement(tag, attributes, modifiers, children, comments, loc) {
    // this is used for backwards compat prior to `comments` being added to the AST
    if (!Array.isArray(comments)) {
        loc = comments;
        comments = [];
    }
    return {
        type: "ElementNode",
        tag: tag || "",
        attributes: attributes || [],
        blockParams: [],
        modifiers: modifiers || [],
        comments: comments || [],
        children: children || [],
        loc: buildLoc(loc)
    };
}
function buildAttr(name, value, loc) {
    return {
        type: "AttrNode",
        name: name,
        value: value,
        loc: buildLoc(loc)
    };
}
function buildText(chars, loc) {
    return {
        type: "TextNode",
        chars: chars || "",
        loc: buildLoc(loc)
    };
}
// Expressions
function buildSexpr(path, params, hash, loc) {
    return {
        type: "SubExpression",
        path: buildPath(path),
        params: params || [],
        hash: hash || buildHash([]),
        loc: buildLoc(loc)
    };
}
function buildPath(original, loc) {
    if (typeof original !== 'string')
        return original;
    let parts = original.split('.');
    if (parts[0] === 'this') {
        parts[0] = null;
    }
    return {
        type: "PathExpression",
        original,
        parts,
        data: false,
        loc: buildLoc(loc)
    };
}
function buildString(value) {
    return {
        type: "StringLiteral",
        value: value,
        original: value
    };
}
function buildBoolean(value) {
    return {
        type: "BooleanLiteral",
        value: value,
        original: value
    };
}
function buildNumber(value) {
    return {
        type: "NumberLiteral",
        value: value,
        original: value
    };
}
function buildNull() {
    return {
        type: "NullLiteral",
        value: null,
        original: null
    };
}
function buildUndefined() {
    return {
        type: "UndefinedLiteral",
        value: undefined,
        original: undefined
    };
}
// Miscellaneous
function buildHash(pairs) {
    return {
        type: "Hash",
        pairs: pairs || []
    };
}
function buildPair(key, value) {
    return {
        type: "HashPair",
        key: key,
        value: value
    };
}
function buildProgram(body, blockParams, loc) {
    return {
        type: "Program",
        body: body || [],
        blockParams: blockParams || [],
        loc: buildLoc(loc)
    };
}
function buildSource(source) {
    return source || null;
}
function buildPosition(line, column) {
    return {
        line: (typeof line === 'number') ? line : null,
        column: (typeof column === 'number') ? column : null
    };
}
function buildLoc(...args) {
    if (args.length === 1) {
        let loc = args[0];
        if (typeof loc === 'object') {
            return {
                source: buildSource(loc.source),
                start: buildPosition(loc.start.line, loc.start.column),
                end: buildPosition(loc.end.line, loc.end.column)
            };
        }
        else {
            return null;
        }
    }
    else {
        let [startLine, startColumn, endLine, endColumn, source] = args;
        return {
            source: buildSource(source),
            start: buildPosition(startLine, startColumn),
            end: buildPosition(endLine, endColumn)
        };
    }
}
exports.default = {
    mustache: buildMustache,
    block: buildBlock,
    partial: buildPartial,
    comment: buildComment,
    mustacheComment: buildMustacheComment,
    element: buildElement,
    elementModifier: buildElementModifier,
    attr: buildAttr,
    text: buildText,
    sexpr: buildSexpr,
    path: buildPath,
    string: buildString,
    boolean: buildBoolean,
    number: buildNumber,
    undefined: buildUndefined,
    null: buildNull,
    concat: buildConcat,
    hash: buildHash,
    pair: buildPair,
    program: buildProgram,
    loc: buildLoc,
    pos: buildPosition
};
//# sourceMappingURL=data:application/json;base64,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