"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const builders_1 = require("../builders");
const utils_1 = require("../utils");
exports.default = {
    Program: function (program) {
        let body = [];
        let node = builders_1.default.program(body, program.blockParams, program.loc);
        let i, l = program.body.length;
        this.elementStack.push(node);
        if (l === 0) {
            return this.elementStack.pop();
        }
        for (i = 0; i < l; i++) {
            this.acceptNode(program.body[i]);
        }
        // Ensure that that the element stack is balanced properly.
        let poppedNode = this.elementStack.pop();
        if (poppedNode !== node) {
            throw new Error("Unclosed element `" + poppedNode.tag + "` (on line " + poppedNode.loc.start.line + ").");
        }
        return node;
    },
    BlockStatement: function (block) {
        delete block.inverseStrip;
        delete block.openString;
        delete block.closeStrip;
        if (this.tokenizer.state === 'comment') {
            this.appendToCommentData('{{' + this.sourceForMustache(block) + '}}');
            return;
        }
        if (this.tokenizer.state !== 'comment' && this.tokenizer.state !== 'data' && this.tokenizer.state !== 'beforeData') {
            throw new Error("A block may only be used inside an HTML element or another block.");
        }
        block = acceptCommonNodes(this, block);
        let program = block.program ? this.acceptNode(block.program) : null;
        let inverse = block.inverse ? this.acceptNode(block.inverse) : null;
        let node = builders_1.default.block(block.path, block.params, block.hash, program, inverse, block.loc);
        let parentProgram = this.currentElement();
        utils_1.appendChild(parentProgram, node);
    },
    MustacheStatement: function (rawMustache) {
        let { tokenizer } = this;
        let { path, params, hash, escaped, loc } = rawMustache;
        let mustache = builders_1.default.mustache(path, params, hash, !escaped, loc);
        if (tokenizer.state === 'comment') {
            this.appendToCommentData('{{' + this.sourceForMustache(mustache) + '}}');
            return;
        }
        acceptCommonNodes(this, mustache);
        switch (tokenizer.state) {
            // Tag helpers
            case "tagName":
                addElementModifier(this.currentNode, mustache);
                tokenizer.state = "beforeAttributeName";
                break;
            case "beforeAttributeName":
                addElementModifier(this.currentNode, mustache);
                break;
            case "attributeName":
            case "afterAttributeName":
                this.beginAttributeValue(false);
                this.finishAttributeValue();
                addElementModifier(this.currentNode, mustache);
                tokenizer.state = "beforeAttributeName";
                break;
            case "afterAttributeValueQuoted":
                addElementModifier(this.currentNode, mustache);
                tokenizer.state = "beforeAttributeName";
                break;
            // Attribute values
            case "beforeAttributeValue":
                appendDynamicAttributeValuePart(this.currentAttribute, mustache);
                tokenizer.state = 'attributeValueUnquoted';
                break;
            case "attributeValueDoubleQuoted":
            case "attributeValueSingleQuoted":
            case "attributeValueUnquoted":
                appendDynamicAttributeValuePart(this.currentAttribute, mustache);
                break;
            // TODO: Only append child when the tokenizer state makes
            // sense to do so, otherwise throw an error.
            default:
                utils_1.appendChild(this.currentElement(), mustache);
        }
        return mustache;
    },
    ContentStatement: function (content) {
        updateTokenizerLocation(this.tokenizer, content);
        this.tokenizer.tokenizePart(content.value);
        this.tokenizer.flushData();
    },
    CommentStatement: function (rawComment) {
        let { tokenizer } = this;
        let { value, loc } = rawComment;
        let comment = builders_1.default.mustacheComment(value, loc);
        if (tokenizer.state === 'comment') {
            this.appendToCommentData('{{' + this.sourceForMustache(comment) + '}}');
            return;
        }
        switch (tokenizer.state) {
            case "beforeAttributeName":
                this.currentNode.comments.push(comment);
                break;
            case 'beforeData':
            case 'data':
                utils_1.appendChild(this.currentElement(), comment);
                break;
            default:
                throw new Error(`Using a Handlebars comment when in the \`${tokenizer.state}\` state is not supported: "${comment.value}" on line ${loc.start.line}:${loc.start.column}`);
        }
        return comment;
    },
    PartialStatement: function (partial) {
        let { name, loc } = partial;
        throw new Error(`Handlebars partials are not supported: "{{> ${name.original}" at L${loc.start.line}:C${loc.start.column}`);
    },
    PartialBlockStatement: function (partialBlock) {
        let { name, loc } = partialBlock;
        throw new Error(`Handlebars partial blocks are not supported: "{{#> ${name.original}" at L${loc.start.line}:C${loc.start.column}`);
    },
    Decorator: function (decorator) {
        let { loc, path } = decorator;
        let source = this.sourceForMustache(decorator);
        throw new Error(`Handlebars decorators are not supported: "{{* ${path.original}" at L${loc.start.line}:C${loc.start.column}`);
    },
    DecoratorBlock: function (decoratorBlock) {
        let { loc, path } = decoratorBlock;
        let source = this.sourceForMustache(decoratorBlock);
        throw new Error(`Handlebars decorator blocks are not supported: "{{#* ${path.original}" at L${loc.start.line}:C${loc.start.column}`);
    },
    SubExpression: function (sexpr) {
        return acceptCommonNodes(this, sexpr);
    },
    PathExpression: function (path) {
        let { original, loc } = path;
        if (original.indexOf('/') !== -1) {
            // TODO add a SyntaxError with loc info
            if (original.slice(0, 2) === './') {
                throw new Error(`Using "./" is not supported in Glimmer and unnecessary: "${path.original}" on line ${loc.start.line}.`);
            }
            if (original.slice(0, 3) === '../') {
                throw new Error(`Changing context using "../" is not supported in Glimmer: "${path.original}" on line ${loc.start.line}.`);
            }
            if (original.indexOf('.') !== -1) {
                throw new Error(`Mixing '.' and '/' in paths is not supported in Glimmer; use only '.' to separate property paths: "${path.original}" on line ${loc.start.line}.`);
            }
            path.parts = [path.parts.join('/')];
        }
        delete path.depth;
        // This is to fix a bug in the Handlebars AST where the path expressions in
        // `{{this.foo}}` (and similarly `{{foo-bar this.foo named=this.foo}}` etc)
        // are simply turned into `{{foo}}`. The fix is to push it back onto the
        // parts array and let the runtime see the difference. However, we cannot
        // simply use the string `this` as it means literally the property called
        // "this" in the current context (it can be expressed in the syntax as
        // `{{[this]}}`, where the square bracket are generally for this kind of
        // escaping – such as `{{foo.["bar.baz"]}}` would mean lookup a property
        // named literally "bar.baz" on `this.foo`). By convention, we use `null`
        // for this purpose.
        if (original.match(/^this(\..+)?$/)) {
            path.parts.unshift(null);
        }
        return path;
    },
    Hash: function (hash) {
        for (let i = 0; i < hash.pairs.length; i++) {
            this.acceptNode(hash.pairs[i].value);
        }
        return hash;
    },
    StringLiteral: function () { },
    BooleanLiteral: function () { },
    NumberLiteral: function () { },
    UndefinedLiteral: function () { },
    NullLiteral: function () { }
};
function calculateRightStrippedOffsets(original, value) {
    if (value === '') {
        // if it is empty, just return the count of newlines
        // in original
        return {
            lines: original.split("\n").length - 1,
            columns: 0
        };
    }
    // otherwise, return the number of newlines prior to
    // `value`
    let difference = original.split(value)[0];
    let lines = difference.split(/\n/);
    let lineCount = lines.length - 1;
    return {
        lines: lineCount,
        columns: lines[lineCount].length
    };
}
function updateTokenizerLocation(tokenizer, content) {
    let line = content.loc.start.line;
    let column = content.loc.start.column;
    let offsets = calculateRightStrippedOffsets(content.original, content.value);
    line = line + offsets.lines;
    if (offsets.lines) {
        column = offsets.columns;
    }
    else {
        column = column + offsets.columns;
    }
    tokenizer.line = line;
    tokenizer.column = column;
}
function acceptCommonNodes(compiler, node) {
    compiler.acceptNode(node.path);
    if (node.params) {
        for (let i = 0; i < node.params.length; i++) {
            compiler.acceptNode(node.params[i]);
        }
    }
    else {
        node.params = [];
    }
    if (node.hash) {
        compiler.acceptNode(node.hash);
    }
    else {
        node.hash = builders_1.default.hash();
    }
    return node;
}
function addElementModifier(element, mustache) {
    let { path, params, hash, loc } = mustache;
    let modifier = builders_1.default.elementModifier(path, params, hash, loc);
    element.modifiers.push(modifier);
}
function appendDynamicAttributeValuePart(attribute, part) {
    attribute.isDynamic = true;
    attribute.parts.push(part);
}
//# sourceMappingURL=data:application/json;base64,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