"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const visitor_keys_1 = require("../types/visitor-keys");
const errors_1 = require("./errors");
function visitNode(visitor, node) {
    let handler = visitor[node.type] || visitor.All;
    let result;
    if (handler && handler.enter) {
        result = handler.enter.call(null, node);
    }
    if (result !== undefined && result !== null) {
        if (JSON.stringify(node) === JSON.stringify(result)) {
            result = undefined;
        }
        else if (Array.isArray(result)) {
            return visitArray(visitor, result) || result;
        }
        else {
            return visitNode(visitor, result) || result;
        }
    }
    if (result === undefined) {
        let keys = visitor_keys_1.default[node.type];
        for (let i = 0; i < keys.length; i++) {
            visitKey(visitor, handler, node, keys[i]);
        }
        if (handler && handler.exit) {
            result = handler.exit.call(null, node);
        }
    }
    return result;
}
function visitKey(visitor, handler, node, key) {
    let value = node[key];
    if (!value) {
        return;
    }
    let keyHandler = handler && (handler.keys[key] || handler.keys.All);
    let result;
    if (keyHandler && keyHandler.enter) {
        result = keyHandler.enter.call(null, node, key);
        if (result !== undefined) {
            throw errors_1.cannotReplaceOrRemoveInKeyHandlerYet(node, key);
        }
    }
    if (Array.isArray(value)) {
        visitArray(visitor, value);
    }
    else {
        let result = visitNode(visitor, value);
        if (result !== undefined) {
            assignKey(node, key, result);
        }
    }
    if (keyHandler && keyHandler.exit) {
        result = keyHandler.exit.call(null, node, key);
        if (result !== undefined) {
            throw errors_1.cannotReplaceOrRemoveInKeyHandlerYet(node, key);
        }
    }
}
function visitArray(visitor, array) {
    for (let i = 0; i < array.length; i++) {
        let result = visitNode(visitor, array[i]);
        if (result !== undefined) {
            i += spliceArray(array, i, result) - 1;
        }
    }
}
function assignKey(node, key, result) {
    if (result === null) {
        throw errors_1.cannotRemoveNode(node[key], node, key);
    }
    else if (Array.isArray(result)) {
        if (result.length === 1) {
            node[key] = result[0];
        }
        else {
            if (result.length === 0) {
                throw errors_1.cannotRemoveNode(node[key], node, key);
            }
            else {
                throw errors_1.cannotReplaceNode(node[key], node, key);
            }
        }
    }
    else {
        node[key] = result;
    }
}
function spliceArray(array, index, result) {
    if (result === null) {
        array.splice(index, 1);
        return 0;
    }
    else if (Array.isArray(result)) {
        array.splice(index, 1, ...result);
        return result.length;
    }
    else {
        array.splice(index, 1, result);
        return 1;
    }
}
function traverse(node, visitor) {
    visitNode(normalizeVisitor(visitor), node);
}
exports.default = traverse;
function normalizeVisitor(visitor) {
    let normalizedVisitor = {};
    for (let type in visitor) {
        let handler = visitor[type] || visitor.All;
        let normalizedKeys = {};
        if (typeof handler === 'object') {
            let keys = handler.keys;
            if (keys) {
                for (let key in keys) {
                    let keyHandler = keys[key];
                    if (typeof keyHandler === 'object') {
                        normalizedKeys[key] = {
                            enter: (typeof keyHandler.enter === 'function') ? keyHandler.enter : null,
                            exit: (typeof keyHandler.exit === 'function') ? keyHandler.exit : null
                        };
                    }
                    else if (typeof keyHandler === 'function') {
                        normalizedKeys[key] = {
                            enter: keyHandler,
                            exit: null
                        };
                    }
                }
            }
            normalizedVisitor[type] = {
                enter: (typeof handler.enter === 'function') ? handler.enter : null,
                exit: (typeof handler.exit === 'function') ? handler.exit : null,
                keys: normalizedKeys
            };
        }
        else if (typeof handler === 'function') {
            normalizedVisitor[type] = {
                enter: handler,
                exit: null,
                keys: normalizedKeys
            };
        }
    }
    return normalizedVisitor;
}
exports.normalizeVisitor = normalizeVisitor;
//# sourceMappingURL=data:application/json;base64,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