// Statements
function buildMustache(path, params, hash, raw, loc) {
    return {
        type: "MustacheStatement",
        path: buildPath(path),
        params: params || [],
        hash: hash || buildHash([]),
        escaped: !raw,
        loc: buildLoc(loc)
    };
}
function buildBlock(path, params, hash, program, inverse, loc) {
    return {
        type: "BlockStatement",
        path: buildPath(path),
        params: params ? params.map(buildPath) : [],
        hash: hash || buildHash([]),
        program: program || null,
        inverse: inverse || null,
        loc: buildLoc(loc)
    };
}
function buildElementModifier(path, params, hash, loc) {
    return {
        type: "ElementModifierStatement",
        path: buildPath(path),
        params: params || [],
        hash: hash || buildHash([]),
        loc: buildLoc(loc)
    };
}
function buildPartial(name, params, hash, indent) {
    return {
        type: "PartialStatement",
        name: name,
        params: params || [],
        hash: hash || buildHash([]),
        indent: indent
    };
}
function buildComment(value, loc) {
    return {
        type: "CommentStatement",
        value: value,
        loc: buildLoc(loc)
    };
}
function buildMustacheComment(value, loc) {
    return {
        type: "MustacheCommentStatement",
        value: value,
        loc: buildLoc(loc)
    };
}
function buildConcat(parts) {
    return {
        type: "ConcatStatement",
        parts: parts || []
    };
}
// Nodes
function buildElement(tag, attributes, modifiers, children, comments, loc) {
    // this is used for backwards compat prior to `comments` being added to the AST
    if (!Array.isArray(comments)) {
        loc = comments;
        comments = [];
    }
    return {
        type: "ElementNode",
        tag: tag || "",
        attributes: attributes || [],
        blockParams: [],
        modifiers: modifiers || [],
        comments: comments || [],
        children: children || [],
        loc: buildLoc(loc)
    };
}
function buildAttr(name, value, loc) {
    return {
        type: "AttrNode",
        name: name,
        value: value,
        loc: buildLoc(loc)
    };
}
function buildText(chars, loc) {
    return {
        type: "TextNode",
        chars: chars || "",
        loc: buildLoc(loc)
    };
}
// Expressions
function buildSexpr(path, params, hash, loc) {
    return {
        type: "SubExpression",
        path: buildPath(path),
        params: params || [],
        hash: hash || buildHash([]),
        loc: buildLoc(loc)
    };
}
function buildPath(original, loc) {
    if (typeof original !== 'string')
        return original;
    let parts = original.split('.');
    if (parts[0] === 'this') {
        parts[0] = null;
    }
    return {
        type: "PathExpression",
        original,
        parts,
        data: false,
        loc: buildLoc(loc)
    };
}
function buildString(value) {
    return {
        type: "StringLiteral",
        value: value,
        original: value
    };
}
function buildBoolean(value) {
    return {
        type: "BooleanLiteral",
        value: value,
        original: value
    };
}
function buildNumber(value) {
    return {
        type: "NumberLiteral",
        value: value,
        original: value
    };
}
function buildNull() {
    return {
        type: "NullLiteral",
        value: null,
        original: null
    };
}
function buildUndefined() {
    return {
        type: "UndefinedLiteral",
        value: undefined,
        original: undefined
    };
}
// Miscellaneous
function buildHash(pairs) {
    return {
        type: "Hash",
        pairs: pairs || []
    };
}
function buildPair(key, value) {
    return {
        type: "HashPair",
        key: key,
        value: value
    };
}
function buildProgram(body, blockParams, loc) {
    return {
        type: "Program",
        body: body || [],
        blockParams: blockParams || [],
        loc: buildLoc(loc)
    };
}
function buildSource(source) {
    return source || null;
}
function buildPosition(line, column) {
    return {
        line: (typeof line === 'number') ? line : null,
        column: (typeof column === 'number') ? column : null
    };
}
function buildLoc(...args) {
    if (args.length === 1) {
        let loc = args[0];
        if (typeof loc === 'object') {
            return {
                source: buildSource(loc.source),
                start: buildPosition(loc.start.line, loc.start.column),
                end: buildPosition(loc.end.line, loc.end.column)
            };
        }
        else {
            return null;
        }
    }
    else {
        let [startLine, startColumn, endLine, endColumn, source] = args;
        return {
            source: buildSource(source),
            start: buildPosition(startLine, startColumn),
            end: buildPosition(endLine, endColumn)
        };
    }
}
export default {
    mustache: buildMustache,
    block: buildBlock,
    partial: buildPartial,
    comment: buildComment,
    mustacheComment: buildMustacheComment,
    element: buildElement,
    elementModifier: buildElementModifier,
    attr: buildAttr,
    text: buildText,
    sexpr: buildSexpr,
    path: buildPath,
    string: buildString,
    boolean: buildBoolean,
    number: buildNumber,
    undefined: buildUndefined,
    null: buildNull,
    concat: buildConcat,
    hash: buildHash,
    pair: buildPair,
    program: buildProgram,
    loc: buildLoc,
    pos: buildPosition
};
//# sourceMappingURL=data:application/json;base64,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