import b from "../builders";
import { appendChild, parseElementBlockParams } from "../utils";
const voidMap = Object.create(null);
let voidTagNames = "area base br col command embed hr img input keygen link meta param source track wbr";
voidTagNames.split(" ").forEach(tagName => {
    voidMap[tagName] = true;
});
export default {
    reset: function () {
        this.currentNode = null;
    },
    // Comment
    beginComment: function () {
        this.currentNode = b.comment("");
        this.currentNode.loc = {
            source: null,
            start: b.pos(this.tagOpenLine, this.tagOpenColumn),
            end: null
        };
    },
    appendToCommentData: function (char) {
        this.currentNode.value += char;
    },
    finishComment: function () {
        this.currentNode.loc.end = b.pos(this.tokenizer.line, this.tokenizer.column);
        appendChild(this.currentElement(), this.currentNode);
    },
    // Data
    beginData: function () {
        this.currentNode = b.text();
        this.currentNode.loc = {
            source: null,
            start: b.pos(this.tokenizer.line, this.tokenizer.column),
            end: null
        };
    },
    appendToData: function (char) {
        this.currentNode.chars += char;
    },
    finishData: function () {
        this.currentNode.loc.end = b.pos(this.tokenizer.line, this.tokenizer.column);
        appendChild(this.currentElement(), this.currentNode);
    },
    // Tags - basic
    tagOpen: function () {
        this.tagOpenLine = this.tokenizer.line;
        this.tagOpenColumn = this.tokenizer.column;
    },
    beginStartTag: function () {
        this.currentNode = {
            type: 'StartTag',
            name: "",
            attributes: [],
            modifiers: [],
            comments: [],
            selfClosing: false,
            loc: null
        };
    },
    beginEndTag: function () {
        this.currentNode = {
            type: 'EndTag',
            name: "",
            attributes: [],
            modifiers: [],
            comments: [],
            selfClosing: false,
            loc: null
        };
    },
    finishTag: function () {
        let { line, column } = this.tokenizer;
        let tag = this.currentNode;
        tag.loc = b.loc(this.tagOpenLine, this.tagOpenColumn, line, column);
        if (tag.type === 'StartTag') {
            this.finishStartTag();
            if (voidMap[tag.name] || tag.selfClosing) {
                this.finishEndTag(true);
            }
        }
        else if (tag.type === 'EndTag') {
            this.finishEndTag(false);
        }
    },
    finishStartTag: function () {
        let { name, attributes, modifiers, comments } = this.currentNode;
        let loc = b.loc(this.tagOpenLine, this.tagOpenColumn);
        let element = b.element(name, attributes, modifiers, [], comments, loc);
        this.elementStack.push(element);
    },
    finishEndTag: function (isVoid) {
        let tag = this.currentNode;
        let element = this.elementStack.pop();
        let parent = this.currentElement();
        validateEndTag(tag, element, isVoid);
        element.loc.end.line = this.tokenizer.line;
        element.loc.end.column = this.tokenizer.column;
        parseElementBlockParams(element);
        appendChild(parent, element);
    },
    markTagAsSelfClosing: function () {
        this.currentNode.selfClosing = true;
    },
    // Tags - name
    appendToTagName: function (char) {
        this.currentNode.name += char;
    },
    // Tags - attributes
    beginAttribute: function () {
        let tag = this.currentNode;
        if (tag.type === 'EndTag') {
            throw new Error(`Invalid end tag: closing tag must not have attributes, ` +
                `in \`${tag.name}\` (on line ${this.tokenizer.line}).`);
        }
        this.currentAttribute = {
            name: "",
            parts: [],
            isQuoted: false,
            isDynamic: false,
            start: b.pos(this.tokenizer.line, this.tokenizer.column),
            valueStartLine: null,
            valueStartColumn: null
        };
    },
    appendToAttributeName: function (char) {
        this.currentAttribute.name += char;
    },
    beginAttributeValue: function (isQuoted) {
        this.currentAttribute.isQuoted = isQuoted;
        this.currentAttribute.valueStartLine = this.tokenizer.line;
        this.currentAttribute.valueStartColumn = this.tokenizer.column;
    },
    appendToAttributeValue: function (char) {
        let parts = this.currentAttribute.parts;
        let lastPart = parts[parts.length - 1];
        if (lastPart && lastPart.type === 'TextNode') {
            lastPart.chars += char;
            // update end location for each added char
            lastPart.loc.end.line = this.tokenizer.line;
            lastPart.loc.end.column = this.tokenizer.column;
        }
        else {
            // initially assume the text node is a single char
            let loc = b.loc(this.tokenizer.line, this.tokenizer.column, this.tokenizer.line, this.tokenizer.column);
            // correct for `\n` as first char
            if (char === '\n') {
                loc.start.line -= 1;
                loc.start.column = lastPart ? lastPart.loc.end.column : this.currentAttribute.valueStartColumn;
            }
            let text = b.text(char, loc);
            parts.push(text);
        }
    },
    finishAttributeValue: function () {
        let { name, parts, isQuoted, isDynamic, valueStartLine, valueStartColumn } = this.currentAttribute;
        let value = assembleAttributeValue(parts, isQuoted, isDynamic, this.tokenizer.line);
        value.loc = b.loc(valueStartLine, valueStartColumn, this.tokenizer.line, this.tokenizer.column);
        let loc = b.loc(this.currentAttribute.start.line, this.currentAttribute.start.column, this.tokenizer.line, this.tokenizer.column);
        let attribute = b.attr(name, value, loc);
        this.currentNode.attributes.push(attribute);
    },
    reportSyntaxError: function (message) {
        throw new Error(`Syntax error at line ${this.tokenizer.line} col ${this.tokenizer.column}: ${message}`);
    }
};
function assembleAttributeValue(parts, isQuoted, isDynamic, line) {
    if (isDynamic) {
        if (isQuoted) {
            return assembleConcatenatedValue(parts);
        }
        else {
            if (parts.length === 1 || (parts.length === 2 && parts[1].chars === '/')) {
                return parts[0];
            }
            else {
                throw new Error(`An unquoted attribute value must be a string or a mustache, ` +
                    `preceeded by whitespace or a '=' character, and ` +
                    `followed by whitespace, a '>' character, or '/>' (on line ${line})`);
            }
        }
    }
    else {
        return parts.length > 0 ? parts[0] : b.text("");
    }
}
function assembleConcatenatedValue(parts) {
    for (let i = 0; i < parts.length; i++) {
        let part = parts[i];
        if (part.type !== 'MustacheStatement' && part.type !== 'TextNode') {
            throw new Error("Unsupported node in quoted attribute value: " + part.type);
        }
    }
    return b.concat(parts);
}
function validateEndTag(tag, element, selfClosing) {
    let error;
    if (voidMap[tag.name] && !selfClosing) {
        // EngTag is also called by StartTag for void and self-closing tags (i.e.
        // <input> or <br />, so we need to check for that here. Otherwise, we would
        // throw an error for those cases.
        error = "Invalid end tag " + formatEndTagInfo(tag) + " (void elements cannot have end tags).";
    }
    else if (element.tag === undefined) {
        error = "Closing tag " + formatEndTagInfo(tag) + " without an open tag.";
    }
    else if (element.tag !== tag.name) {
        error = "Closing tag " + formatEndTagInfo(tag) + " did not match last open tag `" + element.tag + "` (on line " +
            element.loc.start.line + ").";
    }
    if (error) {
        throw new Error(error);
    }
}
function formatEndTagInfo(tag) {
    return "`" + tag.name + "` (on line " + tag.loc.end.line + ")";
}
//# sourceMappingURL=data:application/json;base64,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