"use strict";
var express = require('express');
var basicAuth = require('./basic-auth');
function noop() { }
var ExpressHTTPServer = (function () {
    function ExpressHTTPServer(options) {
        options = options || {};
        this.ui = options.ui;
        this.distPath = options.distPath;
        this.username = options.username;
        this.password = options.password;
        this.cache = options.cache;
        this.gzip = options.gzip || false;
        this.beforeMiddleware = options.beforeMiddleware || noop;
        this.afterMiddleware = options.afterMiddleware || noop;
        this.app = express();
    }
    ExpressHTTPServer.prototype.serve = function (fastbootMiddleware) {
        var _this = this;
        var app = this.app;
        var username = this.username;
        var password = this.password;
        this.beforeMiddleware(app);
        if (this.gzip) {
            this.app.use(require('compression')());
        }
        if (username !== undefined || password !== undefined) {
            this.ui.writeLine("adding basic auth; username=" + username + "; password=" + password);
            app.use(basicAuth(username, password));
        }
        if (this.cache) {
            app.get('/*', this.buildCacheMiddleware());
        }
        if (this.distPath) {
            app.get('/', fastbootMiddleware);
            app.use(express.static(this.distPath));
            app.get('/assets/*', function (req, res) {
                res.sendStatus(404);
            });
        }
        app.get('/*', fastbootMiddleware);
        this.afterMiddleware(app);
        return new Promise(function (resolve) {
            var listener = app.listen(process.env.PORT || 3000, function () {
                var host = listener.address().address;
                var port = listener.address().port;
                _this.ui.writeLine('HTTP server started; url=http://%s:%s', host, port);
                resolve();
            });
        });
    };
    ExpressHTTPServer.prototype.buildCacheMiddleware = function () {
        var _this = this;
        return function (req, res, next) {
            var path = req.path;
            Promise.resolve(_this.cache.fetch(path, req))
                .then(function (response) {
                if (response) {
                    _this.ui.writeLine("cache hit; path=" + path);
                    res.send(response);
                }
                else {
                    _this.ui.writeLine("cache miss; path=" + path);
                    _this.interceptResponseCompletion(path, res);
                    next();
                }
            })
                .catch(function () { return next(); });
        };
    };
    ExpressHTTPServer.prototype.interceptResponseCompletion = function (path, res) {
        var _this = this;
        var send = res.send.bind(res);
        res.send = function (body) {
            var ret = send(body);
            _this.cache.put(path, body, res)
                .then(function () {
                _this.ui.writeLine("stored in cache; path=" + path);
            })
                .catch(function () {
                var truncatedBody = body.replace(/\n/g).substr(0, 200);
                _this.ui.writeLine("error storing cache; path=" + path + "; body=" + truncatedBody + "...");
            });
            res.send = send;
            return ret;
        };
    };
    return ExpressHTTPServer;
}());
module.exports = ExpressHTTPServer;
