"use strict";
var FastBoot = require('fastboot');
var fastbootMiddleware = require('fastboot-express-middleware');
var ExpressHTTPServer = require('./express-http-server');
var Worker = (function () {
    function Worker(options) {
        this.distPath = options.distPath;
        this.httpServer = options.httpServer;
        this.ui = options.ui;
        this.cache = options.cache;
        this.gzip = options.gzip;
        this.beforeMiddleware = options.beforeMiddleware;
        this.afterMiddleware = options.afterMiddleware;
        if (!this.httpServer) {
            this.httpServer = new ExpressHTTPServer({
                ui: this.ui,
                distPath: this.distPath,
                cache: this.cache,
                gzip: this.gzip,
                beforeMiddleware: this.beforeMiddleware,
                afterMiddleware: this.afterMiddleware,
            });
        }
        if (!this.httpServer.cache) {
            this.httpServer.cache = this.cache;
        }
        if (!this.httpServer.distPath) {
            this.httpServer.distPath = this.distPath;
        }
        if (!this.httpServer.ui) {
            this.httpServer.ui = this.ui;
        }
    }
    Worker.prototype.start = function () {
        if (!this.distPath) {
            this.middleware = this.noAppMiddleware();
        }
        else {
            this.middleware = this.buildMiddleware();
        }
        this.bindEvents();
        this.serveHTTP();
    };
    Worker.prototype.bindEvents = function () {
        var _this = this;
        process.on('message', function (message) { return _this.handleMessage(message); });
    };
    Worker.prototype.handleMessage = function (message) {
        switch (message.event) {
            case 'reload':
                this.fastboot.reload();
                break;
            case 'error':
                this.error = message.error;
                break;
            case 'shutdown':
                process.exit(0);
                break;
        }
    };
    Worker.prototype.buildMiddleware = function () {
        this.fastboot = new FastBoot({
            distPath: this.distPath,
        });
        return fastbootMiddleware({
            fastboot: this.fastboot
        });
    };
    Worker.prototype.serveHTTP = function () {
        this.ui.writeLine('starting HTTP server');
        return this.httpServer.serve(this.middleware)
            .then(function () {
            process.send({ event: 'http-online' });
        });
    };
    Worker.prototype.noAppMiddleware = function () {
        var _this = this;
        return function (req, res) {
            var html = '<h1>No Application Found</h1>';
            if (_this.error) {
                html += '<pre style="color: red">' + _this.error + '</pre>';
            }
            res.status(500).send(html);
        };
    };
    return Worker;
}());
module.exports = Worker;
