(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory() :
    typeof define === 'function' && define.amd ? define(factory) :
    (factory());
}(this, function () { 'use strict';

    var assert = require('assert');
    var cluster = require('cluster');
    var os = require('os');
    var Worker = require('./worker');
    var FastBootAppServer = (function () {
        function FastBootAppServer(options) {
            options = options || {};
            this.distPath = options.distPath;
            this.downloader = options.downloader;
            this.notifier = options.notifier;
            this.cache = options.cache;
            this.ui = options.ui;
            this.gzip = options.gzip;
            this.httpServer = options.httpServer;
            this.beforeMiddleware = options.beforeMiddleware;
            this.afterMiddleware = options.afterMiddleware;
            if (!this.ui) {
                var UI = require('./ui');
                this.ui = new UI();
            }
            this.propagateUI();
            if (cluster.isWorker) {
                this.worker = new Worker({
                    ui: this.ui,
                    distPath: this.distPath || process.env.FASTBOOT_DIST_PATH,
                    cache: this.cache,
                    gzip: this.gzip,
                    httpServer: this.httpServer,
                    beforeMiddleware: this.beforeMiddleware,
                    afterMiddleware: this.afterMiddleware
                });
                this.worker.start();
            }
            else {
                this.workerCount = options.workerCount ||
                    (process.env.NODE_ENV === 'test' ? 1 : null) ||
                    os.cpus().length;
                assert(this.distPath || this.downloader, "FastBootAppServer must be provided with either a distPath or a downloader option.");
                assert(!(this.distPath && this.downloader), "FastBootAppServer must be provided with either a distPath or a downloader option, but not both.");
            }
        }
        FastBootAppServer.prototype.start = function () {
            var _this = this;
            if (cluster.isWorker) {
                return;
            }
            return this.initializeApp()
                .then(function () { return _this.subscribeToNotifier(); })
                .then(function () { return _this.forkWorkers(); })
                .then(function () {
                if (_this.initializationError) {
                    _this.broadcast({ event: 'error', error: _this.initializationError.stack });
                }
            })
                .catch(function (err) {
                _this.ui.writeLine(err.stack);
            });
        };
        FastBootAppServer.prototype.stop = function () {
            this.broadcast({ event: 'shutdown' });
        };
        FastBootAppServer.prototype.propagateUI = function () {
            if (this.downloader) {
                this.downloader.ui = this.ui;
            }
            if (this.notifier) {
                this.notifier.ui = this.ui;
            }
            if (this.cache) {
                this.cache.ui = this.ui;
            }
            if (this.httpServer) {
                this.httpServer.ui = this.ui;
            }
        };
        FastBootAppServer.prototype.initializeApp = function () {
            var _this = this;
            // If there's a downloader, it returns a promise for downloading the app
            if (this.downloader) {
                return this.downloadApp()
                    .catch(function (err) {
                    _this.ui.writeLine('Error downloading app');
                    _this.ui.writeLine(err.stack);
                    _this.initializationError = err;
                });
            }
            this.ui.writeLine("using distPath; path=" + this.distPath);
            return Promise.resolve();
        };
        FastBootAppServer.prototype.downloadApp = function () {
            var _this = this;
            this.ui.writeLine('downloading app');
            return this.downloader.download()
                .then(function (distPath) {
                _this.distPath = distPath;
            })
                .catch(function (err) {
                if (err.name.match(/AppNotFound/)) {
                    _this.ui.writeError('app not downloaded');
                }
                else {
                    throw err;
                }
            });
        };
        FastBootAppServer.prototype.subscribeToNotifier = function () {
            var _this = this;
            if (this.notifier) {
                this.ui.writeLine('subscribing to update notifications');
                return this.notifier.subscribe(function () {
                    _this.ui.writeLine('reloading server');
                    _this.initializeApp()
                        .then(function () { return _this.reload(); });
                })
                    .catch(function (err) {
                    _this.ui.writeLine('Error subscribing');
                    _this.ui.writeLine(err.stack);
                    _this.initializationError = err;
                });
            }
        };
        FastBootAppServer.prototype.broadcast = function (message) {
            var workers = cluster.workers;
            for (var id in workers) {
                workers[id].send(message);
            }
        };
        FastBootAppServer.prototype.reload = function () {
            this.broadcast({ event: 'reload' });
        };
        FastBootAppServer.prototype.forkWorkers = function () {
            var promises = [];
            for (var i = 0; i < this.workerCount; i++) {
                promises.push(this.forkWorker());
            }
            return Promise.all(promises);
        };
        FastBootAppServer.prototype.forkWorker = function () {
            var _this = this;
            var env = this.buildWorkerEnv();
            var worker = cluster.fork(env);
            this.ui.writeLine("forked worker " + worker.process.pid);
            worker.on('exit', function (code, signal) {
                if (signal) {
                    _this.ui.writeLine("worker was killed by signal: " + signal);
                }
                else if (code !== 0) {
                    _this.ui.writeLine("worker exited with error code: " + code);
                }
                else {
                    _this.ui.writeLine("worker exited");
                }
                _this.forkWorker();
            });
            return new Promise(function (resolve) {
                _this.ui.writeLine('worker online');
                worker.on('message', function (message) {
                    if (message.event === 'http-online') {
                        resolve();
                    }
                });
            });
        };
        FastBootAppServer.prototype.buildWorkerEnv = function () {
            var env = {};
            if (this.distPath) {
                env.FASTBOOT_DIST_PATH = this.distPath;
            }
            return env;
        };
        return FastBootAppServer;
    }());
    module.exports = FastBootAppServer;

}));