'use strict';
var fs = require('fs');
var path = require('path');
var RSVP = require('rsvp');
var najax = require('najax');
var SimpleDOM = require('simple-dom');
var existsSync = require('exists-sync');
var debug = require('debug')('fastboot:ember-app');
var FastBootInfo = require('./fastboot-info');
var Result = require('./result');
/**
 * @private
 *
 * The `EmberApp` class serves as a non-sandboxed wrapper around a sandboxed
 * `Ember.Application`. This bridge allows the FastBoot to quickly spin up new
 * `ApplicationInstances` initialized at a particular route, then destroy them
 * once the route has finished rendering.
 */
var EmberApp = (function () {
    /**
     * Create a new EmberApp.
     * @param {Object} options
     * @param {string} options.distPath - path to the built Ember application
     * @param {Sandbox} [options.sandbox=VMSandbox] - sandbox to use
     * @param {Object} [options.sandboxGlobals] - sandbox variables that can be added or used for overrides in the sandbox.
     */
    function EmberApp(options) {
        var distPath = path.resolve(options.distPath);
        var config = this.readPackageJSON(distPath);
        this.appFilePath = config.appFile;
        this.vendorFilePath = config.vendorFile;
        this.moduleWhitelist = config.moduleWhitelist;
        this.hostWhitelist = config.hostWhitelist;
        this.appConfig = config.appConfig;
        if (process.env.APP_CONFIG) {
            this.appConfig = JSON.parse(process.env.APP_CONFIG);
        }
        this.html = fs.readFileSync(config.htmlFile, 'utf8');
        this.sandbox = this.buildSandbox(distPath, options.sandbox, options.sandboxGlobals);
        this.app = this.retrieveSandboxedApp();
    }
    /**
     * @private
     *
     * Builds and initializes a new sandbox to run the Ember application in.
     *
     * @param {string} distPath path to the built Ember app to load
     * @param {Sandbox} [sandboxClass=VMSandbox] sandbox class to use
     * @param {Object} [sandboxGlobals={}] any additional variables to expose in the sandbox or override existing in the sandbox
     */
    EmberApp.prototype.buildSandbox = function (distPath, sandboxClass, sandboxGlobals) {
        var Sandbox = sandboxClass || require('./vm-sandbox');
        var sandboxRequire = this.buildWhitelistedRequire(this.moduleWhitelist, distPath);
        var config = this.appConfig;
        function appConfig() {
            return { default: config };
        }
        // add any additional user provided variables or override the default globals in the sandbox
        var globals = {
            najax: najax,
            FastBoot: {
                require: sandboxRequire,
                config: appConfig
            }
        };
        for (var key in sandboxGlobals) {
            if (sandboxGlobals.hasOwnProperty(key)) {
                globals[key] = sandboxGlobals[key];
            }
        }
        return new Sandbox({
            globals: globals
        });
    };
    /**
     * @private
     *
     * The Ember app runs inside a sandbox that doesn't have access to the normal
     * Node.js environment, including the `require` function. Instead, we provide
     * our own `require` method that only allows whitelisted packages to be
     * requested.
     *
     * This method takes an array of whitelisted package names and the path to the
     * built Ember app and constructs this "fake" `require` function that gets made
     * available globally inside the sandbox.
     *
     * @param {string[]} whitelist array of whitelisted package names
     * @param {string} distPath path to the built Ember app
     */
    EmberApp.prototype.buildWhitelistedRequire = function (whitelist, distPath) {
        whitelist.forEach(function (whitelistedModule) {
            debug("module whitelisted; module=%s", whitelistedModule);
        });
        return function (moduleName) {
            if (whitelist.indexOf(moduleName) > -1) {
                var nodeModulesPath = path.join(distPath, 'node_modules', moduleName);
                if (existsSync(nodeModulesPath)) {
                    return require(nodeModulesPath);
                }
                else {
                    // If it's not on disk, assume it's a built-in node package
                    return require(moduleName);
                }
            }
            else {
                throw new Error("Unable to require module '" + moduleName + "' because it was not in the whitelist.");
            }
        };
    };
    /**
     * @private
     *
     * Loads the app and vendor files in the sandbox (Node vm).
     *
    */
    EmberApp.prototype.loadAppFiles = function () {
        var sandbox = this.sandbox;
        var appFilePath = this.appFilePath;
        var vendorFilePath = this.vendorFilePath;
        sandbox.eval('sourceMapSupport.install(Error);');
        var appFile = fs.readFileSync(appFilePath, 'utf8');
        var vendorFile = fs.readFileSync(vendorFilePath, 'utf8');
        debug("evaluating app; app=%s; vendor=%s", appFilePath, vendorFilePath);
        sandbox.eval(vendorFile, vendorFilePath);
        debug("vendor file evaluated");
        sandbox.eval(appFile, appFilePath);
        debug("app file evaluated");
    };
    /**
     * @private
     *
     * Create the ember application in the sandbox.
     *
     */
    EmberApp.prototype.createEmberApp = function () {
        var sandbox = this.sandbox;
        var appFilePath = this.appFilePath;
        // Retrieve the application factory from within the sandbox
        var AppFactory = sandbox.run(function (ctx) {
            return ctx.require('~fastboot/app-factory');
        });
        // If the application factory couldn't be found, throw an error
        if (!AppFactory || typeof AppFactory['default'] !== 'function') {
            throw new Error('Failed to load Ember app from ' + appFilePath + ', make sure it was built for FastBoot with the `ember fastboot:build` command.');
        }
        // Otherwise, return a new `Ember.Application` instance
        return AppFactory['default']();
    };
    /**
     * @private
     *
     * Initializes the sandbox by evaluating the Ember app's JavaScript
     * code, then retrieves the application factory from the sandbox and creates a new
     * `Ember.Application`.
     *
     * @returns {Ember.Application} the Ember application from the sandbox
     */
    EmberApp.prototype.retrieveSandboxedApp = function () {
        this.loadAppFiles();
        return this.createEmberApp();
    };
    /**
     * Destroys the app and its sandbox.
     */
    EmberApp.prototype.destroy = function () {
        if (this.app) {
            this.app.destroy();
        }
        this.sandbox = null;
    };
    /**
     * @private
     *
     * Creates a new `ApplicationInstance` from the sandboxed `Application`.
     *
     * @returns {Promise<Ember.ApplicationInstance>} instance
     */
    EmberApp.prototype.buildAppInstance = function () {
        return this.app.boot().then(function (app) {
            debug('building instance');
            return app.buildInstance();
        });
    };
    /**
     * @private
     *
     * Main funtion that creates the app instance for every `visit` request, boots
     * the app instance and then visits the given route and destroys the app instance
     * when the route is finished its render cycle.
     *
     * @param {string} path the URL path to render, like `/photos/1`
     * @param {Object} fastbootInfo An object holding per request info
     * @param {Object} bootOptions An object containing the boot options that are used by
     *                             by ember to decide whether it needs to do rendering or not.
     * @param {Object} result
     * @return {Promise<instance>} instance
     */
    EmberApp.prototype.visitRoute = function (path, fastbootInfo, bootOptions, result) {
        var instance;
        return this.buildAppInstance()
            .then(function (appInstance) {
            instance = appInstance;
            result.instance = instance;
            registerFastBootInfo(fastbootInfo, instance);
            return instance.boot(bootOptions);
        })
            .then(function () { return result.instanceBooted = true; })
            .then(function () { return instance.visit(path, bootOptions); })
            .then(function () { return waitForApp(instance); })
            .then(function () {
            return instance;
        });
    };
    /**
     * Creates a new application instance and renders the instance at a specific
     * URL, returning a promise that resolves to a {@link Result}. The `Result`
     * givesg you access to the rendered HTML as well as metadata about the
     * request such as the HTTP status code.
     *
     * If this call to `visit()` is to service an incoming HTTP request, you may
     * provide Node's `ClientRequest` and `ServerResponse` objects as options
     * (e.g., the `res` and `req` arguments passed to Express middleware).  These
     * are provided to the Ember application via the FastBoot service.
     *
     * @param {string} path the URL path to render, like `/photos/1`
     * @param {Object} options
     * @param {string} [options.html] the HTML document to insert the rendered app into
     * @param {Object} [options.metadata] Per request specific data used in the app.
     * @param {Boolean} [options.shouldRender] whether the app should do rendering or not. If set to false, it puts the app in routing-only.
     * @param {Boolean} [options.disableShoebox] whether we should send the API data in the shoebox. If set to false, it will not send the API data used for rendering the app on server side in the index.html.
     * @param {Integer} [options.destroyAppInstanceInMs] whether to destroy the instance in the given number of ms. This is a failure mechanism to not wedge the Node process (See: https://github.com/ember-fastboot/fastboot/issues/90)
     * @param {ClientRequest}
     * @param {ClientResponse}
     * @returns {Promise<Result>} result
     */
    EmberApp.prototype.visit = function (path, options) {
        var req = options.request;
        var res = options.response;
        var html = options.html || this.html;
        var disableShoebox = options.disableShoebox || false;
        var destroyAppInstanceInMs = options.destroyAppInstanceInMs;
        var shouldRender = (options.shouldRender !== undefined) ? options.shouldRender : true;
        var bootOptions = buildBootOptions(shouldRender);
        var fastbootInfo = new FastBootInfo(req, res, { hostWhitelist: this.hostWhitelist, metadata: options.metadata });
        var doc = bootOptions.document;
        var result = new Result({
            doc: doc,
            html: html,
            fastbootInfo: fastbootInfo
        });
        var destroyAppInstanceTimer;
        if (parseInt(destroyAppInstanceInMs, 10) > 0) {
            // start a timer to destroy the appInstance forcefully in the given ms.
            // This is a failure mechanism so that node process doesn't get wedged if the `visit` never completes.
            destroyAppInstanceTimer = setTimeout(function () {
                if (instance && !result.instanceDestroyed) {
                    result.instanceDestroyed = true;
                    result.error = new Error('App instance was forcefully destroyed in ' + destroyAppInstanceInMs + 'ms');
                    instance.destroy();
                }
            }, destroyAppInstanceInMs);
        }
        var instance;
        return this.visitRoute(path, fastbootInfo, bootOptions, result)
            .then(function (appInstance) {
            instance = appInstance;
        })
            .then(function () {
            if (!disableShoebox) {
                // if shoebox is not disabled, then create the shoebox and send API data
                createShoebox(doc, fastbootInfo);
            }
        })
            .catch(function (error) { return result.error = error; })
            .then(function () { return result._finalize(); })
            .finally(function () {
            if (instance && !result.instanceDestroyed) {
                result.instanceDestroyed = true;
                instance.destroy();
                if (destroyAppInstanceTimer) {
                    clearTimeout(destroyAppInstanceTimer);
                }
            }
        });
    };
    /**
     * Given the path to a built Ember app, reads the FastBoot manifest
     * information from its `package.json` file.
     */
    EmberApp.prototype.readPackageJSON = function (distPath) {
        var pkgPath = path.join(distPath, 'package.json');
        var file;
        try {
            file = fs.readFileSync(pkgPath);
        }
        catch (e) {
            throw new Error("Couldn't find " + pkgPath + ". You may need to update your version of ember-cli-fastboot.");
        }
        var manifest;
        var pkg;
        try {
            pkg = JSON.parse(file);
            manifest = pkg.fastboot.manifest;
        }
        catch (e) {
            throw new Error(pkgPath + " was malformed or did not contain a manifest. Ensure that you have a compatible version of ember-cli-fastboot.");
        }
        return {
            appFile: path.join(distPath, manifest.appFile),
            vendorFile: path.join(distPath, manifest.vendorFile),
            htmlFile: path.join(distPath, manifest.htmlFile),
            moduleWhitelist: pkg.fastboot.moduleWhitelist,
            hostWhitelist: pkg.fastboot.hostWhitelist,
            appConfig: pkg.fastboot.appConfig
        };
    };
    return EmberApp;
}());
/*
 * Builds an object with the options required to boot an ApplicationInstance in
 * FastBoot mode.
 */
function buildBootOptions(shouldRender) {
    var doc = new SimpleDOM.Document();
    var rootElement = doc.body;
    return {
        isBrowser: false,
        document: doc,
        rootElement: rootElement,
        shouldRender: shouldRender
    };
}
/*
 * Ember apps can manually defer rendering in FastBoot mode if they're waiting
 * on something async the router doesn't know about.  This function fetches
 * that promise for deferred rendering from the app.
 */
function waitForApp(instance) {
    var fastbootInfo = instance.lookup('info:-fastboot');
    return fastbootInfo.deferredPromise.then(function () {
        return instance;
    });
}
/*
 * Writes the shoebox into the DOM for the browser rendered app to consume.
 * Uses a script tag with custom type so that the browser will treat as plain
 * text, and not expend effort trying to parse contents of the script tag.
 * Each key is written separately so that the browser rendered app can
 * parse the specific item at the time it is needed instead of everything
 * all at once.
 */
function createShoebox(doc, fastbootInfo) {
    var shoebox = fastbootInfo.shoebox;
    if (!shoebox) {
        return RSVP.resolve();
    }
    for (var key in shoebox) {
        if (!shoebox.hasOwnProperty(key)) {
            continue;
        }
        var value = shoebox[key];
        var textValue = JSON.stringify(value);
        textValue = escapeJSONString(textValue);
        var scriptText = doc.createRawHTMLSection(textValue);
        var scriptEl = doc.createElement('script');
        scriptEl.setAttribute('type', 'fastboot/shoebox');
        scriptEl.setAttribute('id', "shoebox-" + key);
        scriptEl.appendChild(scriptText);
        doc.body.appendChild(scriptEl);
    }
    return RSVP.resolve();
}
var JSON_ESCAPE = {
    '&': '\\u0026',
    '>': '\\u003e',
    '<': '\\u003c',
    '\u2028': '\\u2028',
    '\u2029': '\\u2029'
};
var JSON_ESCAPE_REGEXP = /[\u2028\u2029&><]/g;
function escapeJSONString(string) {
    return string.replace(JSON_ESCAPE_REGEXP, function (match) {
        return JSON_ESCAPE[match];
    });
}
/*
 * Builds a new FastBootInfo instance with the request and response and injects
 * it into the application instance.
 */
function registerFastBootInfo(info, instance) {
    info.register(instance);
}
module.exports = EmberApp;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZW1iZXItYXBwLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiZW1iZXItYXBwLmpzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBLFlBQVksQ0FBQztBQUViLElBQU0sRUFBRSxHQUFHLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztBQUN6QixJQUFNLElBQUksR0FBRyxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUM7QUFDN0IsSUFBTSxJQUFJLEdBQUcsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDO0FBRTdCLElBQU0sS0FBSyxHQUFHLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQztBQUMvQixJQUFNLFNBQVMsR0FBRyxPQUFPLENBQUMsWUFBWSxDQUFDLENBQUM7QUFDeEMsSUFBTSxVQUFVLEdBQUcsT0FBTyxDQUFDLGFBQWEsQ0FBQyxDQUFDO0FBQzFDLElBQU0sS0FBSyxHQUFHLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0FBRXJELElBQU0sWUFBWSxHQUFHLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0FBQ2hELElBQU0sTUFBTSxHQUFHLE9BQU8sQ0FBQyxVQUFVLENBQUMsQ0FBQztBQUVuQzs7Ozs7OztHQU9HO0FBQ0g7SUFDRTs7Ozs7O09BTUc7SUFDSCxrQkFBWSxPQUFPO1FBQ2pCLElBQUksUUFBUSxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQzlDLElBQUksTUFBTSxHQUFHLElBQUksQ0FBQyxlQUFlLENBQUMsUUFBUSxDQUFDLENBQUM7UUFFNUMsSUFBSSxDQUFDLFdBQVcsR0FBRyxNQUFNLENBQUMsT0FBTyxDQUFDO1FBQ2xDLElBQUksQ0FBQyxjQUFjLEdBQUcsTUFBTSxDQUFDLFVBQVUsQ0FBQztRQUN4QyxJQUFJLENBQUMsZUFBZSxHQUFHLE1BQU0sQ0FBQyxlQUFlLENBQUM7UUFDOUMsSUFBSSxDQUFDLGFBQWEsR0FBRyxNQUFNLENBQUMsYUFBYSxDQUFDO1FBQzFDLElBQUksQ0FBQyxTQUFTLEdBQUcsTUFBTSxDQUFDLFNBQVMsQ0FBQztRQUVsQyxFQUFFLENBQUMsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUM7WUFDM0IsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDdEQsQ0FBQztRQUVELElBQUksQ0FBQyxJQUFJLEdBQUcsRUFBRSxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsUUFBUSxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBRXJELElBQUksQ0FBQyxPQUFPLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxRQUFRLEVBQUUsT0FBTyxDQUFDLE9BQU8sRUFBRSxPQUFPLENBQUMsY0FBYyxDQUFDLENBQUM7UUFDcEYsSUFBSSxDQUFDLEdBQUcsR0FBRyxJQUFJLENBQUMsb0JBQW9CLEVBQUUsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSCwrQkFBWSxHQUFaLFVBQWEsUUFBUSxFQUFFLFlBQVksRUFBRSxjQUFjO1FBQ2pELElBQUksT0FBTyxHQUFHLFlBQVksSUFBSSxPQUFPLENBQUMsY0FBYyxDQUFDLENBQUM7UUFDdEQsSUFBSSxjQUFjLEdBQUcsSUFBSSxDQUFDLHVCQUF1QixDQUFDLElBQUksQ0FBQyxlQUFlLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDbEYsSUFBSSxNQUFNLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQztRQUM1QjtZQUNFLE1BQU0sQ0FBQyxFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUUsQ0FBQztRQUM3QixDQUFDO1FBRUQsNEZBQTRGO1FBQzVGLElBQUksT0FBTyxHQUFHO1lBQ1osS0FBSyxFQUFFLEtBQUs7WUFDWixRQUFRLEVBQUU7Z0JBQ1IsT0FBTyxFQUFFLGNBQWM7Z0JBQ3ZCLE1BQU0sRUFBRSxTQUFTO2FBQ2xCO1NBQ0YsQ0FBQztRQUNGLEdBQUcsQ0FBQyxDQUFDLElBQUksR0FBRyxJQUFJLGNBQWMsQ0FBQyxDQUFDLENBQUM7WUFDL0IsRUFBRSxDQUFDLENBQUMsY0FBYyxDQUFDLGNBQWMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ3ZDLE9BQU8sQ0FBQyxHQUFHLENBQUMsR0FBRyxjQUFjLENBQUMsR0FBRyxDQUFDLENBQUM7WUFDckMsQ0FBQztRQUNILENBQUM7UUFFRCxNQUFNLENBQUMsSUFBSSxPQUFPLENBQUM7WUFDakIsT0FBTyxFQUFFLE9BQU87U0FDakIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0gsMENBQXVCLEdBQXZCLFVBQXdCLFNBQVMsRUFBRSxRQUFRO1FBQ3pDLFNBQVMsQ0FBQyxPQUFPLENBQUMsVUFBUyxpQkFBaUI7WUFDMUMsS0FBSyxDQUFDLCtCQUErQixFQUFFLGlCQUFpQixDQUFDLENBQUM7UUFDNUQsQ0FBQyxDQUFDLENBQUM7UUFFSCxNQUFNLENBQUMsVUFBUyxVQUFVO1lBQ3hCLEVBQUUsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUN2QyxJQUFJLGVBQWUsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxjQUFjLEVBQUUsVUFBVSxDQUFDLENBQUM7Z0JBRXRFLEVBQUUsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDLENBQUM7b0JBQ2hDLE1BQU0sQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUFDLENBQUM7Z0JBQ2xDLENBQUM7Z0JBQUMsSUFBSSxDQUFDLENBQUM7b0JBQ04sMkRBQTJEO29CQUMzRCxNQUFNLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDO2dCQUM3QixDQUFDO1lBQ0gsQ0FBQztZQUFDLElBQUksQ0FBQyxDQUFDO2dCQUNOLE1BQU0sSUFBSSxLQUFLLENBQUMsNEJBQTRCLEdBQUcsVUFBVSxHQUFHLHdDQUF3QyxDQUFDLENBQUM7WUFDeEcsQ0FBQztRQUNILENBQUMsQ0FBQztJQUNKLENBQUM7SUFFRDs7Ozs7TUFLRTtJQUNGLCtCQUFZLEdBQVo7UUFDRSxJQUFJLE9BQU8sR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDO1FBQzNCLElBQUksV0FBVyxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUM7UUFDbkMsSUFBSSxjQUFjLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQztRQUV6QyxPQUFPLENBQUMsSUFBSSxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFFakQsSUFBSSxPQUFPLEdBQUcsRUFBRSxDQUFDLFlBQVksQ0FBQyxXQUFXLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDbkQsSUFBSSxVQUFVLEdBQUcsRUFBRSxDQUFDLFlBQVksQ0FBQyxjQUFjLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFFekQsS0FBSyxDQUFDLG1DQUFtQyxFQUFFLFdBQVcsRUFBRSxjQUFjLENBQUMsQ0FBQztRQUV4RSxPQUFPLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxjQUFjLENBQUMsQ0FBQztRQUN6QyxLQUFLLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUUvQixPQUFPLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxXQUFXLENBQUMsQ0FBQztRQUNuQyxLQUFLLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUM5QixDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSCxpQ0FBYyxHQUFkO1FBQ0UsSUFBSSxPQUFPLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQztRQUMzQixJQUFJLFdBQVcsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDO1FBRW5DLDJEQUEyRDtRQUMzRCxJQUFJLFVBQVUsR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLFVBQVMsR0FBRztZQUN2QyxNQUFNLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQzlDLENBQUMsQ0FBQyxDQUFDO1FBRUgsK0RBQStEO1FBQy9ELEVBQUUsQ0FBQyxDQUFDLENBQUMsVUFBVSxJQUFJLE9BQU8sVUFBVSxDQUFDLFNBQVMsQ0FBQyxLQUFLLFVBQVUsQ0FBQyxDQUFDLENBQUM7WUFDL0QsTUFBTSxJQUFJLEtBQUssQ0FBQyxnQ0FBZ0MsR0FBRyxXQUFXLEdBQUcsZ0ZBQWdGLENBQUMsQ0FBQztRQUNySixDQUFDO1FBRUQsdURBQXVEO1FBQ3ZELE1BQU0sQ0FBQyxVQUFVLENBQUMsU0FBUyxDQUFDLEVBQUUsQ0FBQztJQUNqQyxDQUFDO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSCx1Q0FBb0IsR0FBcEI7UUFDRSxJQUFJLENBQUMsWUFBWSxFQUFFLENBQUM7UUFFcEIsTUFBTSxDQUFDLElBQUksQ0FBQyxjQUFjLEVBQUUsQ0FBQztJQUMvQixDQUFDO0lBRUQ7O09BRUc7SUFDSCwwQkFBTyxHQUFQO1FBQ0UsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7WUFDYixJQUFJLENBQUMsR0FBRyxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBQ3JCLENBQUM7UUFFRCxJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0gsbUNBQWdCLEdBQWhCO1FBQ0UsTUFBTSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsSUFBSSxFQUFFLENBQUMsSUFBSSxDQUFDLFVBQVMsR0FBRztZQUN0QyxLQUFLLENBQUMsbUJBQW1CLENBQUMsQ0FBQztZQUMzQixNQUFNLENBQUMsR0FBRyxDQUFDLGFBQWEsRUFBRSxDQUFDO1FBQzdCLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSCw2QkFBVSxHQUFWLFVBQVcsSUFBSSxFQUFFLFlBQVksRUFBRSxXQUFXLEVBQUUsTUFBTTtRQUNoRCxJQUFJLFFBQVEsQ0FBQztRQUViLE1BQU0sQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLEVBQUU7YUFDM0IsSUFBSSxDQUFDLFVBQUEsV0FBVztZQUNmLFFBQVEsR0FBRyxXQUFXLENBQUM7WUFDdkIsTUFBTSxDQUFDLFFBQVEsR0FBRyxRQUFRLENBQUM7WUFDM0Isb0JBQW9CLENBQUMsWUFBWSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1lBRTdDLE1BQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBQ3BDLENBQUMsQ0FBQzthQUNELElBQUksQ0FBQyxjQUFNLE9BQUEsTUFBTSxDQUFDLGNBQWMsR0FBRyxJQUFJLEVBQTVCLENBQTRCLENBQUM7YUFDeEMsSUFBSSxDQUFDLGNBQU0sT0FBQSxRQUFRLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSxXQUFXLENBQUMsRUFBakMsQ0FBaUMsQ0FBQzthQUM3QyxJQUFJLENBQUMsY0FBTSxPQUFBLFVBQVUsQ0FBQyxRQUFRLENBQUMsRUFBcEIsQ0FBb0IsQ0FBQzthQUNoQyxJQUFJLENBQUM7WUFDSixNQUFNLENBQUMsUUFBUSxDQUFDO1FBQ2xCLENBQUMsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FxQkc7SUFDSCx3QkFBSyxHQUFMLFVBQU0sSUFBSSxFQUFFLE9BQU87UUFDakIsSUFBSSxHQUFHLEdBQUcsT0FBTyxDQUFDLE9BQU8sQ0FBQztRQUMxQixJQUFJLEdBQUcsR0FBRyxPQUFPLENBQUMsUUFBUSxDQUFDO1FBQzNCLElBQUksSUFBSSxHQUFHLE9BQU8sQ0FBQyxJQUFJLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQztRQUNyQyxJQUFJLGNBQWMsR0FBRyxPQUFPLENBQUMsY0FBYyxJQUFJLEtBQUssQ0FBQztRQUNyRCxJQUFJLHNCQUFzQixHQUFHLE9BQU8sQ0FBQyxzQkFBc0IsQ0FBQztRQUU1RCxJQUFJLFlBQVksR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEtBQUssU0FBUyxDQUFDLEdBQUcsT0FBTyxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUM7UUFDdEYsSUFBSSxXQUFXLEdBQUcsZ0JBQWdCLENBQUMsWUFBWSxDQUFDLENBQUM7UUFDakQsSUFBSSxZQUFZLEdBQUcsSUFBSSxZQUFZLENBQ2pDLEdBQUcsRUFDSCxHQUFHLEVBQ0gsRUFBRSxhQUFhLEVBQUUsSUFBSSxDQUFDLGFBQWEsRUFBRSxRQUFRLEVBQUUsT0FBTyxDQUFDLFFBQVEsRUFBRSxDQUNsRSxDQUFDO1FBRUYsSUFBSSxHQUFHLEdBQUcsV0FBVyxDQUFDLFFBQVEsQ0FBQztRQUUvQixJQUFJLE1BQU0sR0FBRyxJQUFJLE1BQU0sQ0FBQztZQUN0QixHQUFHLEVBQUUsR0FBRztZQUNSLElBQUksRUFBRSxJQUFJO1lBQ1YsWUFBWSxFQUFFLFlBQVk7U0FDM0IsQ0FBQyxDQUFDO1FBRUgsSUFBSSx1QkFBdUIsQ0FBQztRQUM1QixFQUFFLENBQUMsQ0FBQyxRQUFRLENBQUMsc0JBQXNCLEVBQUUsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUM3Qyx1RUFBdUU7WUFDdkUsc0dBQXNHO1lBQ3RHLHVCQUF1QixHQUFHLFVBQVUsQ0FBQztnQkFDbkMsRUFBRSxDQUFDLENBQUMsUUFBUSxJQUFJLENBQUMsTUFBTSxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQztvQkFDMUMsTUFBTSxDQUFDLGlCQUFpQixHQUFHLElBQUksQ0FBQztvQkFDaEMsTUFBTSxDQUFDLEtBQUssR0FBRyxJQUFJLEtBQUssQ0FBQywyQ0FBMkMsR0FBRyxzQkFBc0IsR0FBRyxJQUFJLENBQUMsQ0FBQztvQkFDdEcsUUFBUSxDQUFDLE9BQU8sRUFBRSxDQUFDO2dCQUNyQixDQUFDO1lBQ0gsQ0FBQyxFQUFFLHNCQUFzQixDQUFDLENBQUM7UUFDN0IsQ0FBQztRQUVELElBQUksUUFBUSxDQUFDO1FBQ2IsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFLFlBQVksRUFBRSxXQUFXLEVBQUUsTUFBTSxDQUFDO2FBQzVELElBQUksQ0FBQyxVQUFBLFdBQVc7WUFDZixRQUFRLEdBQUcsV0FBVyxDQUFDO1FBQ3pCLENBQUMsQ0FBQzthQUNELElBQUksQ0FBQztZQUNKLEVBQUUsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQztnQkFDcEIsd0VBQXdFO2dCQUN4RSxhQUFhLENBQUMsR0FBRyxFQUFFLFlBQVksQ0FBQyxDQUFDO1lBQ25DLENBQUM7UUFDSCxDQUFDLENBQUM7YUFDRCxLQUFLLENBQUMsVUFBQSxLQUFLLElBQUksT0FBQSxNQUFNLENBQUMsS0FBSyxHQUFHLEtBQUssRUFBcEIsQ0FBb0IsQ0FBQzthQUNwQyxJQUFJLENBQUMsY0FBTSxPQUFBLE1BQU0sQ0FBQyxTQUFTLEVBQUUsRUFBbEIsQ0FBa0IsQ0FBQzthQUM5QixPQUFPLENBQUM7WUFDUCxFQUFFLENBQUMsQ0FBQyxRQUFRLElBQUksQ0FBQyxNQUFNLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxDQUFDO2dCQUMxQyxNQUFNLENBQUMsaUJBQWlCLEdBQUcsSUFBSSxDQUFDO2dCQUNoQyxRQUFRLENBQUMsT0FBTyxFQUFFLENBQUM7Z0JBRW5CLEVBQUUsQ0FBQyxDQUFDLHVCQUF1QixDQUFDLENBQUMsQ0FBQztvQkFDNUIsWUFBWSxDQUFDLHVCQUF1QixDQUFDLENBQUM7Z0JBQ3hDLENBQUM7WUFDSCxDQUFDO1FBQ0gsQ0FBQyxDQUFDLENBQUM7SUFDUCxDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsa0NBQWUsR0FBZixVQUFnQixRQUFRO1FBQ3RCLElBQUksT0FBTyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBQ2xELElBQUksSUFBSSxDQUFDO1FBRVQsSUFBSSxDQUFDO1lBQ0gsSUFBSSxHQUFHLEVBQUUsQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDbEMsQ0FBRTtRQUFBLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDWCxNQUFNLElBQUksS0FBSyxDQUFDLG1CQUFpQixPQUFPLGlFQUE4RCxDQUFDLENBQUM7UUFDMUcsQ0FBQztRQUVELElBQUksUUFBUSxDQUFDO1FBQ2IsSUFBSSxHQUFHLENBQUM7UUFFUixJQUFJLENBQUM7WUFDSCxHQUFHLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUN2QixRQUFRLEdBQUcsR0FBRyxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUM7UUFDbkMsQ0FBRTtRQUFBLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDWCxNQUFNLElBQUksS0FBSyxDQUFJLE9BQU8sbUhBQWdILENBQUMsQ0FBQztRQUM5SSxDQUFDO1FBRUQsTUFBTSxDQUFDO1lBQ0wsT0FBTyxFQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLFFBQVEsQ0FBQyxPQUFPLENBQUM7WUFDL0MsVUFBVSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLFFBQVEsQ0FBQyxVQUFVLENBQUM7WUFDcEQsUUFBUSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLFFBQVEsQ0FBQyxRQUFRLENBQUM7WUFDaEQsZUFBZSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsZUFBZTtZQUM3QyxhQUFhLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxhQUFhO1lBQ3pDLFNBQVMsRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLFNBQVM7U0FDbEMsQ0FBQztJQUNKLENBQUM7SUFDSCxlQUFDO0FBQUQsQ0FBQyxBQWxWRCxJQWtWQztBQUVEOzs7R0FHRztBQUNILDBCQUEwQixZQUFZO0lBQ3BDLElBQUksR0FBRyxHQUFHLElBQUksU0FBUyxDQUFDLFFBQVEsRUFBRSxDQUFDO0lBQ25DLElBQUksV0FBVyxHQUFHLEdBQUcsQ0FBQyxJQUFJLENBQUM7SUFFM0IsTUFBTSxDQUFDO1FBQ0wsU0FBUyxFQUFFLEtBQUs7UUFDaEIsUUFBUSxFQUFFLEdBQUc7UUFDYixhQUFBLFdBQVc7UUFDWCxjQUFBLFlBQVk7S0FDYixDQUFDO0FBQ0osQ0FBQztBQUVEOzs7O0dBSUc7QUFDSCxvQkFBb0IsUUFBUTtJQUMxQixJQUFJLFlBQVksR0FBRyxRQUFRLENBQUMsTUFBTSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFFckQsTUFBTSxDQUFDLFlBQVksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDO1FBQ3ZDLE1BQU0sQ0FBQyxRQUFRLENBQUM7SUFDbEIsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDO0FBRUQ7Ozs7Ozs7R0FPRztBQUNILHVCQUF1QixHQUFHLEVBQUUsWUFBWTtJQUN0QyxJQUFJLE9BQU8sR0FBRyxZQUFZLENBQUMsT0FBTyxDQUFDO0lBQ25DLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztRQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLENBQUM7SUFBQyxDQUFDO0lBRXhDLEdBQUcsQ0FBQyxDQUFDLElBQUksR0FBRyxJQUFJLE9BQU8sQ0FBQyxDQUFDLENBQUM7UUFDeEIsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUFDLFFBQVEsQ0FBQztRQUFDLENBQUM7UUFFL0MsSUFBSSxLQUFLLEdBQUcsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3pCLElBQUksU0FBUyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDdEMsU0FBUyxHQUFHLGdCQUFnQixDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBRXhDLElBQUksVUFBVSxHQUFHLEdBQUcsQ0FBQyxvQkFBb0IsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNyRCxJQUFJLFFBQVEsR0FBRyxHQUFHLENBQUMsYUFBYSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBRTNDLFFBQVEsQ0FBQyxZQUFZLENBQUMsTUFBTSxFQUFFLGtCQUFrQixDQUFDLENBQUM7UUFDbEQsUUFBUSxDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUsYUFBVyxHQUFLLENBQUMsQ0FBQztRQUM5QyxRQUFRLENBQUMsV0FBVyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQ2pDLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFFRCxNQUFNLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxDQUFDO0FBQ3hCLENBQUM7QUFFRCxJQUFNLFdBQVcsR0FBRztJQUNsQixHQUFHLEVBQUUsU0FBUztJQUNkLEdBQUcsRUFBRSxTQUFTO0lBQ2QsR0FBRyxFQUFFLFNBQVM7SUFDZCxRQUFRLEVBQUUsU0FBUztJQUNuQixRQUFRLEVBQUUsU0FBUztDQUNwQixDQUFDO0FBRUYsSUFBTSxrQkFBa0IsR0FBRyxvQkFBb0IsQ0FBQztBQUVoRCwwQkFBMEIsTUFBTTtJQUM5QixNQUFNLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxrQkFBa0IsRUFBRSxVQUFTLEtBQUs7UUFDdEQsTUFBTSxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUM1QixDQUFDLENBQUMsQ0FBQztBQUNMLENBQUM7QUFFRDs7O0dBR0c7QUFDSCw4QkFBOEIsSUFBSSxFQUFFLFFBQVE7SUFDMUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsQ0FBQztBQUMxQixDQUFDO0FBRUQsTUFBTSxDQUFDLE9BQU8sR0FBRyxRQUFRLENBQUMifQ==