'use strict';
var EmberApp = require('./ember-app');
/**
 * FastBoot renders your Ember.js applications in Node.js. Start by
 * instantiating this class with the path to your compiled Ember app:
 *
 *
 * #### Sandboxing
 *
 * For security and correctness reasons, Ember applications running in FastBoot
 * are run inside a sandbox that prohibits them from accessing the normal
 * Node.js environment.
 *
 * By default, this sandbox is the built-in `VMSandbox` class, which uses
 * Node's `vm` module. You may provide your own sandbox implementation by
 * passing the `sandbox` option or add and/or override sandbox variables by
 * passing the `addOrOverrideSandboxGlobals` option.
 *
 * @example
 * const FastBoot = require('fastboot');
 *
 * let app = new FastBoot({
 *   distPath: 'path/to/dist',
 *   sandbox: 'path/to/sandboxClass',
 *   sandboxGlobals: {...}
 * });
 *
 * app.visit('/photos')
 *   .then(result => result.html())
 *   .then(html => res.send(html));
 */
var FastBoot = (function () {
    /**
     * Create a new FastBoot instance.
     * @param {Object} options
     * @param {string} options.distPath the path to the built Ember application
     * @param {Boolean} [options.resilient=false] if true, errors during rendering won't reject the `visit()` promise but instead resolve to a {@link Result}
     * @param {Sandbox} [options.sandbox=VMSandbox] the sandbox to use
     * @param {Object} [options.sandboxGlobals={}] any additional sandbox variables that an app server wants to override and/or add in the sandbox
     */
    function FastBoot(options) {
        options = options || {};
        this.distPath = options.distPath;
        this.sandbox = options.sandbox;
        this.sandboxGlobals = options.sandboxGlobals || {};
        this.resilient = !!options.resilient || false;
        this._buildEmberApp(this.distPath, this.sandbox, this.sandboxGlobals);
    }
    /**
     * Renders the Ember app at a specific URL, returning a promise that resolves
     * to a {@link Result}, giving you access to the rendered HTML as well as
     * metadata about the request such as the HTTP status code.
     *
     * @param {string} path the URL path to render, like `/photos/1`
     * @param {Object} options
     * @param {Boolean} [options.resilient] whether to reject the returned promise if there is an error during rendering. Overrides the instance's `resilient` setting
     * @param {string} [options.html] the HTML document to insert the rendered app into. Uses the built app's index.html by default.
     * @param {Object} [options.metadata] per request meta data that need to be exposed in the app.
     * @param {Boolean} [options.shouldRender] whether the app should do rendering or not. If set to false, it puts the app in routing-only.
     * @param {Boolean} [options.disableShoebox] whether we should send the API data in the shoebox. If set to false, it will not send the API data used for rendering the app on server side in the index.html.
     * @param {Integer} [options.destroyAppInstanceInMs] whether to destroy the instance in the given number of ms. This is a failure mechanism to not wedge the Node process (See: https://github.com/ember-fastboot/fastboot/issues/90)
     * @returns {Promise<Result>} result
     */
    FastBoot.prototype.visit = function (path, options) {
        options = options || {};
        var resilient = options.resilient;
        if (resilient === undefined) {
            resilient = this.resilient;
        }
        return this._app.visit(path, options)
            .then(function (result) {
            if (!resilient && result.error) {
                throw result.error;
            }
            else {
                return result;
            }
        });
    };
    FastBoot.prototype.reload = function (options) {
        if (this._app) {
            this._app.destroy();
        }
        this._buildEmberApp(options ? options.distPath : null);
    };
    FastBoot.prototype._buildEmberApp = function (distPath, sandbox, sandboxGlobals) {
        distPath = distPath || this.distPath;
        if (!distPath) {
            throw new Error('You must instantiate FastBoot with a distPath ' +
                'option that contains a path to a dist directory ' +
                'produced by running ember fastboot:build in your Ember app:' +
                '\n\n' +
                'new FastBootServer({\n' +
                '  distPath: \'path/to/dist\'\n' +
                '});');
        }
        this.distPath = distPath;
        this._app = new EmberApp({
            distPath: distPath,
            sandbox: sandbox,
            sandboxGlobals: sandboxGlobals
        });
    };
    return FastBoot;
}());
module.exports = FastBoot;
//# sourceMappingURL=data:application/json;base64,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