'use strict';
var SimpleDOM = require('simple-dom');
var HTMLSerializer = new SimpleDOM.HTMLSerializer(SimpleDOM.voidMap);
/**
 * Represents the rendered result of visiting an Ember app at a particular URL.
 * A `Result` object is returned from calling {@link FastBoot}'s `visit()`
 * method.
 */
var Result = (function () {
    function Result(options) {
        this.instanceBooted = false;
        this.instanceDestroyed = false;
        this._doc = options.doc;
        this._html = options.html;
        this._fastbootInfo = options.fastbootInfo;
    }
    /**
     * Returns the HTML representation of the rendered route, inserted
     * into the application's `index.html`.
     *
     * @returns {Promise<String>} the application's DOM serialized to HTML
     */
    Result.prototype.html = function () {
        var response = this._fastbootInfo.response;
        var statusCode = response && this._fastbootInfo.response.statusCode;
        if (statusCode === 204) {
            this._html = '';
            this._head = '';
            this._body = '';
        }
        else if (statusCode >= 300 && statusCode <= 399) {
            var location = response.headers.get('location');
            this._html = '<body><!-- EMBER_CLI_FASTBOOT_BODY --></body>';
            this._head = '';
            this._body = '';
            if (location) {
                this._body = "<h1>Redirecting to <a href=\"" + location + "\">" + location + "</a></h1>";
            }
        }
        return Promise.resolve(insertIntoIndexHTML(this._html, this._head, this._body));
    };
    /**
     * Returns the serialized representation of DOM HEAD and DOM BODY
     *
     * @returns {Object} serialized version of DOM
     */
    Result.prototype.domContents = function () {
        return {
            head: this._head,
            body: this._body
        };
    };
    /**
     * @private
     *
     * Called once the Result has finished being constructed and the application
     * instance has finished rendering. Once `finalize()` is called, state is
     * gathered from the completed application instance and statically copied
     * to this Result instance.
     */
    Result.prototype._finalize = function () {
        if (this.finalized) {
            throw new Error("Results cannot be finalized more than once");
        }
        // Grab some metadata from the sandboxed application instance
        // and copy it to this Result object.
        var instance = this.instance;
        if (instance) {
            this._finalizeMetadata(instance);
        }
        this._finalizeHTML();
        this.finalized = true;
        return this;
    };
    Result.prototype._finalizeMetadata = function (instance) {
        if (this.instanceBooted) {
            this.url = instance.getURL();
        }
        var response = this._fastbootInfo.response;
        if (response) {
            this.headers = response.headers;
            this.statusCode = response.statusCode;
        }
    };
    Result.prototype._finalizeHTML = function () {
        var head = this._doc.head;
        var body = this._doc.body;
        if (head) {
            head = HTMLSerializer.serializeChildren(head);
        }
        body = HTMLSerializer.serializeChildren(body);
        this._head = head;
        this._body = body;
    };
    return Result;
}());
function insertIntoIndexHTML(html, head, body) {
    html = html.replace("<!-- EMBER_CLI_FASTBOOT_BODY -->", body);
    if (head) {
        html = html.replace("<!-- EMBER_CLI_FASTBOOT_HEAD -->", head);
    }
    return html;
}
module.exports = Result;
//# sourceMappingURL=data:application/json;base64,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