/*
Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements.  See the NOTICE file
distributed with this work for additional information
regarding copyright ownership.  The ASF licenses this file
to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.
*/

/* Fixed Data in ROM - Field and Curve parameters */

package amcl

type Chunk int64


/* First the 32/64-bit dependent BIG code */
/* Note that because of the lack of a 128-bit integer, 32 and 64-bit code needs to be done differently */

func pexceed(a *BIG,b *BIG) bool {
	ea:=EXCESS(a)
	eb:=EXCESS(b)
	if (ea+1)>FEXCESS/(eb+1) {return true}
	return false
}

func sexceed(a *BIG) bool {
	ea:=EXCESS(a)
	if (ea+1)>FEXCESS/(ea+1) {return true}
	return false
}

func ff_pexceed(a *BIG,b *BIG) bool {
	ea:=FF_EXCESS(a)
	eb:=FF_EXCESS(b)
	if (ea+1)>P_FEXCESS/(eb+1) {return true}
	return false
}

func ff_sexceed(a *BIG) bool {
	ea:=FF_EXCESS(a)
	if (ea+1)>P_FEXCESS/(ea+1) {return true}
	return false
}

/* return a*b as DBIG */
func mul(a *BIG,b *BIG) *DBIG {
	c:=NewDBIG()
	carry:= Chunk(0)
//	a.norm()
//	b.norm()

	for i:=0;i<NLEN;i++ {
		carry=0
		for j:=0;j<NLEN;j++ {
			carry,c.w[i+j]=muladd(a.w[i],b.w[j],carry,c.w[i+j])
			//carry=c.muladd(a.w[i],b.w[j],carry,i+j)
		}
		c.w[NLEN+i]=carry
	}
	
	return c
}

/* return a^2 as DBIG */
func sqr(a *BIG) *DBIG {
	c:=NewDBIG()
	carry:= Chunk(0)
//	a.norm()
	for i:=0;i<NLEN;i++ {
		carry=0;
		for j:=i+1;j<NLEN;j++ {
			carry,c.w[i+j]=muladd(2*a.w[i],a.w[j],carry,c.w[i+j])
			//carry=c.muladd(2*a.w[i],a.w[j],carry,i+j)
		}
		c.w[NLEN+i]=carry
	}

	for i:=0;i<NLEN;i++ {
		top,bot:=muladd(a.w[i],a.w[i],0,c.w[2*i])
		c.w[2*i]=bot
		c.w[2*i+1]+=top
		//c.w[2*i+1]+=c.muladd(a.w[i],a.w[i],0,2*i)

	}
	c.norm()
	return c
}

func monty(d* DBIG) *BIG {
	md:=NewBIGints(Modulus)
	carry:=Chunk(0)
	m:=Chunk(0)
	for i:=0;i<NLEN;i++ {
		if (MConst==-1) { 
			m=(-d.w[i])&BMASK
		} else {
			if (MConst==1) {
				m=d.w[i]
			} else {m=(MConst*d.w[i])&BMASK}
		}

		carry=0
		for j:=0;j<NLEN;j++ {
			carry,d.w[i+j]=muladd(m,md.w[j],carry,d.w[i+j])
				//carry=d.muladd(m,md.w[j],carry,i+j)
		}
		d.w[NLEN+i]+=carry
	}

	b:=NewBIG()
	for i:=0;i<NLEN;i++ {
		b.w[i]=d.w[NLEN+i]
	}
	b.norm()
	return b		
}

/* set this[i]+=x*y+c, and return high part */
func muladd(a Chunk,b Chunk,c Chunk,r Chunk) (Chunk,Chunk) {
	x0:=a&HMASK
	x1:=(a>>HBITS)
	y0:=b&HMASK;
	y1:=(b>>HBITS)
	bot:=x0*y0
	top:=x1*y1
	mid:=x0*y1+x1*y0
	x0=mid&HMASK;
	x1=(mid>>HBITS)
	bot+=x0<<HBITS; bot+=c; bot+=r 
	top+=x1;
	carry:=bot>>BASEBITS
	bot&=BMASK
	top+=carry
	return top,bot
}


/********************************************/

/* Set Curve */
/* Don't Modify from here... */

const CHUNK int=64	/* Set word size */

const NOT_SPECIAL int=0
const PSEUDO_MERSENNE int=1
const MONTGOMERY_FRIENDLY int=2
const GENERALISED_MERSENNE int=3
const WEIERSTRASS int=0
const EDWARDS int=1
const MONTGOMERY int=2
const BN_CURVE int=0
const BLS_CURVE int=1

/* ...to here */


/*** Enter Some Field details here  ***/
// Curve 25519
//  const MODBITS uint=255
//  const MOD8 uint=5
//  const BASEBITS uint=56
//  const AES_S uint= 0

// NIST256 or Brainpool
//  const MODBITS uint=256
//  const MOD8 uint=7 
//  const BASEBITS uint=56
//  const AES_S uint= 0

// MF254 
//  const MODBITS uint=254
//  const MOD8 uint=7 
//  const BASEBITS uint=56
//  const AES_S uint= 0

// MS255
//  const MODBITS uint= 255
//  const MOD8 uint= 3
//  const BASEBITS uint=56
//  const AES_S uint= 0

// MF256
//  const MODBITS uint=256 
//  const MOD8 uint=7 
//  const BASEBITS uint=56
//  const AES_S uint= 0

// MS256
//  const MODBITS uint= 256
//  const MOD8 uint= 3
//  const BASEBITS uint=56
//  const AES_S uint= 0

// ANSSI
//  const MODBITS uint= 256
//  const MOD8 uint= 3
//  const BASEBITS uint=56
//  const AES_S uint= 0

// BN254 Curve
const MODBITS uint=254 /* Number of bits in Modulus */
const MOD8 uint=3  /* Modulus mod 8 */
const BASEBITS uint=56
const AES_S uint= 0

// BN454 Curve
//const MODBITS uint=454 /* Number of bits in Modulus */
//const MOD8 uint=3  /* Modulus mod 8 */
//const BASEBITS uint=60
//const AES_S uint= 128

// BLS383 Curve
//const MODBITS uint=383 /* Number of bits in Modulus */
//const MOD8 uint=3  /* Modulus mod 8 */
//const BASEBITS uint=56
//const AES_S uint= 0

// BLS455 Curve
//const MODBITS uint=455 /* Number of bits in Modulus */
//const MOD8 uint=3  /* Modulus mod 8 */
//const BASEBITS uint=60
//const AES_S uint= 128

// HIFIVE Curve
//  const MODBITS uint=336
//  const MOD8 uint=5 
//  const BASEBITS uint=60
//  const AES_S uint= 128

// GOLDILOCKS
//  const MODBITS uint=448
//  const MOD8 uint=7
//  const BASEBITS uint=60
//  const AES_S uint= 0

// NIST384
//  const MODBITS uint=384
//  const MOD8 uint=7
//  const BASEBITS uint=60
//  const AES_S uint= 0

// C41417
//  const MODBITS uint=414
//  const MOD8 uint=7
//  const BASEBITS uint=60	
//  const AES_S uint= 0

// NIST521
//  const MODBITS uint=521
//  const MOD8 uint=7
//  const BASEBITS uint=60
//  const AES_S uint= 0

// BN646 Curve
//  const MODBITS uint=646
//  const MOD8 uint=3
//  const BASEBITS uint=60
//  const AES_S uint= 192

/* RSA/DH modulus length as multiple of BIGBITS */
const FFLEN int=4

/* Don't Modify from here... */
const NLEN int=int((1+((MODBITS-1)/BASEBITS)))
const DNLEN int=2*NLEN
const BMASK Chunk= ((Chunk(1)<<BASEBITS)-1)
const MODBYTES uint=(1+(MODBITS-1)/8)
const HBITS uint=(BASEBITS/2)
const HMASK Chunk= ((Chunk(1)<<HBITS)-1)
const NEXCESS int=(1<<(uint(CHUNK)-BASEBITS-1));
const FEXCESS Chunk=(Chunk(1)<<(BASEBITS*uint(NLEN)-MODBITS))
const OMASK Chunk= ((Chunk(-1))<<(MODBITS%BASEBITS))
const TBITS uint=MODBITS%BASEBITS // Number of active bits in top word 
const TMASK Chunk=(Chunk(1)<<TBITS)-1
const BIGBITS int=int(MODBYTES*8)

/* Finite field support - for RSA, DH etc. */
const FF_BITS int=(BIGBITS*FFLEN) /* Finite Field Size in bits - must be 256.2^n */
const HFLEN int=(FFLEN/2)  /* Useful for half-size RSA private key operations */

const P_MBITS uint=MODBYTES*8
const P_MB uint=(P_MBITS%BASEBITS)
const P_OMASK Chunk=(Chunk(-1)<<(P_MBITS%BASEBITS))
const P_FEXCESS Chunk=(Chunk(1)<<(BASEBITS*uint(NLEN)-P_MBITS))
const P_TBITS uint=(P_MBITS%BASEBITS)

/* ...to here */


// START SPECIFY FIELD DETAILS HERE
//*********************************************************************************
// Curve25519 Modulus 
//  const MODTYPE int=PSEUDO_MERSENNE
//  var Modulus = [...]Chunk {0xFFFFFFFFFFFFED,0xFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFF,0x7FFFFFFF}
//  const MConst Chunk=0x13

// NIST-256 Curve 
//  const MODTYPE int=NOT_SPECIAL
//  var Modulus = [...]Chunk {0xFFFFFFFFFFFFFF,0xFFFFFFFFFF,0x0,0x1000000,0xFFFFFFFF}
//  const MConst Chunk=0x1

// MF254 Modulus
//  const MODTYPE int=MONTGOMERY_FRIENDLY
//  var Modulus= [...]Chunk {0xFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFF,0x3F80FFFF}
//  const MConst Chunk=0x3F810000

// MS255 Modulus
//  const MODTYPE int= 1
//  var Modulus= [...]Chunk {0xFFFFFFFFFFFD03,0xFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFF,0x7FFFFFFF}
//  const MConst Chunk=0x2FD

// MF256 Modulus
//  const MODTYPE int= 2
//  var Modulus= [...]Chunk {0xFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFF,0xFFA7FFFF}
//  const MConst Chunk=0xFFA80000

// MS256 Modulus
//  const MODTYPE int= 1
//  var Modulus= [...]Chunk {0xFFFFFFFFFFFF43,0xFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFF,0xFFFFFFFF}
//  const MConst Chunk=0xBD

// Brainpool
//  const MODTYPE int= NOT_SPECIAL
//  var Modulus= [...]Chunk {0x13481D1F6E5377,0xF623D526202820,0x909D838D726E3B,0xA1EEA9BC3E660A,0xA9FB57DB}
//  const MConst Chunk =0xA75590CEFD89B9

// ANSSI
//const MODTYPE int= 0
//  var Modulus= [...]Chunk{0xFCF353D86E9C03,0xADBCABC8CA6DE8,0xE8CE42435B3961,0xB3AD58F10126D,0xF1FD178C}
//  const MConst Chunk=0x97483A164E1155

// BNCX Curve Modulus
//const MODTYPE int=NOT_SPECIAL
//var Modulus= [...]Chunk {0x6623EF5C1B55B3,0xD6EE18093EE1BE,0x647A6366D3243F,0x8702A0DB0BDDF,0x24000000}
//const MConst Chunk=0x4E205BF9789E85

// HIFIVE Curve
//  const MODTYPE int=PSEUDO_MERSENNE
//  var Modulus= [...]Chunk {0xFFFFFFFFFFFFFFD,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFF}
//  const MConst Chunk=0x3

//GOLDILOCKS
//  const MODTYPE int=GENERALISED_MERSENNE
//  var Modulus= [...]Chunk {0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFEFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFF}
//  const MConst Chunk=0x1

// NIST384 Curve Modulus
//  const MODTYPE int=NOT_SPECIAL
//  var Modulus= [...]Chunk {0xFFFFFFFF,0xFFFFFF000000000,0xFFFFFFFFFFFFEFF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFF}
//  const MConst Chunk=0x100000001

// C41417 Curve Modulus
//  const MODTYPE int=PSEUDO_MERSENNE
//  var Modulus= [...]Chunk {0xFFFFFFFFFFFFFEF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0x3FFFFFFFFFFFFF}
//  const MConst Chunk=0x11

// NIST521
//  const MODTYPE int=PSEUDO_MERSENNE
//  var Modulus= [...]Chunk {0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0x1FFFFFFFFFF}
//  const MConst Chunk=0x1

// BN646
//  const MODTYPE int=NOT_SPECIAL
//  var Modulus= [...]Chunk {0x2406C08404E013,0x240510420138000,0xE01920840000000,0x601B00000901441,0x400000006C0A206,0xD814423414402,0x6C051024000000,0xD8000000000000,0x9006C,0x240000,0x240000000000}
//  const MConst Chunk=0xCE50F5CF5F615E5	


// BN254 Curve
const MODTYPE int=NOT_SPECIAL
var Modulus= [...]Chunk  {0x13,0x13A7,0x80000000086121,0x40000001BA344D,0x25236482}
const MConst Chunk=0x435E50D79435E5

// BN454 Curve
//  const MODTYPE int=NOT_SPECIAL
//  var Modulus= [...]Chunk  {0x4E00000013,0x4E006C4E0000840,0x800014508002508,0x888401D5080091C,0x21C00021D4400DA,0x25B06C01B144009,0xD80000909000,0x240000900}
//  const MConst Chunk=0xF33C46ED79435E5

// BLS383 Curve
//  const MODTYPE int=NOT_SPECIAL
//  var Modulus= [...]Chunk  {0xACAAB52AAD556B,0x1BB01475F75D7A,0xCF73083D5D7520,0x531820F99EB16,0x2C01355A68EA32,0x5C6105C552A785,0x7AC52080A9F7}
//  const MConst Chunk=0xA59AB3B123D0BD

// BLS455 Curve
//  const MODTYPE int=NOT_SPECIAL
//  var Modulus= [...]Chunk  {0xAA00001800002AB,0xC589556B2AA956A,0xB9994ACE86D1BA6,0x3954FCB314B8B3D,0xE3A5B1D56234BD9,0x95B49203003F665,0x57955572AA00E0F,0x555559555}
//  const MConst Chunk=0xB3EF8137F4017FD


// BNT Curve
//  const MODTYPE int=NOT_SPECIAL
//  var Modulus= [...]Chunk {0x9DBBFEEEB4A713,0x555614F464BABE,0x3696F8D5F06E8A,0x6517014EFA0BAB,0x240120DB}
//  const MConst Chunk=0xC5A872D914C4E5

// BNT2 Curve
//  const MODTYPE int=NOT_SPECIAL
//  var Modulus= [...]Chunk {0xB2DC2BB460A48B,0x93E428F0D651E8,0xF3B89D00081CF,0x410F5AADB74E20,0x24000482}
//  const MConst Chunk=0xFE6A47A6505CDD

// START SPECIFY CURVE DETAILS HERE
//*********************************************************************************

// Ed25519 Curve 
//  const CURVETYPE int=EDWARDS
//  const CURVE_A int = -1
//  var CURVE_B = [...]Chunk {0xEB4DCA135978A3,0xA4D4141D8AB75,0x797779E8980070,0x2B6FFE738CC740,0x52036CEE}
//  var CURVE_Order = [...]Chunk {0x12631A5CF5D3ED,0xF9DEA2F79CD658,0x14DE,0x0,0x10000000}
//  var CURVE_Gx = [...]Chunk {0x562D608F25D51A,0xC7609525A7B2C9,0x31FDD6DC5C692C,0xCD6E53FEC0A4E2,0x216936D3}
//  var CURVE_Gy = [...]Chunk {0x66666666666658,0x66666666666666,0x66666666666666,0x66666666666666,0x66666666}

// NIST-256 Curve
//  const CURVETYPE int=WEIERSTRASS
//  const CURVE_A int = -3
//  var CURVE_B = [...]Chunk {0xCE3C3E27D2604B,0x6B0CC53B0F63B,0x55769886BC651D,0xAA3A93E7B3EBBD,0x5AC635D8}
//  var CURVE_Order = [...]Chunk {0xB9CAC2FC632551,0xFAADA7179E84F3,0xFFFFFFFFFFBCE6,0xFFFFFF,0xFFFFFFFF}
//  var CURVE_Gx =[...]Chunk {0xA13945D898C296,0x7D812DEB33A0F4,0xE563A440F27703,0xE12C4247F8BCE6,0x6B17D1F2}
//  var CURVE_Gy =[...]Chunk {0xB6406837BF51F5,0x33576B315ECECB,0x4A7C0F9E162BCE,0xFE1A7F9B8EE7EB,0x4FE342E2}

// MF254 Modulus, Weierstrass Curve w-254-mont
//  const CURVETYPE int= WEIERSTRASS
//  const CURVE_A int = -3
//  var CURVE_B = [...]Chunk {0xFFFFFFFFFFD08D,0xFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFF,0x3F80FFFF}
//  var CURVE_Order=[...]Chunk {0xA419C4AF8DF83F,0x8BEA0DA375C06F,0xFFFFFFFFFFEB81,0xFFFFFFFFFFFFFF,0x3F80FFFF}
//  var CURVE_Gx =[...]Chunk {0x2,0x0,0x0,0x0,0x0}
//  var CURVE_Gy =[...]Chunk {0x65DF37F90D4EBC,0x38E3F8511931AD,0x75BD778AEBDFB7,0x3B2E56014AE15A,0x140E3FD3}

// MF254 Modulus, Edwards Curve ed-254-mont
//  const CURVETYPE int= EDWARDS
//  const CURVE_A int= -1
//  var CURVE_B = [...]Chunk {0x367B,0x0,0x0,0x0,0x0}
//  var CURVE_Order=[...]Chunk {0xF3D3FEC46E98C7,0x306C8BD62FB0EA,0xFFFFFFFFFFEB95,0xFFFFFFFFFFFFFF,0xFE03FFF}
//  var CURVE_Gx =[...]Chunk {0x1,0x0,0x0,0x0,0x0}
//  var CURVE_Gy =[...]Chunk {0x52D0FDAF2701E5,0x9A840E3212187C,0xD502363F4E3632,0xD6A4C335951D00,0x19F0E690}

// MF254 Modulus, Montgomery Curve
//  const CURVETYPE int=MONTGOMERY
//  const CURVE_A int= -55790
//  var CURVE_B = [...]Chunk {0x0,0x0,0x0,0x0,0x0} // not used
//  var CURVE_Order=[...]Chunk {0xF3D3FEC46E98C7,0x306C8BD62FB0EA,0xFFFFFFFFFFEB95,0xFFFFFFFFFFFFFF,0xFE03FFF}
//  var CURVE_Gx =[...]Chunk {0x3,0x0,0x0,0x0,0x0}
//  var CURVE_Gy =[...]Chunk {0x0,0x0,0x0,0x0,0x0} // not used

// MS255 Modulus, Weierstrass Curve
//  const CURVETYPE int= WEIERSTRASS
//  const CURVE_A int= -3
//  var CURVE_B = [...]Chunk {0xFFFFFFFFFFAB46,0xFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFF,0x7FFFFFFF}
//  var CURVE_Order=[...]Chunk {0x8FAC983C594AEB,0x38283AD2B3DFAB,0xFFFFFFFFFF864A,0xFFFFFFFFFFFFFF,0x7FFFFFFF}
//  var CURVE_Gx =[...]Chunk {0x1,0x0,0x0,0x0,0x0};
//  var CURVE_Gy =[...]Chunk {0x33FF6769CB44BA,0xC78CDDFDA60D17,0xF9B2FF7D177DB6,0xEDBA7833921EBF,0x6F7A6AC0}

// MS255 Modulus, Edwards Curve
//  const CURVETYPE int= EDWARDS
//  const CURVE_A int= -1
//  var CURVE_B = [...]Chunk{0xEA97,0x0,0x0,0x0,0x0}
//  var CURVE_Order=[...]Chunk{0x49D1ED0436EB75,0xA785EDA6832EAC,0xFFFFFFFFFFDCF1,0xFFFFFFFFFFFFFF,0x1FFFFFFF}
//  var CURVE_Gx =[...]Chunk{0x4,0x0,0x0,0x0,0x0}
//  var CURVE_Gy =[...]Chunk{0x2A255BD08736A0,0x4B8AED445A45BA,0xDD8E0C47E55291,0x4A7BB545EC254C,0x26CB7853}

// MS255 Modulus, Montgomery Curve
//  const CURVETYPE int=MONTGOMERY
//  const CURVE_A int=-240222
//  var CURVE_B = [...]Chunk {0x0,0x0,0x0,0x0,0x0}
//  var CURVE_Order=[...]Chunk {0x49D1ED0436EB75,0xA785EDA6832EAC,0xFFFFFFFFFFDCF1,0xFFFFFFFFFFFFFF,0x1FFFFFFF}
//  var CURVE_Gx =[...]Chunk {0x4,0x0,0x0,0x0,0x0}
//  var CURVE_Gy =[...]Chunk {0x0,0x0,0x0,0x0,0x0};

// MF256 Modulus, Weierstrass Curve
//  const CURVETYPE int= WEIERSTRASS
//  const CURVE_A int= -3
//  var CURVE_B = [...]Chunk {0x14E6A,0x0,0x0,0x0,0x0}
//  var CURVE_Order=[...]Chunk {0x10C5E1A79857EB,0x7513E6E5074B9D,0xFFFFFFFFFFFC51,0xFFFFFFFFFFFFFF,0xFFA7FFFF}
//  var CURVE_Gx =[...]Chunk {0x1,0x0,0x0,0x0,0x0}
//  var CURVE_Gy =[...]Chunk {0x7954C2B724D2A,0x47EB8D94DC6610,0x26123DAE289569,0xBE1808CE7BABBA,0x20887C87}

// MF256, Edwards Curve
//  const CURVETYPE int= EDWARDS
//  const CURVE_A int= -1
//  var CURVE_B = [...]Chunk {0x350A,0x0,0x0,0x0,0x0}
//  var CURVE_Order=[...]Chunk {0xD92EDED8EC7BAB,0xBBAFB86733C966,0xFFFFFFFFFFB154,0xFFFFFFFFFFFFFF,0x3FE9FFFF}
//  var CURVE_Gx =[...]Chunk {0x1,0x0,0x0,0x0,0x0}
//  var CURVE_Gy =[...]Chunk {0xEAA722F2F3C908,0x5E648DFEA68D7D,0xF3DB2C1AACA0C0,0xF8CC4D5AEAEBEE,0xDAD8D4F8}

// MF256 Modulus, Montgomery Curve
//  const CURVETYPE int=MONTGOMERY
//  const CURVE_A int= -54314
//  var CURVE_B = [...]Chunk {0x0,0x0,0x0,0x0,0x0} // not used
//  var CURVE_Order=[...]Chunk {0xD92EDED8EC7BAB,0xBBAFB86733C966,0xFFFFFFFFFFB154,0xFFFFFFFFFFFFFF,0x3FE9FFFF}
//  var CURVE_Gx =[...]Chunk {0x8,0x0,0x0,0x0,0x0}
//  var CURVE_Gy =[...]Chunk {0x0,0x0,0x0,0x0,0x0} // not used

// MS256, Weierstrass Curve
//  const CURVETYPE int= WEIERSTRASS
//  const CURVE_A int= -3
//  var CURVE_B = [...]Chunk {0x25581,0x0,0x0,0x0,0x0}
//  var CURVE_Order=[...]Chunk {0xAB20294751A825,0x8275EA265C6020,0xFFFFFFFFFFE43C,0xFFFFFFFFFFFFFF,0xFFFFFFFF}
//  var CURVE_Gx =[...]Chunk {0x1,0x0,0x0,0x0,0x0}
//  var CURVE_Gy =[...]Chunk {0xF46306C2B56C77,0x2F9375894EC10B,0x6CCEEEDD6BD02C,0xC1E466D7FC82C9,0x696F1853}

// MS256, Edwards Curve
//  const CURVETYPE int= EDWARDS
//  const CURVE_A int= -1
//  var CURVE_B = [...]Chunk {0x3BEE,0x0,0x0,0x0,0x0}
//  var CURVE_Order=[...]Chunk {0xB84E6F1122B4AD,0xA55AD0A6BC64E5,0xFFFFFFFFFFBE6A,0xFFFFFFFFFFFFFF,0x3FFFFFFF}
//  var CURVE_Gx =[...]Chunk {0xD,0x0,0x0,0x0,0x0}
//  var CURVE_Gy =[...]Chunk {0x7F6FB5331CADBA,0x6D63824D303F70,0xB39FA046BFBE2A,0x2A1276DBA3D330,0x7D0AB41E}

// MS256 Modulus, Montgomery Curve
//  const CURVETYPE int=MONTGOMERY
//  const CURVE_A int=-61370
//  var CURVE_B = [...]Chunk  {0x0,0x0,0x0,0x0,0x0} // not used
//  var CURVE_Order= [...]Chunk {0xB84E6F1122B4AD,0xA55AD0A6BC64E5,0xFFFFFFFFFFBE6A,0xFFFFFFFFFFFFFF,0x3FFFFFFF}
//  var CURVE_Gx = [...]Chunk {0xb,0x0,0x0,0x0,0x0}
//  var CURVE_Gy = [...]Chunk {0x0,0x0,0x0,0x0,0x0} // not used

// Brainpool
//  const CURVETYPE int= WEIERSTRASS
//  const CURVE_A int= -3
//  var CURVE_B = [...]Chunk  {0xE58101FEE92B04,0xEBC4AF2F49256A,0x733D0B76B7BF93,0x30D84EA4FE66A7,0x662C61C4}
//  var CURVE_Order= [...]Chunk {0x1E0E82974856A7,0x7AA3B561A6F790,0x909D838D718C39,0xA1EEA9BC3E660A,0xA9FB57DB}
//  var CURVE_Gx = [...]Chunk {0xA191562E1305F4,0x42C47AAFBC2B79,0xB23A656149AFA1,0xC1CFE7B7732213,0xA3E8EB3C}
//  var CURVE_Gy = [...]Chunk {0xABE8F35B25C9BE,0xB6DE39D027001D,0xE14644417E69BC,0x3439C56D7F7B22,0x2D996C82}

// ANSSI
//  const CURVETYPE int= WEIERSTRASS
//  const CURVE_A int= -3
//  var CURVE_B = [...]Chunk  {0x75ED967B7BB73F,0xC9AE4B1A18030,0x754A44C00FDFEC,0x5428A9300D4ABA,0xEE353FCA}
//  var CURVE_Order=[...]Chunk  {0xFDD459C6D655E1,0x67E140D2BF941F,0xE8CE42435B53DC,0xB3AD58F10126D,0xF1FD178C}
//  var CURVE_Gx =[...]Chunk  {0xC97A2DD98F5CFF,0xD2DCAF98B70164,0x4749D423958C27,0x56C139EB31183D,0xB6B3D4C3}
//  var CURVE_Gy =[...]Chunk  {0x115A1554062CFB,0xC307E8E4C9E183,0xF0F3ECEF8C2701,0xC8B204911F9271,0x6142E0F7}

// HIFIVE
//  const CURVETYPE int= EDWARDS
//  const CURVE_A int= 1
//  var CURVE_B = [...]Chunk  {0x2B67,0x0,0x0,0x0,0x0,0x0}
//  var CURVE_Order=[...]Chunk  {0xB2F95973E9FA805,0xC0BD6B87F93BAA7,0x71415FA9850,0x0,0x0,0x200000000}
//  var CURVE_Gx =[...]Chunk  {0xC,0x0,0x0,0x0,0x0,0x0}
//  var CURVE_Gy =[...]Chunk  {0x2BEC68505FE8632,0x5D5650CA0365DB1,0x3811C7EF435B6DB,0x7853D1B14B46C,0x56502E18E1C161D,0xC0DC616B}

// GOLDILOCKS
//  const CURVETYPE int= EDWARDS
//  const CURVE_A int= 1
//  var CURVE_B = [...]Chunk  {0xFFFFFFFFFFF6756,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFEFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFF}
//  var CURVE_Order=[...]Chunk  {0x378C292AB5844F3,0x6CC2728DC58F552,0xEDB49AED6369021,0xFFFF7CCA23E9C44,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0x3FFFFFF}
//  var CURVE_Gx =[...]Chunk  {0x555555555555555,0x555555555555555,0x555555555555555,0xAAA955555555555,0xAAAAAAAAAAAAAAA,0xAAAAAAAAAAAAAAA,0xAAAAAAAAAAAAAAA,0xAAAAAAA}
//  var CURVE_Gy =[...]Chunk  {0xAEAFBCDEA9386ED,0xBCB2BED1CDA06BD,0x565833A2A3098BB,0x6D728AD8C4B80D6,0x7A035884DD7B7E3,0x205086C2B0036ED,0x34AD7048DB359D6,0xAE05E96}

// NIST384
//  const CURVETYPE int= WEIERSTRASS
//  const CURVE_A int= -3
//  var CURVE_B = [...]Chunk  {0xA85C8EDD3EC2AEF,0x56398D8A2ED19D2,0x4088F5013875AC6,0x9C6EFE814112031,0x56BE3F82D19181D,0xA7E23EE7E4988E0,0xB3312F}
//  var CURVE_Order=[...]Chunk  {0xCEC196ACCC52973,0x1A0DB248B0A77AE,0x34D81F4372DDF58,0xFFFFFFFFFFFFC76,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFF}
//  var CURVE_Gx =[...]Chunk  {0xA545E3872760AB7,0x2F25DBF55296C3,0x741E082542A3855,0x3B628BA79B9859F,0x71EF320AD746E1D,0x22BE8B05378EB1C,0xAA87CA}
//  var CURVE_Gy =[...]Chunk  {0xA431D7C90EA0E5F,0x60B1CE1D7E819D7,0xA3113B5F0B8C00A,0x1DBD289A147CE9D,0x8BF9292DC29F8F4,0x4A96262C6F5D9E9,0x3617DE}

// C41417
//  const CURVETYPE int= EDWARDS
//  const CURVE_A int= 1
//  var CURVE_B = [...]Chunk  {0xE21,0x0,0x0,0x0,0x0,0x0,0x0}
//  var CURVE_Order=[...]Chunk  {0xB0E71A5E106AF79,0x1C0338AD63CF181,0x414CF706022B36F,0xFFFFFFFFEB3CC92,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0x7FFFFFFFFFFFF}
//  var CURVE_Gx =[...]Chunk  {0x4FD3812F3CBC595,0x1A73FAA8537C64C,0x4AB4D6D6BA11130,0x3EC7F57FF35498A,0xE5FCD46369F44C0,0x300218C0631C326,0x1A334905141443}
//  var CURVE_Gy =[...]Chunk  {0x22,0x0,0x0,0x0,0x0,0x0,0x0}

// NIST521
//  const CURVETYPE int= WEIERSTRASS
//  const CURVE_A int= -3
//  var CURVE_B = [...]Chunk  {0xF451FD46B503F00,0x73DF883D2C34F1E,0x2C0BD3BB1BF0735,0x3951EC7E937B165,0x9918EF109E15619,0x5B99B315F3B8B48,0xB68540EEA2DA72,0x8E1C9A1F929A21A,0x51953EB961}
//  var CURVE_Order=[...]Chunk  {0xB6FB71E91386409,0xB5C9B8899C47AEB,0xC0148F709A5D03B,0x8783BF2F966B7FC,0xFFFFFFFFFFA5186,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFF,0x1FFFFFFFFFF}
//  var CURVE_Gx =[...]Chunk  {0x97E7E31C2E5BD66,0x48B3C1856A429BF,0xDC127A2FFA8DE33,0x5E77EFE75928FE1,0xF606B4D3DBAA14B,0x39053FB521F828A,0x62395B4429C6481,0x404E9CD9E3ECB6,0xC6858E06B7}
//  var CURVE_Gy =[...]Chunk  {0x8BE94769FD16650,0x3C7086A272C2408,0xB9013FAD076135,0x72995EF42640C55,0xD17273E662C97EE,0x49579B446817AFB,0x42C7D1BD998F544,0x9A3BC0045C8A5FB,0x11839296A78}


// BN646 Curve
/*
const CURVETYPE int=WEIERSTRASS
const CURVE_PAIRING_TYPE int=BN_CURVE
const CURVE_A int= 0

var CURVE_Order=[...]Chunk {0x2406C07E04200D,0x2405103F0108000,0xD418607E0000000,0x301B00000901441,0x400000006C0A206,0xD814422E14402,0x6C051024000000,0xD8000000000000,0x9006C,0x240000,0x240000000000}
var CURVE_B = [...]Chunk {0x2,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0}
var CURVE_Cof = [...]Chunk {0x1,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0}
var CURVE_Gx =[...]Chunk {0x2406C08404E012,0x240510420138000,0xE01920840000000,0x601B00000901441,0x400000006C0A206,0xD814423414402,0x6C051024000000,0xD8000000000000,0x9006C,0x240000,0x240000000000}
var CURVE_Gy =[...]Chunk {0x1,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0}

var CURVE_Bnx=[...]Chunk {0x1001,0x4000,0x10000000000,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0}
var CURVE_Cru=[...]Chunk {0x1202401B007,0xD812006C000,0x480510240000000,0x200480000000360,0x1B01,0x3602403600,0xD800000000,0x0,0x12,0x0,0x0}
var CURVE_Fra=[...]Chunk {0x76EAD944929A14,0xB7E0A0BE10CDF74,0x4FB8A8A2B93166B,0xECDFC4B0F037D9,0xC27307962815598,0xCF5EF558D2135D8,0x8CE651FD85AF9C3,0x80AF6D02A45219F,0x69B8147979A41A4,0x9577C152A374C5B,0x9FF28B3A478}
var CURVE_Frb=[...]Chunk {0xFAD1BE73F7245FF,0x6C246F83F06A08B,0x906077E146CE994,0x514D03B4F9FDC68,0x7D8CF86A43F4C6E,0x31791EE96200E29,0x79D9FF04BA5063C,0x8CD092FD5BADE60,0x9647EB8686EBEC7,0x6A883EAD5ECB3A4,0x1A00D74C5B87}
var CURVE_Pxa=[...]Chunk {0x4233F273CCC5E10,0x6408117FB1B1FFA,0xA7978AC166486AA,0xDA6417BDCFDC1D0,0xDCE981D68FA7F5C,0x4169ED790F45048,0xBA06CCE894F26BE,0xB0C98247FD18141,0x30CE35212F353A6,0xFF9A1B9162B0B9E,0xFD835F078BF}
var CURVE_Pxb=[...]Chunk {0x837F879267F4BC0,0x443F1581FA770ED,0x8FBA0763CB82027,0xA8F7E5DE4945F11,0x780AAE5D219F786,0xF3BB745205342E9,0xCC0A34FF057013C,0xA6DFAF3E689709E,0xD69FBD8D12B6B79,0x6A1715D9469215D,0x1764FA509C41}
var CURVE_Pya=[...]Chunk {0xC570D1DCC3FB414,0xD0ABFE7F36161E4,0xABA2F61496C849A,0x4E4A05030CD4F3F,0x70F7CECD5CEF83,0x9D4711CBF491613,0xDF8011EA770418E,0x56548E514EC94EC,0xC9E853DFF35EE42,0x5481ABBAAD13633,0x78D8A63783F}
var CURVE_Pyb=[...]Chunk {0xAFE3E3DB33908AD,0x7CD4290A506594C,0xFA043D85973EC3F,0xFAC7A642BB5E628,0x71A36A52DA82EE2,0x12C2231EDAE0C76,0x994DF2771091B48,0xCFFF7637B76831C,0xD463BD03DA14916,0x1074F23C580C40,0x12188D99546B}
var CURVE_W=[2][NLEN]Chunk {{0x6008003,0x30020000,0xC0080060000000,0x300000000000000,0x0,0x600000,0x0,0x0,0x0,0x0,0x0},{0x2001,0x8000,0x20000000000,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0}}
var CURVE_SB=[2][2][NLEN]Chunk {{{0x600A004,0x30028000,0xC00A0060000000,0x300000000000000,0x0,0x600000,0x0,0x0,0x0,0x0,0x0},{0x2001,0x8000,0x20000000000,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0}},{{0x2001,0x8000,0x20000000000,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x2406C07803A00A,0x2405103C00E8000,0xC817E0780000000,0x1B00000901441,0x400000006C0A206,0xD814422814402,0x6C051024000000,0xD8000000000000,0x9006C,0x240000,0x240000000000}}}
var CURVE_WB=[4][NLEN]Chunk {{0x2001000,0x10004000,0x40010020000000,0x100000000000000,0x0,0x200000,0x0,0x0,0x0,0x0,0x0},{0xC01C015005,0x900E0054000,0x3803901C0000000,0xE00300000000240,0x1200,0x2401C02400,0x9000000000,0x0,0xC,0x0,0x0},{0x600E00B003,0x4807002C000,0x1C01D00E0000000,0x700180000000120,0x900,0x1200E01200,0x4800000000,0x0,0x6,0x0,0x0},{0x2003001,0x1000C000,0x40030020000000,0x100000000000000,0x0,0x200000,0x0,0x0,0x0,0x0,0x0}}
var CURVE_BB=[4][4][NLEN]Chunk {{{0x2406C07E04100D,0x2405103F0104000,0xD418507E0000000,0x301B00000901441,0x400000006C0A206,0xD814422E14402,0x6C051024000000,0xD8000000000000,0x9006C,0x240000,0x240000000000},{0x2406C07E04100C,0x2405103F0104000,0xD418507E0000000,0x301B00000901441,0x400000006C0A206,0xD814422E14402,0x6C051024000000,0xD8000000000000,0x9006C,0x240000,0x240000000000},{0x2406C07E04100C,0x2405103F0104000,0xD418507E0000000,0x301B00000901441,0x400000006C0A206,0xD814422E14402,0x6C051024000000,0xD8000000000000,0x9006C,0x240000,0x240000000000},{0x2002,0x8000,0x20000000000,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0}},{{0x2001,0x8000,0x20000000000,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x2406C07E04100C,0x2405103F0104000,0xD418507E0000000,0x301B00000901441,0x400000006C0A206,0xD814422E14402,0x6C051024000000,0xD8000000000000,0x9006C,0x240000,0x240000000000},{0x2406C07E04100D,0x2405103F0104000,0xD418507E0000000,0x301B00000901441,0x400000006C0A206,0xD814422E14402,0x6C051024000000,0xD8000000000000,0x9006C,0x240000,0x240000000000},{0x2406C07E04100C,0x2405103F0104000,0xD418507E0000000,0x301B00000901441,0x400000006C0A206,0xD814422E14402,0x6C051024000000,0xD8000000000000,0x9006C,0x240000,0x240000000000}},{{0x2002,0x8000,0x20000000000,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x2001,0x8000,0x20000000000,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x2001,0x8000,0x20000000000,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x2001,0x8000,0x20000000000,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0}},{{0x1002,0x4000,0x10000000000,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x4002,0x10000,0x40000000000,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x2406C07E04000A,0x2405103F0100000,0xD418407E0000000,0x301B00000901441,0x400000006C0A206,0xD814422E14402,0x6C051024000000,0xD8000000000000,0x9006C,0x240000,0x240000000000},{0x1002,0x4000,0x10000000000,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0}}}

const USE_GLV bool=true
const USE_GS_G2 bool=true
const USE_GS_GT bool=true	
const GT_STRONG bool=false
*/
// BNCX Curve 
/*
const CURVETYPE int=WEIERSTRASS
const CURVE_PAIRING_TYPE int=BN_CURVE
const CURVE_A int= 0
var CURVE_B = [...]Chunk {0x2,0x0,0x0,0x0,0x0}
var CURVE_Cof = [...]Chunk {0x1,0x0,0x0,0x0,0x0}

var CURVE_Order=[...]Chunk {0x11C0A636EB1F6D,0xD6EE0CC906CEBE,0x647A6366D2C43F,0x8702A0DB0BDDF,0x24000000}
var CURVE_Bnx=[...]Chunk {0x3C012B1,0x40,0x0,0x0,0x0}
var CURVE_Cru=[...]Chunk {0xE0931794235C97,0xDF6471EF875631,0xCA83F1440BD,0x480000,0x0}
var CURVE_Fra=[...]Chunk {0xD9083355C80EA3,0x7326F173F8215B,0x8AACA718986867,0xA63A0164AFE18B,0x1359082F}
var CURVE_Frb=[...]Chunk {0x8D1BBC06534710,0x63C7269546C062,0xD9CDBC4E3ABBD8,0x623628A900DC53,0x10A6F7D0}
var CURVE_Pxa=[...]Chunk {0x851CEEE4D2EC74,0x85BFA03E2726C0,0xF5C34BBB907C,0x7053B256358B25,0x19682D2C}
var CURVE_Pxb=[...]Chunk {0xA58E8B2E29CFE1,0x97B0C209C30F47,0x37A8E99743F81B,0x3E19F64AA011C9,0x1466B9EC}
var CURVE_Pya=[...]Chunk {0xFBFCEBCF0BE09F,0xB33D847EC1B30C,0x157DAEE2096361,0x72332B8DD81E22,0xA79EDD9}
var CURVE_Pyb=[...]Chunk {0x904B228898EE9D,0x4EA569D2EDEBED,0x512D8D3461C286,0xECC4C09035C6E4,0x6160C39}
var CURVE_Gx =[...]Chunk {0x6623EF5C1B55B2,0xD6EE18093EE1BE,0x647A6366D3243F,0x8702A0DB0BDDF,0x24000000}
var CURVE_Gy =[...]Chunk {0x1,0x0,0x0,0x0,0x0}
var CURVE_W=[2][5]Chunk{{0x546349162FEB83,0xB40381200,0x6000,0x0,0x0},{0x7802561,0x80,0x0,0x0,0x0}}
var CURVE_SB=[2][2][5]Chunk {{{0x5463491DB010E4,0xB40381280,0x6000,0x0,0x0},{0x7802561,0x80,0x0,0x0,0x0}},{{0x7802561,0x80,0x0,0x0,0x0},{0xBD5D5D20BB33EA,0xD6EE0188CEBCBD,0x647A6366D2643F,0x8702A0DB0BDDF,0x24000000}}}
var CURVE_WB=[4][5]Chunk {{0x1C2118567A84B0,0x3C012B040,0x2000,0x0,0x0},{0xCDF995BE220475,0x94EDA8CA7F9A36,0x8702A0DC07E,0x300000,0x0},{0x66FCCAE0F10B93,0x4A76D4653FCD3B,0x4381506E03F,0x180000,0x0},{0x1C21185DFAAA11,0x3C012B0C0,0x2000,0x0,0x0}}
var CURVE_BB=[4][4][5]Chunk {{{0x11C0A6332B0CBD,0xD6EE0CC906CE7E,0x647A6366D2C43F,0x8702A0DB0BDDF,0x24000000},{0x11C0A6332B0CBC,0xD6EE0CC906CE7E,0x647A6366D2C43F,0x8702A0DB0BDDF,0x24000000},{0x11C0A6332B0CBC,0xD6EE0CC906CE7E,0x647A6366D2C43F,0x8702A0DB0BDDF,0x24000000},{0x7802562,0x80,0x0,0x0,0x0}},{{0x7802561,0x80,0x0,0x0,0x0},{0x11C0A6332B0CBC,0xD6EE0CC906CE7E,0x647A6366D2C43F,0x8702A0DB0BDDF,0x24000000},{0x11C0A6332B0CBD,0xD6EE0CC906CE7E,0x647A6366D2C43F,0x8702A0DB0BDDF,0x24000000},{0x11C0A6332B0CBC,0xD6EE0CC906CE7E,0x647A6366D2C43F,0x8702A0DB0BDDF,0x24000000}},{{0x7802562,0x80,0x0,0x0,0x0},{0x7802561,0x80,0x0,0x0,0x0},{0x7802561,0x80,0x0,0x0,0x0},{0x7802561,0x80,0x0,0x0,0x0}},{{0x3C012B2,0x40,0x0,0x0,0x0},{0xF004AC2,0x100,0x0,0x0,0x0},{0x11C0A62F6AFA0A,0xD6EE0CC906CE3E,0x647A6366D2C43F,0x8702A0DB0BDDF,0x24000000},{0x3C012B2,0x40,0x0,0x0,0x0}}}

const USE_GLV bool=true
const USE_GS_G2 bool=true
const USE_GS_GT bool=true	
const GT_STRONG bool=true

*/
// BN254 Curve
const CURVETYPE int=WEIERSTRASS
const CURVE_PAIRING_TYPE int=BN_CURVE
const CURVE_A int= 0
var CURVE_B = [...]Chunk {0x2,0x0,0x0,0x0,0x0}
var CURVE_Cof = [...]Chunk {0x1,0x0,0x0,0x0,0x0}
var CURVE_Order=[...]Chunk {0xD,0x800000000010A1,0x8000000007FF9F,0x40000001BA344D,0x25236482}
var CURVE_Bnx=[...]Chunk {0x80000000000001,0x40,0x0,0x0,0x0}
var CURVE_Cru=[...]Chunk {0x80000000000007,0x6CD,0x40000000024909,0x49B362,0x0}
var CURVE_Fra=[...]Chunk {0x7DE6C06F2A6DE9,0x74924D3F77C2E1,0x50A846953F8509,0x212E7C8CB6499B,0x1B377619}
var CURVE_Frb=[...]Chunk {0x82193F90D5922A,0x8B6DB2C08850C5,0x2F57B96AC8DC17,0x1ED1837503EAB2,0x9EBEE69}
var CURVE_Pxa=[...]Chunk {0xEE4224C803FB2B,0x8BBB4898BF0D91,0x7E8C61EDB6A464,0x519EB62FEB8D8C,0x61A10BB}
var CURVE_Pxb=[...]Chunk {0x8C34C1E7D54CF3,0x746BAE3784B70D,0x8C5982AA5B1F4D,0xBA737833310AA7,0x516AAF9}
var CURVE_Pya=[...]Chunk {0xF0E07891CD2B9A,0xAE6BDBE09BD19,0x96698C822329BD,0x6BAF93439A90E0,0x21897A0}
var CURVE_Pyb=[...]Chunk {0x2D1AEC6B3ACE9B,0x6FFD739C9578A,0x56F5F38D37B090,0x7C8B15268F6D44,0xEBB2B0E}
var CURVE_Gx =[...]Chunk {0x12,0x13A7,0x80000000086121,0x40000001BA344D,0x25236482}
var CURVE_Gy =[...]Chunk {0x1,0x0,0x0,0x0,0x0}
var CURVE_W=[2][5]Chunk {{0x3,0x80000000000204,0x6181,0x0,0x0},{0x1,0x81,0x0,0x0,0x0}}
var CURVE_SB=[2][2][5]Chunk {{{0x4,0x80000000000285,0x6181,0x0,0x0},{0x1,0x81,0x0,0x0,0x0}},{{0x1,0x81,0x0,0x0,0x0},{0xA,0xE9D,0x80000000079E1E,0x40000001BA344D,0x25236482}}}
var CURVE_WB=[4][5]Chunk {{0x80000000000000,0x80000000000040,0x2080,0x0,0x0},{0x80000000000005,0x54A,0x8000000001C707,0x312241,0x0},{0x80000000000003,0x800000000002C5,0xC000000000E383,0x189120,0x0},{0x80000000000001,0x800000000000C1,0x2080,0x0,0x0}}
var CURVE_BB=[4][4][5]Chunk {{{0x8000000000000D,0x80000000001060,0x8000000007FF9F,0x40000001BA344D,0x25236482},{0x8000000000000C,0x80000000001060,0x8000000007FF9F,0x40000001BA344D,0x25236482},{0x8000000000000C,0x80000000001060,0x8000000007FF9F,0x40000001BA344D,0x25236482},{0x2,0x81,0x0,0x0,0x0}},{{0x1,0x81,0x0,0x0,0x0},{0x8000000000000C,0x80000000001060,0x8000000007FF9F,0x40000001BA344D,0x25236482},{0x8000000000000D,0x80000000001060,0x8000000007FF9F,0x40000001BA344D,0x25236482},{0x8000000000000C,0x80000000001060,0x8000000007FF9F,0x40000001BA344D,0x25236482}},{{0x2,0x81,0x0,0x0,0x0},{0x1,0x81,0x0,0x0,0x0},{0x1,0x81,0x0,0x0,0x0},{0x1,0x81,0x0,0x0,0x0}},{{0x80000000000002,0x40,0x0,0x0,0x0},{0x2,0x102,0x0,0x0,0x0},{0xA,0x80000000001020,0x8000000007FF9F,0x40000001BA344D,0x25236482},{0x80000000000002,0x40,0x0,0x0,0x0}}}

const USE_GLV bool=true
const USE_GS_G2 bool=true
const USE_GS_GT bool=true	
const GT_STRONG bool=false

// BN454 Curve
/*
const CURVETYPE int=WEIERSTRASS
const CURVE_PAIRING_TYPE int=BN_CURVE
const CURVE_A int= 0

var CURVE_Order=[...]Chunk {0x420000000D,0x42006C4200007E0,0x2000144FC0024FC,0x887E01D4FC0091C,0x21C00021D4400DA,0x25B06C01B144009,0xD80000909000,0x240000900};
var CURVE_B = [...]Chunk {0x2,0x0,0x0,0x0,0x0,0x0,0x0,0x0};
var CURVE_Cof = [...]Chunk {0x1,0x0,0x0,0x0,0x0,0x0,0x0,0x0};
var CURVE_Gx =[...]Chunk {0x4E00000012,0x4E006C4E0000840,0x800014508002508,0x888401D5080091C,0x21C00021D4400DA,0x25B06C01B144009,0xD80000909000,0x240000900};
var CURVE_Gy =[...]Chunk {0x1,0x0,0x0,0x0,0x0,0x0,0x0,0x0};

var CURVE_Bnx=[...]Chunk {0x100000001,0x10000010000000,0x0,0x0,0x0,0x0,0x0,0x0};
var CURVE_Cru=[...]Chunk {0x1B00000007,0x1B00121B0000240,0xA00003648000048,0x6C2400364800005,0x36000012360000,0x12000036000,0x0,0x0};
var CURVE_Fra=[...]Chunk {0x302DC9FD573B0EC,0x7AEEE6C3A90B9B8,0xD06B734A02B17EC,0xDBD9562D6A87D,0x929326ECAFCB5A2,0x32139C5D6DDCE26,0x7A687EBE1CCD92B,0x12E79EC72};
var CURVE_Frb=[...]Chunk {0xCFD23AE2A8C4F27,0xD311858A56F4E87,0xAF94A1067D50D1B,0x87A8287EDA9609E,0x8F2CD9352474B38,0xF39CCFA443671E2,0x85A50141EC3B6D4,0x111861C8D};
var CURVE_Pxa=[...]Chunk {0xFBA3BAC20F02B14,0x6BC30BEF0CC2BB1,0xFE2DA6B5FCEFFF3,0x3682EDFC1FE1E7E,0xAD66CC979483454,0xE163BB5D98CBBA1,0xD55083668F14AAF,0x1403D4F80};
var CURVE_Pxb=[...]Chunk {0x9C06D0EC45BF266,0x365ACBB7823C36E,0xD2E4624790DBC52,0xE0803BE6F2F769F,0x7BA9E2AA951971D,0x25841737EBE5665,0x142E8014B1E40D5,0x15952D69E};
var CURVE_Pya=[...]Chunk {0x939AB6BBE1B15C7,0xDF26024555C6CB4,0x671C46519474FB5,0xBAA277A8FC5DD54,0xB00871EE82F2FAA,0xFE0A26E1C64C3A,0x7ABC6DB12C5287A,0x1D67A4004};
var CURVE_Pyb=[...]Chunk {0xC10E6FB60571D60,0xE9271847B128D8E,0xBE70ED2753A65C7,0x75C2604B7433768,0x4F0B46811003851,0xD91F0CA48ABC5EA,0x842C9ADF53F903E,0x976E2360};
var CURVE_W=[2][8]Chunk {{0x800000003,0x80000080000060,0x60000000C00000C,0x600000C00000,0x0,0x0,0x0,0x0},{0x200000001,0x20000020000000,0x0,0x0,0x0,0x0,0x0,0x0}};
var CURVE_SB=[2][2][8]Chunk {{{0xA00000004,0xA00000A0000060,0x60000000C00000C,0x600000C00000,0x0,0x0,0x0,0x0},{0x200000001,0x20000020000000,0x0,0x0,0x0,0x0,0x0,0x0}},{{0x200000001,0x20000020000000,0x0,0x0,0x0,0x0,0x0,0x0},{0x3A0000000A,0x3A006C3A0000780,0xC000144F00024F0,0x887801D4F00091B,0x21C00021D4400DA,0x25B06C01B144009,0xD80000909000,0x240000900}}};
var CURVE_WB=[4][8]Chunk {{0x100000000,0x10000010000020,0x200000004000004,0x200000400000,0x0,0x0,0x0,0x0},{0x1500000005,0x15000C1500001C0,0x2438000038,0x481C00243800004,0x2400000C240000,0xC000024000,0x0,0x0},{0xB00000003,0xB00060B00000E0,0x121C00001C,0x240E00121C00002,0x12000006120000,0x6000012000,0x0,0x0},{0x300000001,0x30000030000020,0x200000004000004,0x200000400000,0x0,0x0,0x0,0x0}};
var CURVE_BB=[4][4][8]Chunk {{{0x410000000D,0x41006C4100007E0,0x2000144FC0024FC,0x887E01D4FC0091C,0x21C00021D4400DA,0x25B06C01B144009,0xD80000909000,0x240000900},{0x410000000C,0x41006C4100007E0,0x2000144FC0024FC,0x887E01D4FC0091C,0x21C00021D4400DA,0x25B06C01B144009,0xD80000909000,0x240000900},{0x410000000C,0x41006C4100007E0,0x2000144FC0024FC,0x887E01D4FC0091C,0x21C00021D4400DA,0x25B06C01B144009,0xD80000909000,0x240000900},{0x200000002,0x20000020000000,0x0,0x0,0x0,0x0,0x0,0x0}},{{0x200000001,0x20000020000000,0x0,0x0,0x0,0x0,0x0,0x0},{0x410000000C,0x41006C4100007E0,0x2000144FC0024FC,0x887E01D4FC0091C,0x21C00021D4400DA,0x25B06C01B144009,0xD80000909000,0x240000900},{0x410000000D,0x41006C4100007E0,0x2000144FC0024FC,0x887E01D4FC0091C,0x21C00021D4400DA,0x25B06C01B144009,0xD80000909000,0x240000900},{0x410000000C,0x41006C4100007E0,0x2000144FC0024FC,0x887E01D4FC0091C,0x21C00021D4400DA,0x25B06C01B144009,0xD80000909000,0x240000900}},{{0x200000002,0x20000020000000,0x0,0x0,0x0,0x0,0x0,0x0},{0x200000001,0x20000020000000,0x0,0x0,0x0,0x0,0x0,0x0},{0x200000001,0x20000020000000,0x0,0x0,0x0,0x0,0x0,0x0},{0x200000001,0x20000020000000,0x0,0x0,0x0,0x0,0x0,0x0}},{{0x100000002,0x10000010000000,0x0,0x0,0x0,0x0,0x0,0x0},{0x400000002,0x40000040000000,0x0,0x0,0x0,0x0,0x0,0x0},{0x400000000A,0x40006C4000007E0,0x2000144FC0024FC,0x887E01D4FC0091C,0x21C00021D4400DA,0x25B06C01B144009,0xD80000909000,0x240000900},{0x100000002,0x10000010000000,0x0,0x0,0x0,0x0,0x0,0x0}}};

const USE_GLV bool=true
const USE_GS_G2 bool=true
const USE_GS_GT bool=true	
const GT_STRONG bool=false
*/


// BLS383 Curve
/*
const CURVETYPE int=WEIERSTRASS
const CURVE_PAIRING_TYPE int=BLS_CURVE
const CURVE_A int= 0

var CURVE_Order=[...]Chunk {0xFFF80000FFF001,0xBFDE0070FE7800,0x3000049C5EDF1C,0xC40007F910007A,0x14641004C,0x0,0x0};
var CURVE_B = [...]Chunk {0x9,0x0,0x0,0x0,0x0,0x0,0x0};
var CURVE_Cof = [...]Chunk {0x2A00000052B,0x5560AAAAAB2CA0,0x6055,0x0,0x0,0x0,0x0};
var CURVE_Gx =[...]Chunk {0xD59B348D10786B,0x3477C0E3F54AD0,0xBF25B734578B9B,0x4F6AC007BB6F65,0xEFD5830FF57E9C,0xADB9F88FB6EC02,0xB08CEE4BC98};
var CURVE_Gy =[...]Chunk {0x5DA023D145DDB,0x13F518C5FEF7CC,0x56EC3462B2A66F,0x96F3019C7A925F,0x9061047981223E,0x4810AD8F5BE59,0x1F3909337671};

var CURVE_Bnx=[...]Chunk {0x1000000040,0x110,0x0,0x0,0x0,0x0,0x0};
var CURVE_Cru=[...]Chunk {0xA3AAC4EDA155A9,0xDF2FE8761E5E3D,0xBCDFAADE632625,0x5123128D3035A6,0xDBF3A2BBEAD683,0x5C5FAB20424190,0x7AC52080A9F7};
var CURVE_Fra=[...]Chunk {0x2BA59A92B4508B,0x63DB7A06EEF343,0x40341CB1DFBC74,0x1639E9D32D55D3,0xB19B3F05CC36D4,0xF323EE4D86AB98,0x5A5FB198672};
var CURVE_Frb=[...]Chunk {0x81051A97F904E0,0xB7D49A6F086A37,0x8F3EEB8B7DB8AB,0xEEF7983C6C9543,0x7A65F6549CB35D,0x693D1777CBFBEC,0x751F25672384};
var CURVE_Pxa=[...]Chunk {0x6059885BAC9472,0x7C4D31DE2DC36D,0xBDC90C308C88A7,0x29F01971C688FC,0x3693539C43F167,0xD81E5A561EB8BF,0x4D50722B56BF};
var CURVE_Pxb=[...]Chunk {0x9B4BD7A272AB23,0x7AF19D4F44DCE8,0x3F6F7B93206A34,0x571DD3E2A819FB,0x3A2BA3B635D7EE,0xAC28C780C1A126,0xEE3617C3E5B};
var CURVE_Pya=[...]Chunk {0x81D230977BD4FD,0xB660720DFDFC6,0x41FC9590C89A0C,0x2E1FBCF878287A,0x11C23014EEE65,0x28878816BB325E,0x8F40859A05C};
var CURVE_Pyb=[...]Chunk {0xA5E20A252C4CE6,0x5907A74AFF40C8,0x41760A42448EF3,0xFFEF82B0FDA199,0xA0F29A18D4EA49,0xAC7F7B86E4997B,0x1DCABBA88C12};

// Not used
var CURVE_W=[2][7]Chunk {{0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0}};
var CURVE_SB=[2][2][7]Chunk {{{0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0}},{{0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0}}};
var CURVE_WB=[4][7]Chunk {{0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0}};
var CURVE_BB=[4][4][7]Chunk {{{0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0}},{{0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0}},{{0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0}},{{0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0}}};

const USE_GLV bool=true
const USE_GS_G2 bool=true
const USE_GS_GT bool=true	
const GT_STRONG bool=false
*/

// BLS455 Curve
/*
const CURVETYPE int=WEIERSTRASS
const CURVE_PAIRING_TYPE int=BLS_CURVE
const CURVE_A int= 0

var CURVE_Order=[...]Chunk {0x7FFFFC00001,0xA00000400001C,0x25E000750001D10,0xE0000F10004F000,0x80000380002,0x10,0x0,0x0};
var CURVE_B = [...]Chunk {0xA,0x0,0x0,0x0,0x0,0x0,0x0,0x0};
var CURVE_Cof = [...]Chunk {0xA9557FFAABFFAAB,0xAAB15555B54AAB6,0x555556AA,0x0,0x0,0x0,0x0,0x0};
var CURVE_Gx =[...]Chunk {0x6D4C5DDFDFCEDD1,0x35C6F43B3A034FB,0x7F05B56A579C725,0xB1F2B8ECE11B321,0x9F342AB0CFE8392,0xA5911EE32767994,0x3005E40CC56ABED,0x18855F3B};
var CURVE_Gy =[...]Chunk {0x404FD79A6619B9B,0x69D80A5D6FA0286,0xEE722322D91A493,0xB1EE58431C1E968,0xCA9BC8953801F5F,0xDFAFD40FE9E388E,0x9F8985FC3DEB0D6,0x19A8DB77E};

var CURVE_Bnx=[...]Chunk {0x20000080000800,0x10000,0x0,0x0,0x0,0x0,0x0,0x0};
var CURVE_Cru=[...]Chunk {0x9202FFC00000AA9,0xFA5190F4A3762A,0x8B2B9BDD548FEC9,0xD7B469DB33A586A,0xC91731354CAFD99,0xF5B48D02FFFE695,0x57955572A900E0E,0x555559555};
var CURVE_Fra=[...]Chunk {0x9CCFBDCA2EBF21,0x572F54A73379964,0x72819F887545498,0x22BBC1CAD1F8534,0xA82CD7D435944F0,0x4594F818D030F7B,0xEDCBE3ADC0016A7,0x397EA4973};
var CURVE_Frb=[...]Chunk {0xA033043B5D1438A,0x6E5A00C3F72FC06,0x4717AB46118C70E,0x16993AE842C0609,0x3B78DA012CA06E9,0x501F99EA300E6EA,0x69C971C4E9FF768,0x1BD6B4BE1};
var CURVE_Pxa=[...]Chunk {0x475F20F0C1F542,0x65D6070F8567E10,0xD780698BB33D776,0x71F685ED1531721,0x303D3FEC5B6A49C,0x8DEF064FF553CEB,0xC0E9A31B4C463,0x2ECB12FA8};
var CURVE_Pxb=[...]Chunk {0x99086EE6749F03D,0xE89A55A5AC5EF2E,0x7B41AECD88EA016,0x622450FE6163E06,0x755066E1C8E296F,0xA80F219487326E8,0x66DBFBB0BEAEE59,0xECFFCE0};
var CURVE_Pya=[...]Chunk {0x83235A4581A77F4,0x9F0F367B7A7E10A,0x8FA0C4A66D55B9D,0xEF03F65E0D6EC4C,0x9C7DC299C1A9EC2,0x32453CA21CFA5AC,0x6C3DCD5ABB9C544,0x22471D90A};
var CURVE_Pyb=[...]Chunk {0xF413B6D9E1FDBA2,0xA7E630913DA0356,0xFBC913D9AC488E2,0x72E7CF61B401585,0x656D801B21C89ED,0xF9E921EEE0558F9,0x3D2B7B03CFC8698,0x33503CA8};

// Not used
var CURVE_W=[2][8]Chunk {{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0}};
var CURVE_SB=[2][2][8]Chunk {{{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0}},{{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0}}};
var CURVE_WB=[4][8]Chunk {{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0}};
var CURVE_BB=[4][4][8]Chunk {{{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0}},{{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0}},{{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0}},{{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0}}};

const USE_GLV bool=true
const USE_GS_G2 bool=true
const USE_GS_GT bool=true	
const GT_STRONG bool=false
*/
// BNT Curve
/*
const CURVETYPE int=WEIERSTRASS
const CURVE_PAIRING_TYPE int=BN_CURVE
const CURVE_A int= 0
var CURVE_B = [...]Chunk {0x2,0x0,0x0,0x0,0x0}
var CURVE_Cof = [...]Chunk {0x1,0x0,0x0,0x0,0x0}
var CURVE_Order=[...]Chunk {0x75777E8D30210D,0xD43492B2CB363A,0x3696F8D5F00E88,0x6517014EFA0BAB,0x240120DB}
var CURVE_Bnx=[...]Chunk {0x806000004081,0x40,0x0,0x0,0x0}
var CURVE_Cru=[...]Chunk {0xEB53D5AB4FCD87,0x82A5F2BAB11FAD,0x47651504C9764C,0x4801B1,0x0}
var CURVE_Fra=[...]Chunk {0xF5D14EADC80022,0x4904D6FACCE359,0xF190A13211BE6C,0xC9BBC4394F6509,0x1328A292}
var CURVE_Frb=[...]Chunk {0xA7EAB040ECA6F1,0xC513DF997D764,0x450657A3DEB01E,0x9B5B3D15AAA6A1,0x10D87E48}
var CURVE_Pxa=[...]Chunk {0x8987E2288E65BB,0xAD1CAA6313BE,0x325041548B7CCC,0x4C1339EBCC055,0x14483FCD}
var CURVE_Pxb=[...]Chunk {0x67888808DBE2C0,0x7FE1F81E34853A,0xA631A51B57B95,0x384EC302DA3FC5,0x87F46B3}
var CURVE_Pya=[...]Chunk {0x202C47E020CA1D,0xB4167E8399F36C,0xC6E5439F72C94C,0x102B0BD74A2C69,0x14E8C29C}
var CURVE_Pyb=[...]Chunk {0xD8437C716628F2,0x27E167BCB7DC6B,0xA82C7572681D0A,0x62454BD1EDEC18,0x17AFE2A4}
var CURVE_Gx =[...]Chunk {0x9DBBFEEEB4A712,0x555614F464BABE,0x3696F8D5F06E8A,0x6517014EFA0BAB,0x240120DB}
var CURVE_Gy =[...]Chunk {0x1,0x0,0x0,0x0,0x0}
var CURVE_W=[2][5]Chunk{{0x26430061838403,0x81218241998384,0x6001,0x0,0x0},{0x100C000008101,0x80,0x0,0x0,0x0}}
var CURVE_SB=[2][2][5]Chunk {{{0x2743C061840504,0x81218241998404,0x6001,0x0,0x0},{0x100C000008101,0x80,0x0,0x0,0x0}},{{0x100C000008101,0x80,0x0,0x0,0x0},{0x4F347E2BAC9D0A,0x5313107131B2B6,0x3696F8D5EFAE87,0x6517014EFA0BAB,0x240120DB}}}
var CURVE_WB=[4][5]Chunk {{0x6140602080C080,0x806080C08880C1,0x2000,0x0,0x0},{0xB53904088C4A85,0xAD2FA352DC6C36,0xDA436358868EDE,0x300120,0x0},{0x5ADCB204464583,0x5697D1A96E363B,0x6D21B1AC43476F,0x180090,0x0},{0x62412020814181,0x806080C0888141,0x2000,0x0,0x0}}
var CURVE_BB=[4][4][5]Chunk {{{0x74F71E8D2FE08D,0xD43492B2CB35FA,0x3696F8D5F00E88,0x6517014EFA0BAB,0x240120DB},{0x74F71E8D2FE08C,0xD43492B2CB35FA,0x3696F8D5F00E88,0x6517014EFA0BAB,0x240120DB},{0x74F71E8D2FE08C,0xD43492B2CB35FA,0x3696F8D5F00E88,0x6517014EFA0BAB,0x240120DB},{0x100C000008102,0x80,0x0,0x0,0x0}},{{0x100C000008101,0x80,0x0,0x0,0x0},{0x74F71E8D2FE08C,0xD43492B2CB35FA,0x3696F8D5F00E88,0x6517014EFA0BAB,0x240120DB},{0x74F71E8D2FE08D,0xD43492B2CB35FA,0x3696F8D5F00E88,0x6517014EFA0BAB,0x240120DB},{0x74F71E8D2FE08C,0xD43492B2CB35FA,0x3696F8D5F00E88,0x6517014EFA0BAB,0x240120DB}},{{0x100C000008102,0x80,0x0,0x0,0x0},{0x100C000008101,0x80,0x0,0x0,0x0},{0x100C000008101,0x80,0x0,0x0,0x0},{0x100C000008101,0x80,0x0,0x0,0x0}},{{0x806000004082,0x40,0x0,0x0,0x0},{0x2018000010202,0x100,0x0,0x0,0x0},{0x7476BE8D2FA00A,0xD43492B2CB35BA,0x3696F8D5F00E88,0x6517014EFA0BAB,0x240120DB},{0x806000004082,0x40,0x0,0x0,0x0}}}

const USE_GLV bool=true
const USE_GS_G2 bool=true
const USE_GS_GT bool=true	
const GT_STRONG bool=true

*/

// BNT2 Curve
/*
const CURVETYPE int=WEIERSTRASS
const CURVE_PAIRING_TYPE int=BN_CURVE
const CURVE_A int= 0
var CURVE_B = [...]Chunk {0x2,0x0,0x0,0x0,0x0}
var CURVE_Cof = [...]Chunk {0x1,0x0,0x0,0x0,0x0}
var CURVE_Order=[...]Chunk {0xFB71A511AA2BF5,0x8DE127B73833D7,0xF3B89D00021CF,0x410F5AADB74E20,0x24000482}
var CURVE_Bnx=[...]Chunk {0x20100608205,0x40,0x0,0x0,0x0}
var CURVE_Cru=[...]Chunk {0x5027444866BD33,0x5B773016470EFB,0xC3617BECF23675,0x480006,0x0}
var CURVE_Fra=[...]Chunk {0xB268C973AEF062,0xC69B33C3BCE492,0xF67FA37F195BBC,0x29E8CAB6BD0A41,0x124E0B8D}
var CURVE_Frb=[...]Chunk {0x736240B1B429,0xCD48F52D196D56,0x18BBE650E72612,0x17268FF6FA43DE,0x11B1F8F5}
var CURVE_Pxa=[...]Chunk {0xCC92399F40A3C8,0xCDA4E96611784A,0x7B056961706B35,0x9693C6318279D7,0x16FC17CF}
var CURVE_Pxb=[...]Chunk {0x557A8AD8549540,0x6F7BE6F6510610,0x565907A95D17DB,0xBD5975909C8188,0x1EB5B500}
var CURVE_Pya=[...]Chunk {0x7BECC514220513,0x4A78860E737B14,0x51B83935F12684,0x761422AA9D4DFA,0x1E8EE498}
var CURVE_Pyb=[...]Chunk {0xB9328F577CE78E,0xB746E26FA5781F,0xA93DBC1FB8E27E,0xBAE33BDBA29D76,0x23CEF4CD}
var CURVE_Gx =[...]Chunk {0xB2DC2BB460A48A,0x93E428F0D651E8,0xF3B89D00081CF,0x410F5AADB74E20,0x24000482}
var CURVE_Gy =[...]Chunk {0x1,0x0,0x0,0x0,0x0}
var CURVE_W=[2][5]Chunk {{0xB76282A1347083,0x60301399E1D10,0x6000,0x0,0x0},{0x40200C10409,0x80,0x0,0x0,0x0}}
var CURVE_SB=[2][2][5]Chunk {{{0xB76684A1F5748C,0x60301399E1D90,0x6000,0x0,0x0},{0x40200C10409,0x80,0x0,0x0,0x0}},{{0x40200C10409,0x80,0x0,0x0,0x0},{0x440F227075BB72,0x87DE267D9A16C7,0xF3B89CFFFC1CF,0x410F5AADB74E20,0x24000482}}}
var CURVE_WB=[4][5]Chunk {{0x9272D48A70A224,0x20100688A0945,0x2000,0x0,0x0},{0x5A572CF030EF19,0x9651763543721D,0x8240FD48A1B9A3,0x300004,0x0},{0xAD2C96F848B88F,0xCB28BB1AA1B92E,0x41207EA450DCD1,0x180002,0x0},{0x9276D68B31A62D,0x20100688A09C5,0x2000,0x0,0x0}}
var CURVE_BB=[4][4][5]Chunk {{{0xFB6FA41149A9F1,0x8DE127B7383397,0xF3B89D00021CF,0x410F5AADB74E20,0x24000482},{0xFB6FA41149A9F0,0x8DE127B7383397,0xF3B89D00021CF,0x410F5AADB74E20,0x24000482},{0xFB6FA41149A9F0,0x8DE127B7383397,0xF3B89D00021CF,0x410F5AADB74E20,0x24000482},{0x40200C1040A,0x80,0x0,0x0,0x0}},{{0x40200C10409,0x80,0x0,0x0,0x0},{0xFB6FA41149A9F0,0x8DE127B7383397,0xF3B89D00021CF,0x410F5AADB74E20,0x24000482},{0xFB6FA41149A9F1,0x8DE127B7383397,0xF3B89D00021CF,0x410F5AADB74E20,0x24000482},{0xFB6FA41149A9F0,0x8DE127B7383397,0xF3B89D00021CF,0x410F5AADB74E20,0x24000482}},{{0x40200C1040A,0x80,0x0,0x0,0x0},{0x40200C10409,0x80,0x0,0x0,0x0},{0x40200C10409,0x80,0x0,0x0,0x0},{0x40200C10409,0x80,0x0,0x0,0x0}},{{0x20100608206,0x40,0x0,0x0,0x0},{0x80401820812,0x100,0x0,0x0,0x0},{0xFB6DA310E927EA,0x8DE127B7383357,0xF3B89D00021CF,0x410F5AADB74E20,0x24000482},{0x20100608206,0x40,0x0,0x0,0x0}}}

const USE_GLV bool=true
const USE_GS_G2 bool=true
const USE_GS_GT bool=true	
const GT_STRONG bool=true

*/


