'use strict';

var fs = require('fs');
var path = require('path');
var generators = require('yeoman-generator');
var _ = require('lodash');
var mout = require('mout');
var cordova = require('cordova');
var chalk = require('chalk');
var ionicUtils = require('../utils');
var mkdir = require('mkdirp');

var appPath = path.join(process.cwd(), 'app');

module.exports = generators.Base.extend({
  constructor: function () {
    generators.Base.apply(this, arguments);
    console.log(ionicUtils.greeting);

    this.argument('appName', { type: String, required: false });
    this.option('appName', { type: String, required: false });
    this.option('appId', { type: String, required: false });
    this.option('compass', { type: Boolean, required: false });
    this.option('starter', { type: String, required: false });
    this.option('templates', { type: Array, required: false });
    this.option('plugins', { type: Object, required: false });
    this.option('auth', { type: Object, required: false });
    this.option('flurry', { type: Object, required: false });
    this.option('push', { type: Object, required: false });
    this.options.selected = {};
    this.lplugins = [];
  },

  prompting: {
    askForCompass: function askForCompass() {
      var done = this.async();

      this.prompt([{
        type: 'confirm',
        name: 'compass',
        message: 'Would you like to use Sass with Compass (requires Ruby)?',
        default: (typeof(this.options.compass) !== 'undefined') ? this.options.compass : false
      }], function (props) {
        this.compass = this.options.selected.compass = props.compass;

        done();
      }.bind(this));
    },
    askForAnalytics: function askForAnalytics() {
      var done = this.async();

      this.prompt([{
        type: 'confirm',
        name: 'flurry',
        message: 'Would you like to use Flurry Analytics?',
        default: (typeof(this.options.flurry) !== 'undefined') ? this.options.flurry : false
      }], function (props) {
        this.flurry = this.options.selected.flurry = props.flurry;

        done();
      }.bind(this));
    },
    askForPush: function askForPush() {
      var done = this.async();

      this.prompt([{
        type: 'confirm',
        name: 'push',
        message: 'Would you like to use Parse push notification?',
        default: (typeof(this.options.push) !== 'undefined') ? this.options.push : false
      }], function (props) {
        this.push = this.options.selected.push = props.push;

        done();
      }.bind(this));
    },
    askForPlugins: function askForPlugins() {
      var done = this.async();

      if (this.options.plugins) {
        ionicUtils.mergePlugins(this.options.plugins);
      }

      this.prompt(ionicUtils.plugins.prompts, function (props) {
        this.plugins = this.options.selected.plugins = props.plugins;

        done();
      }.bind(this));
    },
    askForAuthentificationProvider: function askForAuthentificationProvider() {
      var done = this.async();
      console.log('!!!!!!!!!');
      console.log(this.options.auth);
      if (this.options.auth) {
        console.log('!!!!!!!!!');
        ionicUtils.mergePlugins(this.options.auth);
      }
      console.log(this.options.auth);

      this.prompt(ionicUtils.auth.prompts, function (props) {
        this.auth = this.options.selected.auth = props.auth;
        console.log(this.auth);
        done();
      }.bind(this));
    },
    askForStarter: function askForStarter() {
      var done = this.async();

      if (this.options.templates) {
        ionicUtils.mergeStarterTemplates(this.options.templates);
      }

      var defaultIndex = 0;
      if (this.options.starter) {
        defaultIndex = _.findIndex(ionicUtils.starters.templates, { name: this.options.starter });

        if (defaultIndex === -1) {
          defaultIndex = 0;
          this.log(chalk.bgYellow(chalk.black('WARN')) +
            chalk.magenta(' Unable to locate the requested default template: ') +
            this.options.starter);
        }
      }

      this.prompt([{
        type: 'list',
        name: 'starter',
        message: 'Which starter template would you like to use?',
        choices: _.pluck(ionicUtils.starters.templates, 'name'),
        default: defaultIndex
      }], function (props) {
        this.starter = this.options.selected.starter = _.find(ionicUtils.starters.templates, { name: props.starter });
        done();
      }.bind(this));
    }

  },

  configuring: {
    commonVariables: function() {
      this.appName = this.appName || this.options.appName || path.basename(process.cwd());
      this.appName = mout.string.pascalCase(this.appName);
      this.appId = this.options.appId || 'com.example.' + this.appName;
      this.appPath = 'app';
      this.root = process.cwd() + '/' + this.appName;

      this.pkg = JSON.parse(this.readFileAsString(path.join(__dirname, '../package.json')));
    },

    setupEnv: function setupEnv() {
        // Removes thumbnail cache files
        var invisibleFiles = ['Thumbs.db', '.DS_Store'];
        invisibleFiles.forEach(function(filename) {
            var file = path.join(process.cwd(), filename)
            if(fs.existsSync(file) ) {
                fs.unlinkSync(file);
            }
       });
      
      // Copies the contents of the generator example app
      // directory into your users new application path
      this.sourceRoot(path.join(__dirname, '../templates/'));
      this.directory('common/root', '.', true);
    },

    packageFiles: function packageFiles() {
      this.template('common/_bower.json', 'bower.json');
      this.template('common/_bowerrc', '.bowerrc');
      this.template('common/_package.json', 'package.json');
      this.copy('common/_Gruntfile.js', 'Gruntfile.js');
      this.template('common/_gitignore', '.gitignore');
    }
  },

  writing: {
    cordovaInit: function cordovaInit() {
      var done = this.async();
      cordova.create('.', this.appId, this.appName, function (error) {
        if (error) {
          console.log(chalk.yellow(error.message + ': Skipping `cordova create`'));
        } else {
          console.log(chalk.yellow('Created a new Cordova project with name "' + this.appName + '" and id "' + this.appId + '"'));
          mkdir.sync('local_plugins');
        }
        done();
      }.bind(this));
    },
    installAuthProvider: function installAuthProvider() {
      if (this.auth) {
        var done = this.async();
        var self = this;
        console.log(this.auth);
        done();
      } else {
        console.log('NO AUTH');
      }
    },
    installPush: function installPush() {
      if (this.push) {
        var self = this;
        var done = this.async();
        self.prompt({
          type    : 'input',
          name    : 'APP_ID',
          message : 'What\'s your Parse APP_ID',
          store   : true
        }, function(app_id) {
          console.log(app_id);
          self.prompt({
            type    : 'input',
            name    : 'CLIENT_ID',
            message : 'What\'s your Parse CLIENT_ID',
            store   : true
          }, function(client_id) {
            console.log(client_id);
            self.directory('../templates/plugins/com.plugins.parsePushNotifications', 'local_plugins/com.plugins.parsePushNotifications');
            self.lplugins.push('local_plugins/com.plugins.parsePushNotifications --variable APP_ID=' + app_id + ' --variable CLIENT_ID=' + client_id);
            done();
          });
        });
        
      }
    },
    installFlurry: function installFlurry() {
      if (this.flurry) {
        var done = this.async();
        var self = this;
        self.prompt({
            type    : 'input',
            name    : 'FLURRY_ID',
            message : 'What\'s your Flurry APP_ID',
            store   : true
          }, function(flurry_id) {
            console.log(flurry_id);
            self.directory('../templates/plugins/com.plugins.flurryLib', 'local_plugins/com.plugins.flurryLib');
            self.lplugins.push('local_plugins/com.plugins.flurryLib' + ' --variable APP_ID=' + flurry_id );
            done();
          });
        
      }
    },
    installPlugins: function installPlugins() {
      console.log(chalk.yellow('\nInstall plugins registered at plugins.cordova.io: ') + chalk.green('grunt plugin:add:org.apache.cordova.globalization'));
      console.log(chalk.yellow('Or install plugins direct from source: ') + chalk.green('grunt plugin:add:https://github.com/apache/cordova-plugin-console.git\n'));
      if (this.plugins.length > 0) {
        console.log(chalk.yellow('Installing selected Cordova plugins, please wait.'));
        
        // Turns out plugin() doesn't accept a callback so we try/catch instead
        try {
          cordova.plugin('add', this.plugins);
        } catch (e) {
          console.log(e);
          this.log.error(chalk.red('Please run `yo ionic` in an empty directory, or in that of an already existing cordova project.'));
          process.exit(1);
        }
      }
      if (this.lplugins.length > 0) {
        for (var i = this.lplugins.length - 1; i >= 0; i--) {
          var plugin = this.lplugins[i];
        };
      }
    },

    installStarter: function installStarter() {
      console.log(chalk.yellow('Installing starter template. Please wait'));
      var done = this.async();

      var callback = function (error, remote) {
        if (error) {
          done(error);
        }

        // Template remote initialization: Copy from remote root folder (.) to working directory (/app)
        remote.directory('.', 'app');
 
        this.starterCache = remote.cachePath;
        done();
      }.bind(this);

      if (this.starter && this.starter.path) {
        this.log(chalk.bgYellow(chalk.black('WARN')) +
          chalk.magenta(' Getting the template from a local path.  This should only be used for developing new templates.'));
        this.remoteDir(this.starter.path, callback);
      } else if (this.starter.url) {
        this.remote(this.starter.url, callback, true);
      } else {
        this.remote(this.starter.user, this.starter.repo, 'master', callback, true);
      }
    },

    readIndex: function readIndex() {
      this.indexFile = this.engine(this.read(path.join(this.starterCache, 'index.html')), this);
    },

    appJs: function appJs() {
     
      var scriptPrefix = 'js' + path.sep;

      var scripts = [scriptPrefix + 'configuration.js'];
      
      this.fs.store.each(function (file, index) {
        if (file.path.indexOf('.js') !== -1)
        {
          var relPath = path.relative(appPath, file.path);
          if (relPath.indexOf(scriptPrefix) === 0) {
            scripts.push(relPath);
          }
        }
      });

      //this.indexFile = this.appendScripts(this.indexFile, 'scripts/scripts.js', scripts);
    },

    createIndexHtml: function createIndexHtml() {
             
        // Regex: Vendor CSS
        this.indexFile = this.indexFile.replace(/<link href="lib\/ionic\/css\/ionic.css" rel="stylesheet">/g, "<!-- build:css styles\/vendor.css -->\n    <!-- bower:css -->\n    <!-- endbower -->\n    <!-- endbuild -->");
        
        // Regex: User CSS
        //this.indexFile = this.indexFile.replace(/<link href="css\/style.css" rel="stylesheet">/g, "<!-- build:css styles\/vendor.css -->\n    <!-- bower:css -->\n    <!-- endbower -->\n    <!-- endbuild -->");
        
        // Regex: Vendor scripts (vendor.js)
        this.indexFile = this.indexFile.replace(/<script src="lib\/ionic\/js\/ionic.bundle.js"><\/script>/g, "<!-- build:js scripts\/vendor.js -->\n    <!-- bower:js -->\n    <!-- endbower -->\n    <!-- endbuild -->");
      
       // Regex: User scripts (scripts.js)
       this.indexFile = this.indexFile.replace(/<!-- your app's js -->/g,"<!-- your app's js -->\n    <!-- build:js scripts\/scripts.js -->");
       this.indexFile = this.indexFile.replace(/<\/head>/g,"  <script src=\"scripts\/configuration.js\"><\/script>\n    <!-- endbuild -->\n  <\/head>");
       
       // Regex/Rename: Scripts path (Ionics 'js' to 'scripts')
       this.indexFile = this.indexFile.replace(/href="css/g,"href=\"styles");
       
       // Regex/Rename: CSS path (Ionics 'css' to 'styles')
       this.indexFile = this.indexFile.replace(/src="js/g,"src=\"scripts");
     
       // Write index.html
       this.indexFile = this.indexFile.replace(/&apos;/g, "'");
       this.write(path.join(this.appPath, 'index.html'), this.indexFile);
    },

    ensureStyles: function ensureStyles() {
      // Only create a main style file if the starter template didn't
      // have any styles. In the case it does, the starter should
      // supply both main.css and main.scss files, one of which
      // will be deleted
      
      var cssFile = 'main.' + (this.compass ? 'scss' : 'css');
      var unusedFile = 'main.' + (this.compass ? 'css' : 'scss');
      var stylePath = path.join(process.cwd(), 'app', 'css');
      var found = false;

      this.fs.store.each(function (file, index) {
        if (path.dirname(file.path) === stylePath) {
          var name = path.basename(file.path);

          if (name === cssFile) {
            found = true;
          } else if (name === unusedFile) {
            // BUG: The log will still report the the file was created
            this.fs.delete(file.path);
          }
        }
      }.bind(this));

      if (!found) {
        this.copy('styles/' + cssFile, 'app/css/' + cssFile);
      }

    },

    cordovaHooks: function cordovaHooks() {
      this.directory('hooks', 'hooks', true);
    },

    packages: function () {
      this.installDependencies({ skipInstall: this.options['skip-install'] });
    }
  },

  end: {
    hookPerms: function hookPerms() {
      var iconsAndSplash = 'hooks/after_prepare/icons_and_splashscreens.js';
      fs.chmodSync(iconsAndSplash, '755');
    },
    folderNames: function folderNames() {
      // Rename: Scripts path (Ionics 'js' to 'scripts')
      fs.rename(path.join(appPath, 'css'), path.join(appPath, 'styles'), function(err) {
          if ( err ) console.log('ERROR: ' + err);
      });
      
      // Rename: CSS path (Ionics 'css' to 'styles')
      fs.rename(path.join(appPath, 'js'), path.join(appPath, 'scripts'), function(err) {
          if ( err ) console.log('ERROR: ' + err);
      });
      
      // Rename: Images path (Ionics 'img' to 'images')
      fs.rename(path.join(appPath, 'img'), path.join(appPath, 'images'), function(err) {
          if ( err ) console.log('ERROR: ' + err);
      });
    }
  }
});

