/**
 * LZW compression library.
 * @author Felipe Alfonso
 */

var LZW = LZW || {};

LZW.isNode = function () {
    return typeof global !== 'undefined' && 'Buffer' in global;
};

LZW.arrayToBuffer = function (array, octets) {
    var buffer,
        write,
        bufferView,
        length,
        index = 0;

    octets = octets || 1;
    if (this.isNode()) {
        buffer = new Buffer(array.length * octets);
        write = (function () {
            if (octets === 1) {
                return function (value, index) {
                    buffer[index] = value;
                };
            } else if (octets === 2) {
                return function (value, index) {
                    buffer.writeUInt16BE(value, index);
                };
            } else if (octets === 3) {
                return function (value, index) {
                    buffer.writeUInt32BE(value, index);
                };
            } else {
                throw new Error('Incorrect number of octets.');
            }
        }());
        length = buffer.length;
        while (index < length) {
            if (index % octets === 0) {
                write(array[index / octets], index);
            }
            ++index;
        }
    } else {
        buffer = new ArrayBuffer(array.length * octets);
        bufferView = (function () {
            if (octets === 1) {
                return new Uint8Array(buffer);
            } else if (octets === 2) {
                return new Uint16Array(buffer);
            } else if (octets === 3) {
                return new Uint32Array(buffer);
            } else {
                throw new Error('Incorrect number of octets.');
            }
        }()),
        length = array.length;
        while (index < length) {
            bufferView[index] = array[index];
            ++index;
        }
    }
    return buffer;
};

LZW.bufferToArray = function (buffer, octets) {
    var output = [],
        index = 0,
        length,
        bufferView;

    octets = octets || 1;
    if (this.isNode()) {
        read = (function () {
            if (octets === 1) {
                return function (index) {
                    return buffer[index];
                };
            } else if (octets === 2) {
                return function (index) {
                    return buffer.readUInt16BE(index);
                };
            } else if (octets === 3) {
                return function (index) {
                    return buffer.readUInt32BE(index);
                };
            } else {
                throw new Error('Incorrect number of octets.');
            }
        }());

        length = buffer.length;
        while (index < length) {
            if (index % octets === 0) {
                output.push(read(index));
            }
            ++index;
        }
    } else {
        bufferView = (function () {
            if (octets === 1) {
                return new Uint8Array(buffer);
            } else if (octets === 2) {
                return new Uint16Array(buffer);
            } else if (octets === 3) {
                return new Uint32Array(buffer);
            } else {
                throw new Error('Incorrect number of octets.');
            }
        }());
        length = bufferView.length;
        while (index < length) {
            output.push(bufferView[index]);
            ++index;
        }
    }
    return output;
};

LZW.compress = function (string) {
    'use strict';
    var dictionary = {},
        index,
        length = string.length,
        size = (function (x) {
            var max = 255 + 1000;
            x.forEach(function (n) {
                if (n > max) max = n;
            });
            return ++max;
        }(this.stringToArray(string))),
        output = (function (n) {
            return n > 256 + 1000 ? [n] : [];
        }(size)),
        current = '',
        previous = '';

    for (index = 0; index < length; ++index) {
        dictionary[string[index]] = string.charCodeAt(index) + 1000;
    }
    for (index = 0; index < length; ++index) {
        current = previous + string[index];
        if (dictionary[current]) {
            previous = current;
        } else {
            output.push(dictionary[previous]);
            dictionary[current] = size++;
            previous = string[index];
        }
    }
    if (previous.length > 0) {
        output.push(dictionary[previous]);
    }
    return output;
};

LZW.decompress = function (array) {
    'use strict';
    var dictionary = {},
        index,
        startPoint = 0,
        length = array.length,
        size,
        output = '',
        current,
        character = '',
        pointer,
        copy = array.slice();

    for (index = 0; index < copy.length; ++index) {
        copy[index] -= 1000;
    }

    size = (function () {
        if (copy[0] > 256) {
            startPoint = 1;
            return copy[0];
        } else {
            return 256;
        }
    }());

    for (index = 0; index < size; ++index) {
        dictionary[index] = String.fromCharCode(index);
    }

    current = dictionary[copy[startPoint++]];
    output = current;
    for (index = startPoint; index < length; ++index) {
        pointer = copy[index];
        if (dictionary[pointer]) {
            character = dictionary[pointer];
        } else if (pointer === size) {
            character = current + current[0];
        } else {
            return null;
        }
        output += character;
        dictionary[size++] = current + character[0];
        current = character;
    }

    return output;
};

LZW.compressToBuffer = function (string) {
    var compress = this.compress(string);

    return this.arrayToBuffer(compress, 2);
};

LZW.decompressBuffer = function (buffer) {
    var array = this.bufferToArray(buffer, 2),
        decompress = this.decompress(array);

    return decompress;
};

LZW.arrayToString = function (n) {
    var i = 0,
        l = n.length,
        o = '';
    for (i = 0; i < l; ++i) {
        o += String.fromCharCode(n[i] - 1000);
    }
    return o;
};

LZW.stringToArray = function (n) {
    var i = 0,
        l = n.length,
        o = [];
    for (i = 0; i < l; ++i) {
        o.push(n.charCodeAt(i) + 1000);
    }
    return o;
};

LZW.stringToBuffer = function (string) {
    var array = this.stringToArray(string);
    return this.arrayToBuffer(array, 2);
};

LZW.bufferToString = function (buffer) {
    var array = this.bufferToArray(buffer, 2);
    return this.arrayToString(array);
};

if (LZW.isNode()) {
    for (n in LZW) {
        exports[n] = LZW[n];
    }
}
