'use strict';

var React = require('react');
var classNames = require('classnames');

var SIZES = require('./SIZES');
var Button = require('./Button');
var Input = require('./Input');
var Backdrop = require('./Backdrop');

var DEFAULT_SEARCH_PLACEHOLDER = 'Search';

var itemShape = React.PropTypes.oneOfType([React.PropTypes.string, React.PropTypes.object]);

var groupShape = React.PropTypes.shape({
    label: React.PropTypes.string,
    options: React.PropTypes.arrayOf(itemShape)
});

/**
 * Default filter for select
 */
function defaultFilter(query, item, i) {
    return true;
}

/**
 * Default render for options
 */
function defaultComponent(_ref) {
    var option = _ref.option;

    return React.createElement(
        'span',
        null,
        option
    );
}
defaultComponent.propTypes = {
    option: itemShape
};

/**
 * Default render to string for input
 */
function defaultRenderToString(item, i) {
    return String(item);
}

/**
 * Interractive select for forms
 *
 * It renders as a normal select on server and has a custom UI on browser (with search, images support).
 *
 * <Select name="test" options=[] />
 */
var Select = React.createClass({
    displayName: 'Select',

    propTypes: {
        // Current value of the select
        value: React.PropTypes.oneOfType([itemShape, React.PropTypes.arrayOf(itemShape)]),

        // List of items to display
        groups: React.PropTypes.arrayOf(groupShape),
        options: React.PropTypes.arrayOf(itemShape),

        // Function to render the option to a string or element
        component: React.PropTypes.func,

        // Function to render the selected option in the button
        // Defaults to "renderOption"
        componentSelection: React.PropTypes.func,

        // Function to output an option as a string
        // Defaults to a string representation, you have to provide your own value
        // when using a custom option renderer
        renderToString: React.PropTypes.func,

        // Function to filter an element
        filter: React.PropTypes.func,

        // Optional callback when value changed
        onChange: React.PropTypes.func,

        // Name when using server posting
        name: React.PropTypes.string,

        // Text to display when no value is set
        placeholder: React.PropTypes.string,
        searchPlaceholder: React.PropTypes.string,

        // Delimiter for multiple values
        delimiter: React.PropTypes.string,

        // Prevent selection
        disabled: React.PropTypes.bool,

        // Display the search filter?
        search: React.PropTypes.bool,

        // Accept multiple values
        multiple: React.PropTypes.bool,

        // Size of the select to display
        size: React.PropTypes.oneOf(SIZES),

        // Take the whole width
        block: React.PropTypes.bool
    },

    getDefaultProps: function getDefaultProps() {
        return {
            disabled: false,
            search: true,
            delimiter: ',',
            size: SIZES[0],
            multiple: false,
            block: false,
            filter: defaultFilter,
            component: defaultComponent,
            renderToString: defaultRenderToString,
            searchPlaceholder: DEFAULT_SEARCH_PLACEHOLDER,
            placeholder: 'Select'
        };
    },
    getInitialState: function getInitialState() {
        return {
            value: this.props.value,
            query: '',
            opened: false,
            groups: this.propsToGroups(this.props)
        };
    },
    componentWillReceiveProps: function componentWillReceiveProps(newProps) {
        this.setState({
            value: newProps.value,
            groups: this.propsToGroups(newProps),
            opened: newProps.disabled ? false : this.state.opened
        });
    },


    /**
     * Create list of groups from props
     * @param {Object} props
     * @return {Array<groupShape>}
     */
    propsToGroups: function propsToGroups(props) {
        var _props = this.props,
            options = _props.options,
            groups = _props.groups;


        if (groups) {
            return groups;
        }

        return [{ options: options }];
    },


    /**
     * Search query changed
     */
    onSearchChanged: function onSearchChanged(e) {
        this.setState({
            query: e.target.value
        });
    },


    /**
     * Toggle (close/open) the select
     */
    onToggle: function onToggle() {
        this.setState({
            opened: !this.state.opened
        });
    },


    /**
     * Close the select
     */
    close: function close() {
        this.setState({
            opened: false
        });
    },


    /**
     * Open the select
     */
    open: function open() {
        this.setState({
            opened: false
        });
    },


    /**
     * Focus the search if open
     */
    focusOnOpen: function focusOnOpen() {
        if (this.state.opened) {
            this.focusSearch();
        }
    },
    componentDidUpdate: function componentDidUpdate() {
        this.focusOnOpen();
    },
    componentDidMount: function componentDidMount() {
        this.focusOnOpen();
    },


    /**
     * Toggle an option
     */
    onToggleOption: function onToggleOption(addValue, e) {
        if (e) {
            e.preventDefault();
        }

        var _state = this.state,
            value = _state.value,
            multiple = _state.multiple;
        var onChange = this.props.onChange;

        var newState = void 0,
            newValue = void 0;

        if (multiple) {
            newValue = value;

            // Add to selection if not yet selected
            if (!this.hasValue(addValue)) {
                newValue = value.concat([addValue]);
            } else if (value.length > 1) {
                // Unselect if many options are selected
                newValue.splice(newValue.indexOf(addValue), 1);
            }

            newState = {
                value: newValue
            };
        } else {
            newValue = addValue;

            newState = {
                value: addValue,
                opened: false
            };
        }

        this.setState(newState, function () {
            if (onChange) {
                onChange(newValue);
            }
        });
    },


    /**
     * Get current value as a string (for hidden input)
     * @return {String}
     */
    getStringValue: function getStringValue() {
        var renderToString = this.props.renderToString;
        var value = this.state.value;


        if (!value) {
            return '';
        }

        if (!this.props.multiple) {
            return renderToString(value);
        } else {
            return value.map(renderToString).join(this.props.delimiter);
        }
    },


    /**
     * Check if a value is selected
     * @param {String} value
     * @return {Boolean}
     */
    hasValue: function hasValue(value) {
        var currentValue = this.state.value;

        if (!this.props.multiple) {
            return currentValue === value;
        } else {
            return currentValue.indexOf(value) >= 0;
        }
    },


    /**
     * Focus the search input
     */
    focusSearch: function focusSearch() {
        var searchInput = this.refs.searchInput;

        if (!searchInput) {
            return;
        }

        searchInput.focus();
    },


    /**
     * Render button to open select
     */
    renderButton: function renderButton() {
        var _props2 = this.props,
            disabled = _props2.disabled,
            block = _props2.block,
            multiple = _props2.multiple,
            placeholder = _props2.placeholder;
        var _state2 = this.state,
            value = _state2.value,
            opened = _state2.opened;

        var ComponentSelection = this.props.componentSelection || this.props.component;

        var inner = void 0;

        if (value) {
            var values = multiple ? value : [value];
            inner = React.createElement(
                'span',
                { className: 'SelectSelections' },
                values.map(function (val, i) {
                    return React.createElement(
                        'span',
                        { key: i, className: 'SelectSelection' },
                        React.createElement(ComponentSelection, { option: val, index: i })
                    );
                })
            );
        } else {
            inner = React.createElement(
                'span',
                { className: 'SelectPlaceholder' },
                placeholder
            );
        }

        return React.createElement(
            Button,
            { size: this.props.size, block: block, disabled: disabled, active: opened, onClick: this.onToggle },
            inner,
            ' ',
            React.createElement(Button.Caret, null)
        );
    },


    /**
     * Render button to open select
     */
    renderSearch: function renderSearch() {
        var query = this.state.query;


        return React.createElement(
            'div',
            { className: 'SelectSearch' },
            React.createElement(Input, { ref: 'searchInput',
                value: query,
                onChange: this.onSearchChanged,
                placeholder: this.props.placeholder
            })
        );
    },


    /**
     * Render the options selector
     */
    renderGroup: function renderGroup(group, index) {
        var query = this.state.query;
        var filter = this.props.filter;

        var Component = this.props.component;
        var count = 0;

        var options = group.options.map(function (item, i) {
            var _this = this;

            if (!filter(query, item, i)) {
                return '';
            }

            count++;

            return React.createElement(
                'div',
                {
                    key: i,
                    className: classNames('SelectOption', { active: this.hasValue(item) }),
                    onClick: function onClick(e) {
                        return _this.onToggleOption(item);
                    }
                },
                React.createElement(Component, { option: item, index: i })
            );
        }, this);

        // Don't display empty groups (when filtered)
        if (count === 0) {
            return '';
        }

        return React.createElement(
            'div',
            { key: index, className: 'SelectOptGroup' },
            group.label ? React.createElement(
                'div',
                { className: 'GroupLabel' },
                group.label
            ) : '',
            React.createElement(
                'div',
                { className: 'GroupOptions' },
                options
            )
        );
    },


    /**
     * Render the groups
     */
    renderGroups: function renderGroups() {
        var _state3 = this.state,
            opened = _state3.opened,
            groups = _state3.groups;
        var search = this.props.search;


        var className = classNames('SelectContainer', {
            'open': opened
        });

        return React.createElement(
            'div',
            { className: className },
            search ? this.renderSearch() : '',
            React.createElement(
                'div',
                { className: 'SelectGroups' },
                groups.map(this.renderGroup)
            )
        );
    },
    render: function render() {
        var _props3 = this.props,
            name = _props3.name,
            block = _props3.block;
        var opened = this.state.opened;


        var className = classNames('SelectFormControl', {
            block: block
        });

        return React.createElement(
            'div',
            { className: className, onClick: function onClick(e) {
                    return e.stopPropagation();
                } },
            React.createElement('input', { type: 'hidden', name: name, value: this.getStringValue() }),
            this.renderButton(),
            opened ? React.createElement(
                Backdrop,
                { onClose: this.close },
                this.renderGroups()
            ) : ''
        );
    }
});

module.exports = Select;
//# sourceMappingURL=data:application/json;base64,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