'use strict';

var React = require('react');
var moment = require('moment');

var DateSpan = require('./Date');

var TimeLine = React.createClass({
    displayName: 'TimeLine',

    propTypes: {
        currentX: React.PropTypes.number,
        lineTop: React.PropTypes.number,
        lineBottom: React.PropTypes.number,
        lastX: React.PropTypes.number,
        lastPoints: React.PropTypes.array,
        points: React.PropTypes.array,
        setHovered: React.PropTypes.func
    },

    contextTypes: {
        pointRadius: React.PropTypes.number,
        lineWidth: React.PropTypes.number,
        axeColor: React.PropTypes.string
    },

    getInitialState: function getInitialState() {
        return {
            hovered: false
        };
    },
    setHovered: function setHovered(state, params) {
        // Send global state hovered parameters
        if (!state) {
            params = null;
        }

        this.props.setHovered(params);

        this.setState({
            hovered: state
        });
    },
    renderPathLine: function renderPathLine(index) {
        var lastX = this.props.lastX;
        if (!Boolean(lastX)) {
            return null;
        }

        var currentX = this.props.currentX;

        var currentPoint = this.props.points[index];
        var lastPoint = this.props.lastPoints[index];

        return React.createElement('line', { x1: lastX, y1: lastPoint.y, x2: currentX, y2: currentPoint.y,
            stroke: currentPoint.color, strokeWidth: this.context.lineWidth });
    },
    render: function render() {
        var _this = this;

        var hovered = this.state.hovered;
        var _props = this.props,
            currentX = _props.currentX,
            lineTop = _props.lineTop,
            lineBottom = _props.lineBottom,
            points = _props.points;

        // Increase points radius if hovered

        var pointRadius = this.context.pointRadius;
        if (hovered) {
            pointRadius += 2;
        }

        var hoveredParams = {
            x: currentX,
            points: points
        };

        return React.createElement(
            'g',
            null,
            React.createElement('line', { x1: currentX, x2: currentX, y1: lineTop, y2: lineBottom,
                stroke: this.context.axeColor, strokeWidth: this.context.lineWidth }),
            points.map(function (point, index) {
                var color = point.color;
                // Default serie name to index
                var dataSerie = point.serie || index;

                return React.createElement(
                    'g',
                    { key: index },
                    _this.renderPathLine(index),
                    React.createElement('circle', { key: index,
                        onMouseEnter: _this.setHovered.bind(_this, true, hoveredParams),
                        onMouseLeave: _this.setHovered.bind(_this, false),
                        className: 'serie-point' + (hovered ? ' hovered' : ''),
                        r: pointRadius,
                        fill: color,
                        strokeWidth: _this.context.lineWidth * 2,
                        stroke: '#FFFFFF',
                        cx: currentX,
                        cy: point.y,
                        'data-value': point.value,
                        'data-timestamp': Number(point.date),
                        'data-serie': dataSerie }),
                    '}'
                );
            })
        );
    }
});

var Body = React.createClass({
    displayName: 'Body',

    propTypes: {
        dateMin: React.PropTypes.number,
        dateMax: React.PropTypes.number,
        valueMin: React.PropTypes.number,
        valueMax: React.PropTypes.number,
        width: React.PropTypes.number,
        height: React.PropTypes.number,
        innerX: React.PropTypes.number,
        innerY: React.PropTypes.number,
        yBase: React.PropTypes.number,
        series: React.PropTypes.array,
        setHovered: React.PropTypes.func
    },

    contextTypes: {
        seriesStyle: React.PropTypes.array,
        pointRadius: React.PropTypes.number,
        defaultColor: React.PropTypes.string,
        lineWidth: React.PropTypes.number,
        axeColor: React.PropTypes.string
    },

    // Compute X for a point inside Body
    innerPositionX: function innerPositionX(pX) {
        return this.props.innerX + pX * this.props.width;
    },


    // Compute Y for a point inside Body
    innerPositionY: function innerPositionY(pY) {
        return this.props.yBase + this.props.height * (1 - pY);
    },


    // Return inner X value for a date
    getInnerX: function getInnerX(date) {
        return this.innerPositionX((date - this.props.dateMin) / (this.props.dateMax - this.props.dateMin));
    },


    // Return inner Y value for a value
    getInnerY: function getInnerY(value) {
        return this.innerPositionY((value - this.props.valueMin) / (this.props.valueMax - this.props.valueMin));
    },


    // Return correct mapping for a point to draw in TimeLine
    formatPoint: function formatPoint(time, point, index) {
        var seriesStyle = this.context.seriesStyle;
        var serieStyle = seriesStyle[index];

        return {
            color: point.color || (serieStyle ? serieStyle.color : null) || this.context.defaultColor,
            value: point.value,
            serie: point.serie || (serieStyle ? serieStyle.title : index),
            date: time.date,
            y: this.getInnerY(point.value)
        };
    },


    // Draw this time line
    drawTimeLine: function drawTimeLine(time, i, series) {
        // Current time informations
        var currentX = this.getInnerX(time.date);
        var lineTop = this.props.yBase;
        var lineBottom = this.props.yBase + this.props.height;

        var points = time.points.map(this.formatPoint.bind(this, time));

        // Last time informations
        var lastX = null;
        var lastPoints = null;
        var lastTime = series[i - 1];

        if (Boolean(lastTime)) {
            lastX = this.getInnerX(lastTime.date);
            lastPoints = lastTime.points.map(this.formatPoint.bind(this, lastTime));
        }

        return React.createElement(TimeLine, { key: i,
            currentX: currentX,
            lineTop: lineTop,
            lineBottom: lineBottom,
            points: points,
            lastX: lastX,
            lastPoints: lastPoints,
            setHovered: this.props.setHovered });
    },
    render: function render() {
        var series = this.props.series;


        return React.createElement(
            'g',
            null,
            series.map(this.drawTimeLine, this)
        );
    }
});

var XAxis = React.createClass({
    displayName: 'XAxis',

    propTypes: {
        length: React.PropTypes.number,
        dateMin: React.PropTypes.number,
        dateMax: React.PropTypes.number,
        innerX: React.PropTypes.number,
        yTop: React.PropTypes.number
    },

    contextTypes: {
        textColor: React.PropTypes.string,
        textFontSize: React.PropTypes.number,
        textFontFamily: React.PropTypes.string
    },

    getValuePerInterval: function getValuePerInterval() {
        var dateMin = this.props.dateMin;
        var dateMax = this.props.dateMax;

        // Get optimal interval for X axis
        return [{
            format: 'SSSS',
            interval: 1
        }, {
            format: 'SSSS',
            interval: 10
        }, {
            format: 'SSSS',
            interval: 100
        }, {
            format: 'H:mm:ss',
            interval: 1 * 1000
        }, {
            format: 'H:mm',
            interval: 60 * 1000
        }, {
            format: 'H',
            interval: 60 * 60 * 1000
        }, {
            format: 'DD/MM',
            interval: 1 * 24 * 60 * 60 * 1000
        }, {
            format: 'DD/MM',
            interval: 7 * 24 * 60 * 60 * 1000
        }, {
            format: 'DD/MM',
            interval: 30 * 24 * 60 * 60 * 1000
        }, {
            format: 'YYYY',
            interval: 365 * 24 * 60 * 60 * 1000
        }, {
            format: 'YYYY',
            interval: 10 * 365 * 24 * 60 * 60 * 1000
        }, {
            format: 'YYYY',
            interval: 100 * 365 * 24 * 60 * 60 * 1000
        }].filter(function (interval) {
            var count = (dateMax - dateMin) / interval.interval;
            return count > 1;
        }).map(function (interval) {
            var count = (dateMax - dateMin) / interval.interval;
            return { count: count, interval: interval };
        }).sort(function (a, b) {
            return a.count > b.count;
        })[0].interval;
    },
    render: function render() {
        var _this2 = this;

        var _props2 = this.props,
            dateMin = _props2.dateMin,
            dateMax = _props2.dateMax,
            length = _props2.length,
            innerX = _props2.innerX,
            yTop = _props2.yTop;

        // Compute intervals for rendering dates

        var valuePerInterval = this.getValuePerInterval();
        var wPerMS = length / (dateMax - dateMin);
        var axeXInterval = (dateMax - dateMin) * wPerMS / (this.context.textFontSize * 10);

        // Construct range of intervals
        var intervalRange = [];
        for (var i = 0; i < axeXInterval; i++) {
            intervalRange.push(i);
        }

        return React.createElement(
            'g',
            null,
            intervalRange.map(function (i) {
                var value = i * valuePerInterval.interval;
                var date = new Date(dateMin + value);

                var x = innerX + value * wPerMS;
                // Don't draw point if too far
                if (x > innerX + length) {
                    return null;
                }

                return React.createElement(
                    'text',
                    { key: i, x: x, y: yTop,
                        fontFamily: _this2.context.textFontFamily, fontSize: _this2.context.textFontSize,
                        fill: _this2.context.textColor, textAnchor: 'middle' },
                    moment(date).format(valuePerInterval.format)
                );
            })
        );
    }
});

var YAxis = React.createClass({
    displayName: 'YAxis',

    propTypes: {
        length: React.PropTypes.number,
        valueMin: React.PropTypes.number,
        valueMax: React.PropTypes.number,
        innerX: React.PropTypes.number,
        innerY: React.PropTypes.number,
        yBase: React.PropTypes.number,
        axeYInterval: React.PropTypes.number
    },

    // Computes the optimal tick step for the Y axis
    // We assume: range = Math.abs(upper - lower)
    // i.e: range should not be negative
    optimalTickStep: function optimalTickStep(maxTicks) {
        var valueMin = this.props.valueMin;
        var valueMax = this.props.valueMax;

        var range = valueMax - valueMin;
        var minimum = range / maxTicks;
        var magnitude = Math.pow(10, Math.floor(Math.log(minimum) / Math.log(10)));
        var residual = minimum / magnitude;

        // Tick is an amplified magnitude
        // depending on the residual
        if (residual > 5) {
            return 10 * magnitude;
        } else if (residual > 2) {
            return 5 * magnitude;
        } else if (residual > 1) {
            return 2 * magnitude;
        }
        return magnitude;
    },


    contextTypes: {
        lineWidth: React.PropTypes.number,
        axeColor: React.PropTypes.string,
        axeMarkerWidth: React.PropTypes.number,
        textColor: React.PropTypes.string,
        textFontSize: React.PropTypes.number,
        textFontFamily: React.PropTypes.string
    },

    render: function render() {
        var _this3 = this;

        var _props3 = this.props,
            valueMin = _props3.valueMin,
            valueMax = _props3.valueMax,
            length = _props3.length,
            innerX = _props3.innerX,
            innerY = _props3.innerY,
            yBase = _props3.yBase;
        var axeYInterval = this.props.axeYInterval;


        var hPerValue = length / (valueMax - valueMin);
        axeYInterval = (valueMax - valueMin) * hPerValue / (this.context.textFontSize * 4);

        // Calcul perfect value per interval (1, 10, 100, 1000, ...)
        var valuePerInterval = this.optimalTickStep(axeYInterval);
        // Construct range of intervals
        var intervalRange = [];
        for (var i = 0; i < axeYInterval + 1; i++) {
            intervalRange.push(i);
        }

        return React.createElement(
            'g',
            null,
            intervalRange.map(function (i) {
                var value = i * valuePerInterval;
                var y = innerY - value * hPerValue;
                var displayValue = value + valueMin;

                // Don't draw point if is too high
                if (y < yBase) {
                    return null;
                }

                var textX = innerX - 2 * _this3.context.axeMarkerWidth;
                var lineX1 = innerX - _this3.context.axeMarkerWidth;
                var lineX2 = innerX;

                return React.createElement(
                    'g',
                    { key: i },
                    React.createElement(
                        'text',
                        { x: textX, y: y,
                            fontFamily: _this3.context.textFontFamily, fontSize: _this3.context.textFontSize,
                            textAnchor: 'end', fill: _this3.context.textColor },
                        displayValue
                    ),
                    React.createElement('line', { x1: lineX1, x2: lineX2, y1: y, y2: y,
                        stroke: _this3.context.axeColor, strokeWidth: _this3.context.lineWidth })
                );
            })
        );
    }
});

var TimeGraph = React.createClass({
    displayName: 'TimeGraph',

    propTypes: {
        // Series
        series: React.PropTypes.array.isRequired,
        seriesStyle: React.PropTypes.array,
        minValue: React.PropTypes.number,

        // Autofill
        autoFill: React.PropTypes.bool,
        autoFillValue: React.PropTypes.number,
        autoFillStartTime: React.PropTypes.number,
        autoFillEndTime: React.PropTypes.number,

        // Size
        width: React.PropTypes.number,
        height: React.PropTypes.number,
        padding: React.PropTypes.number,

        // Styling
        pointRadius: React.PropTypes.number,
        lineWidth: React.PropTypes.number,
        defaultColor: React.PropTypes.string,

        axeColor: React.PropTypes.string,
        axeMarkerWidth: React.PropTypes.number,

        textColor: React.PropTypes.string,
        textFontSize: React.PropTypes.number,
        textFontFamily: React.PropTypes.string
    },

    childContextTypes: {
        seriesStyle: React.PropTypes.array,
        pointRadius: React.PropTypes.number,
        defaultColor: React.PropTypes.string,
        lineWidth: React.PropTypes.number,

        axeColor: React.PropTypes.string,
        axeMarkerWidth: React.PropTypes.number,

        textColor: React.PropTypes.string,
        textFontSize: React.PropTypes.number,
        textFontFamily: React.PropTypes.string
    },

    getDefaultProps: function getDefaultProps() {
        return {
            // Autofill
            autoFill: false,
            autoFillValue: 0,
            autoFillInterval: 1000 * 60 * 60 * 24,
            autoFillStartTime: null,
            autoFillEndTime: null,

            // Size
            width: 800,
            height: 400,
            padding: 20,

            // Styling
            pointRadius: 4,
            lineWidth: 1,
            defaultColor: '#26cb7c',

            axeColor: '#eee',
            axeMarkerWidth: 10,

            textColor: '#aaa',
            textFontSize: 10,
            textFontFamily: 'helvetica'
        };
    },
    getChildContext: function getChildContext() {
        return {
            seriesStyle: this.props.seriesStyle,
            pointRadius: this.props.pointRadius,
            defaultColor: this.props.defaultColor,
            lineWidth: this.props.lineWidth,
            axeColor: this.props.axeColor,
            axeMarkerWidth: this.props.axeMarkerWidth,
            textColor: this.props.textColor,
            textFontSize: this.props.textFontSize,
            textFontFamily: this.props.textFontFamily
        };
    },
    getInitialState: function getInitialState() {
        return this.getStateFromProps(this.props);
    },
    componentWillReceiveProps: function componentWillReceiveProps(nextProps) {
        this.setState(this.getStateFromProps(nextProps));
    },
    getStateFromProps: function getStateFromProps(props) {
        var width = props.width;
        var height = props.height;
        var padding = props.padding;
        var minValue = props.minValue;
        var defaultColor = props.defaultColor;
        var seriesStyle = props.seriesStyle;
        var series = props.series;

        // Compute values ranges
        var dateMin = null;
        var dateMax = null;
        var valueMin = null;
        var valueMax = null;

        series = series.map(function (time, i) {
            // Set min/max dates
            var date = new Date(time.date).getTime();
            dateMin = Boolean(dateMin) ? Math.min(dateMin, date) : date;
            dateMax = Boolean(dateMax) ? Math.max(dateMax, date) : date;

            // Set min/max values
            var points = time.points;
            points.forEach(function (point) {
                var value = point.value;
                valueMin = Boolean(valueMin) ? Math.min(valueMin, value) : value;
                valueMax = Boolean(valueMax) ? Math.max(valueMax, value) : value;
            });

            return {
                date: date,
                points: points
            };
        });

        // Sort by date
        series.sort(function (a, b) {
            return a.date - b.date;
        });

        // Set minValue if set
        if (typeof minValue != 'undefined' && minValue < valueMin) {
            valueMin = minValue;
        }

        // Auto-fill
        if (props.autoFill) {
            (function () {
                var autoFillStartTime = props.autoFillStartTime;
                var autoFillEndTime = props.autoFillEndTime;
                var autoFillInterval = props.autoFillInterval;
                var autoFillValue = props.autoFillValue;

                // Set autoFill times to timestamps if provided
                if (Boolean(autoFillStartTime)) {
                    autoFillStartTime = new Date(autoFillStartTime).getTime();
                }
                if (Boolean(autoFillEndTime)) {
                    autoFillEndTime = new Date(autoFillEndTime).getTime();
                }
                // Set start and end time depending on options
                dateMin = Boolean(autoFillStartTime) ? autoFillStartTime : dateMin;
                dateMax = Boolean(autoFillEndTime) ? autoFillEndTime : dateMax;

                // Set endTime to construct serie
                var serieEndTime = Boolean(autoFillEndTime) ? dateMax : dateMax + autoFillInterval;

                // Set valueMin and valueMax
                valueMin = Math.min(valueMin, autoFillValue);
                valueMax = Math.max(valueMax, autoFillValue);

                // Construct time range
                var timeRange = [];
                for (var t = dateMin; t < serieEndTime; t += autoFillInterval) {
                    timeRange.push(t);
                }

                // Fill current serie with existing points or with autoFillValue
                var seriesIndex = 0;

                series = timeRange.map(function (time, i) {
                    var data = series[seriesIndex];

                    if (Boolean(data) && data.date == time) {
                        seriesIndex++;
                        return data;
                    } else {
                        // Construct missing points
                        return {
                            date: time,
                            points: seriesStyle.map(function (style, serieI) {
                                return {
                                    serie: style.title || serieI,
                                    value: autoFillValue,
                                    color: style.color || defaultColor
                                };
                            })
                        };
                    }
                });
            })();
        }

        if (valueMax == valueMin) {
            valueMax = valueMin + 1;
        }

        // Compute axe sizes
        var axeXHeight = props.textFontSize * 3;
        var axeYWidth = valueMax.toFixed(0).length * props.textFontSize * 1.5 + props.axeMarkerWidth;

        var axeXLength = width - 2 * padding - axeYWidth;
        var axeYLength = height - 2 * padding - axeXHeight;

        // Compute body size
        var innerX = axeYWidth + padding;
        var innerY = height - (axeXHeight + padding);

        var yBase = padding;
        var yTop = height - padding;

        // No tooltip by default
        var hovered = null;

        return {
            width: width,
            height: height,
            series: series,
            dateMin: dateMin,
            dateMax: dateMax,
            valueMin: valueMin,
            valueMax: valueMax,
            axeXLength: axeXLength,
            axeYLength: axeYLength,
            innerX: innerX,
            innerY: innerY,
            yTop: yTop,
            yBase: yBase,
            hovered: hovered
        };
    },
    setHovered: function setHovered(params) {
        this.setState({
            hovered: params
        });
    },
    renderTooltip: function renderTooltip() {
        var width = this.props.width;
        var hovered = this.state.hovered;

        if (!hovered) {
            return null;
        }

        var x = hovered.x,
            points = hovered.points;


        var sumPointsY = points.reduce(function (total, point) {
            return total + point.y;
        }, 0);
        var tooltipY = Math.floor(sumPointsY / points.length);

        // Pretty display date
        var pointsDateStr = new Date(points[0].date).toLocaleString();
        pointsDateStr = pointsDateStr.split(' ')[0].split('-').join('/');

        var tooltipStyle = {
            top: tooltipY - 20
        };
        // Set tooltip to right or left depending on values position in TimeGraph
        if (x > width / 2) {
            tooltipStyle.right = width - x + 15;
        } else {
            tooltipStyle.left = x + 15;
        }

        return React.createElement(
            'div',
            { className: 'time-graph-tooltip', style: tooltipStyle },
            React.createElement(
                'h6',
                { className: 'points-date' },
                React.createElement(DateSpan, { date: new Date(points[0].date), format: 'dddd, MMMM D, YYYY' })
            ),
            React.createElement(
                'table',
                { className: 'points-details' },
                React.createElement(
                    'tbody',
                    null,
                    React.createElement(
                        'tr',
                        { className: 'points-colors' },
                        points.map(function (point, index) {
                            return React.createElement('td', { key: index, style: { backgroundColor: point.color } });
                        })
                    ),
                    React.createElement(
                        'tr',
                        { className: 'points-values' },
                        points.map(function (point, index) {
                            return React.createElement(
                                'td',
                                { key: index },
                                React.createElement(
                                    'b',
                                    null,
                                    point.value
                                ),
                                ' ',
                                point.serie
                            );
                        })
                    )
                )
            )
        );
    },
    render: function render() {
        var _state = this.state,
            width = _state.width,
            height = _state.height,
            series = _state.series,
            dateMin = _state.dateMin,
            dateMax = _state.dateMax,
            valueMin = _state.valueMin,
            valueMax = _state.valueMax,
            axeXLength = _state.axeXLength,
            axeYLength = _state.axeYLength,
            innerX = _state.innerX,
            innerY = _state.innerY,
            yTop = _state.yTop,
            yBase = _state.yBase;


        return React.createElement(
            'div',
            { className: 'time-graph' },
            this.renderTooltip(),
            React.createElement(
                'svg',
                { width: width, height: height, viewBox: '0 0 ' + width + ' ' + height, preserveAspectRatio: 'xMidYMid meet' },
                React.createElement(XAxis, { length: axeXLength,
                    dateMin: dateMin,
                    dateMax: dateMax,
                    innerX: innerX,
                    yTop: yTop }),
                React.createElement(YAxis, { length: axeYLength,
                    valueMin: valueMin,
                    valueMax: valueMax,
                    innerX: innerX,
                    innerY: innerY,
                    yBase: yBase }),
                React.createElement(Body, { series: series,
                    dateMin: dateMin,
                    dateMax: dateMax,
                    valueMin: valueMin,
                    valueMax: valueMax,
                    width: axeXLength,
                    height: axeYLength,
                    innerX: innerX,
                    innerY: innerY,
                    yBase: yBase,
                    setHovered: this.setHovered })
            )
        );
    }
});

module.exports = TimeGraph;
//# sourceMappingURL=data:application/json;base64,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