'use strict';

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

function _objectWithoutProperties(obj, keys) { var target = {}; for (var i in obj) { if (keys.indexOf(i) >= 0) continue; if (!Object.prototype.hasOwnProperty.call(obj, i)) continue; target[i] = obj[i]; } return target; }

var React = require('react');
var classNames = require('classnames');

var Button = require('./Button');
var Icon = require('./Icon');
var Backdrop = require('./Backdrop');
var warning = require('./utils/warning');

/**
 * Dropdown (or up). Automatically bound to child Button.
 * See ButtonDropdown implementation to wrap something else than a button.
 *
 * <Dropdown>
 *
 *     <Button ... />
 *
 *     <Dropdown.Item header>Category 1</Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *
 *     <Dropdown.Item divider />
 *     <Dropdown.Item header>Category 2</Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *
 *     <Dropdown.Item>
 *         A submenu
 *         <Dropdown.Menu open={false}> // CSS takes care of display on hover
 *             <Dropdown.Item href={...}> Subitem </Dropdown.Item>
 *         </Dropdown.Menu>
 *     </Dropdown.Item>
 *
 * </Dropdown>
 */

var ButtonDropdown = React.createClass({
    displayName: 'ButtonDropdown',

    propTypes: {
        className: React.PropTypes.string,
        children: React.PropTypes.node,
        up: React.PropTypes.bool,
        width: React.PropTypes.string
    },

    getInitialState: function getInitialState() {
        return {
            open: false
        };
    },


    /**
     * Toggle the dopdown
     * @param  {Event} e?
     */
    toggle: function toggle(e) {
        if (e) {
            e.stopPropagation();
        }

        this.setState({
            open: !this.state.open
        });
    },


    /**
     * Close the dropdown
     */
    close: function close() {
        this.setState({
            open: false
        });
    },
    render: function render() {
        var that = this;
        var inner = [];
        var items = [];

        var _props = this.props,
            className = _props.className,
            children = _props.children,
            up = _props.up,
            width = _props.width,
            otherProps = _objectWithoutProperties(_props, ['className', 'children', 'up', 'width']);

        var open = this.state.open;


        className = classNames('dropdown', className, {
            'dropup': up
        });

        inner = React.Children.map(children, function (child) {
            // If the Button is connected through Redux.connect, it is
            // renamed to "Connect(Button...)"
            if (child && child.type && child.type.displayName && child.type.displayName.indexOf('Button') !== -1) {
                if (!child.props.onClick && !child.props.href) {
                    return React.cloneElement(child, {
                        onClick: that.toggle,
                        dropdownToggle: true
                    });
                }
                return child;
            }

            return null;
        });

        items = React.Children.map(children, function (child) {
            if (child && child.type && (child.type.displayName == 'DropdownItem' || child.type.displayName == 'DropdownDivider')) {
                return React.cloneElement(child, {
                    onClick: function onClick() {
                        if (child.props.onClick) {
                            child.props.onClick();
                        }
                        that.close();
                    }
                });
            }
            return null;
        });

        var content = React.createElement(
            Button.Group,
            _extends({}, otherProps, { className: className }),
            inner
        );

        // Wrap in a backdrop when open
        if (open) {
            return React.createElement(
                Backdrop,
                { wrapper: content, onClose: this.close },
                open ? React.createElement(
                    DropdownMenu,
                    { width: width },
                    items
                ) : null
            );
        } else {
            return content;
        }
    }
});

var DropdownItem = React.createClass({
    displayName: 'DropdownItem',

    propTypes: {
        children: React.PropTypes.node,
        onClick: React.PropTypes.func,
        href: React.PropTypes.string,
        disabled: React.PropTypes.bool,
        divider: React.PropTypes.bool,
        header: React.PropTypes.bool,
        checked: React.PropTypes.bool
    },

    onClick: function onClick(e) {
        if (!this.props.href) {
            e.preventDefault();
            e.stopPropagation();

            if (this.props.onClick) this.props.onClick();
        }
    },
    isInner: function isInner(child) {
        return !child || !child.type || child.type.displayName !== 'DropdownMenu';
    },
    render: function render() {
        var _props2 = this.props,
            divider = _props2.divider,
            header = _props2.header,
            checked = _props2.checked;


        if (divider) {
            warning('Prop "divider" on Dropdown.Item is deprecated, use Dropdown.Divider instead');
            return React.createElement(DropdownDivider, null);
        }
        if (header) {
            return React.createElement(
                'li',
                { className: 'dropdown-header' },
                this.props.children
            );
        }

        var inner = [],
            outer = [];

        inner = React.Children.map(this.props.children, function (child) {
            if (this.isInner(child)) return child;
            return null;
        }, this);

        outer = React.Children.map(this.props.children, function (child) {
            if (!this.isInner(child)) return child;
            return null;
        }, this);

        return React.createElement(
            'li',
            { className: this.props.disabled ? 'disabled' : '' },
            React.createElement(
                'a',
                _extends({}, this.props, { href: this.props.href || '#', onClick: this.props.disabled ? null : this.onClick }),
                checked ? React.createElement(
                    'div',
                    { className: 'dropdown-icon pull-left' },
                    React.createElement(Icon, { id: 'check' })
                ) : '',
                inner
            ),
            outer
        );
    }
});

var DropdownDivider = React.createClass({
    displayName: 'DropdownDivider',

    propTypes: {
        children: React.PropTypes.node
    },

    render: function render() {
        return React.createElement('li', { className: 'divider' });
    }
});

var DropdownMenu = React.createClass({
    displayName: 'DropdownMenu',

    propTypes: {
        className: React.PropTypes.string,
        children: React.PropTypes.node,
        open: React.PropTypes.bool,
        width: React.PropTypes.string
    },

    getDefaultProps: function getDefaultProps() {
        return {
            open: true,
            width: null
        };
    },
    render: function render() {
        var width = this.props.width;

        var className = classNames('dropdown-menu', width ? 'dropdown-' + width : '', {
            open: this.props.open
        });

        return React.createElement(
            'ul',
            { className: className },
            this.props.children
        );
    }
});

var ItemHeader = React.createClass({
    displayName: 'ItemHeader',

    propTypes: {
        children: React.PropTypes.node
    },

    render: function render() {
        return React.createElement(
            'div',
            { className: 'dropdown-itemheader' },
            this.props.children
        );
    }
});

var ItemDesc = React.createClass({
    displayName: 'ItemDesc',

    propTypes: {
        children: React.PropTypes.node
    },

    render: function render() {
        return React.createElement(
            'div',
            { className: 'dropdown-itemdesc' },
            this.props.children
        );
    }
});

module.exports = ButtonDropdown;
module.exports.Divider = DropdownDivider;
module.exports.Item = DropdownItem;
module.exports.Item.Header = ItemHeader;
module.exports.Item.Desc = ItemDesc;
module.exports.Menu = DropdownMenu;
//# sourceMappingURL=data:application/json;base64,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