'use strict';

var React = require('react');
var classNames = require('classnames');

var SIZES = require('./SIZES');
var Button = require('./Button');
var Input = require('./Input');
var Backdrop = require('./Backdrop');

var DEFAULT_SEARCH_PLACEHOLDER = 'Search';

var itemShape = React.PropTypes.oneOfType([React.PropTypes.string, React.PropTypes.object]);

var groupShape = React.PropTypes.shape({
    label: React.PropTypes.string,
    options: React.PropTypes.arrayOf(itemShape)
});

/**
 * Default filter for select
 */
function defaultFilter(query, item, i) {
    return true;
}

/**
 * Default render for options
 */
function defaultComponent(_ref) {
    var option = _ref.option;

    return React.createElement(
        'span',
        null,
        option
    );
}
defaultComponent.propTypes = {
    option: itemShape
};

/**
 * Default render to string for input
 */
function defaultRenderToString(item, i) {
    return String(item);
}

/**
 * Interractive select for forms
 *
 * It renders as a normal select on server and has a custom UI on browser (with search, images support).
 *
 * <Select name="test" options=[] />
 */
var Select = React.createClass({
    displayName: 'Select',

    propTypes: {
        // Current value of the select
        value: React.PropTypes.oneOfType([itemShape, React.PropTypes.arrayOf(itemShape)]),

        // List of items to display
        groups: React.PropTypes.arrayOf(groupShape),
        options: React.PropTypes.arrayOf(itemShape),

        // Function to render the option to a string or element
        component: React.PropTypes.func,

        // Function to render the selected option in the button
        // Defaults to "renderOption"
        componentSelection: React.PropTypes.func,

        // Function to output an option as a string
        // Defaults to a string representation, you have to provide your own value
        // when using a custom option renderer
        renderToString: React.PropTypes.func,

        // Function to filter an element
        filter: React.PropTypes.func,

        // Optional callback when value changed
        onChange: React.PropTypes.func,

        // Name when using server posting
        name: React.PropTypes.string,

        // Text to display when no value is set
        placeholder: React.PropTypes.string,
        searchPlaceholder: React.PropTypes.string,

        // Delimiter for multiple values
        delimiter: React.PropTypes.string,

        // Prevent selection
        disabled: React.PropTypes.bool,

        // Display the search filter?
        search: React.PropTypes.bool,

        // Accept multiple values
        multiple: React.PropTypes.bool,

        // Size of the select to display
        size: React.PropTypes.oneOf(SIZES),

        // Take the whole width
        block: React.PropTypes.bool
    },

    getDefaultProps: function getDefaultProps() {
        return {
            disabled: false,
            search: true,
            delimiter: ',',
            size: SIZES[0],
            multiple: false,
            block: false,
            filter: defaultFilter,
            component: defaultComponent,
            renderToString: defaultRenderToString,
            searchPlaceholder: DEFAULT_SEARCH_PLACEHOLDER,
            placeholder: 'Select'
        };
    },
    getInitialState: function getInitialState() {
        return {
            value: this.props.value,
            query: '',
            opened: false,
            groups: this.propsToGroups(this.props)
        };
    },
    componentWillReceiveProps: function componentWillReceiveProps(newProps) {
        this.setState({
            value: newProps.value,
            groups: this.propsToGroups(newProps),
            opened: newProps.disabled ? false : this.state.opened
        });
    },


    /**
     * Create list of groups from props
     * @param {Object} props
     * @return {Array<groupShape>}
     */
    propsToGroups: function propsToGroups(props) {
        var options = props.options,
            groups = props.groups;


        if (groups) {
            return groups;
        }

        return [{ options: options }];
    },


    /**
     * Search query changed
     */
    onSearchChanged: function onSearchChanged(e) {
        this.setState({
            query: e.target.value
        });
    },


    /**
     * Toggle (close/open) the select
     */
    onToggle: function onToggle() {
        this.setState({
            opened: !this.state.opened
        });
    },


    /**
     * Close the select
     */
    close: function close() {
        this.setState({
            opened: false
        });
    },


    /**
     * Open the select
     */
    open: function open() {
        this.setState({
            opened: false
        });
    },


    /**
     * Focus the search if open
     */
    focusOnOpen: function focusOnOpen() {
        if (this.state.opened) {
            this.focusSearch();
        }
    },
    componentDidUpdate: function componentDidUpdate() {
        this.focusOnOpen();
    },
    componentDidMount: function componentDidMount() {
        this.focusOnOpen();
    },


    /**
     * Toggle an option
     */
    onToggleOption: function onToggleOption(addValue, e) {
        if (e) {
            e.preventDefault();
        }

        var _state = this.state,
            value = _state.value,
            multiple = _state.multiple;
        var onChange = this.props.onChange;

        var newState = void 0,
            newValue = void 0;

        if (multiple) {
            newValue = value;

            // Add to selection if not yet selected
            if (!this.hasValue(addValue)) {
                newValue = value.concat([addValue]);
            } else if (value.length > 1) {
                // Unselect if many options are selected
                newValue.splice(newValue.indexOf(addValue), 1);
            }

            newState = {
                value: newValue
            };
        } else {
            newValue = addValue;

            newState = {
                value: addValue,
                opened: false
            };
        }

        this.setState(newState, function () {
            if (onChange) {
                onChange(newValue);
            }
        });
    },


    /**
     * Get current value as a string (for hidden input)
     * @return {String}
     */
    getStringValue: function getStringValue() {
        var renderToString = this.props.renderToString;
        var value = this.state.value;


        if (!value) {
            return '';
        }

        if (!this.props.multiple) {
            return renderToString(value);
        } else {
            return value.map(renderToString).join(this.props.delimiter);
        }
    },


    /**
     * Check if a value is selected
     * @param {String} value
     * @return {Boolean}
     */
    hasValue: function hasValue(value) {
        var currentValue = this.state.value;

        if (!this.props.multiple) {
            return currentValue === value;
        } else {
            return currentValue.indexOf(value) >= 0;
        }
    },


    /**
     * Focus the search input
     */
    focusSearch: function focusSearch() {
        var searchInput = this.refs.searchInput;

        if (!searchInput) {
            return;
        }

        searchInput.focus();
    },


    /**
     * Render button to open select
     */
    renderButton: function renderButton() {
        var _props = this.props,
            disabled = _props.disabled,
            block = _props.block,
            multiple = _props.multiple,
            placeholder = _props.placeholder;
        var _state2 = this.state,
            value = _state2.value,
            opened = _state2.opened;

        var ComponentSelection = this.props.componentSelection || this.props.component;

        var inner = void 0;

        if (value) {
            var values = multiple ? value : [value];
            inner = React.createElement(
                'span',
                { className: 'SelectSelections' },
                values.map(function (val, i) {
                    return React.createElement(
                        'span',
                        { key: i, className: 'SelectSelection' },
                        React.createElement(ComponentSelection, { option: val, index: i })
                    );
                })
            );
        } else {
            inner = React.createElement(
                'span',
                { className: 'SelectPlaceholder' },
                placeholder
            );
        }

        return React.createElement(
            Button,
            { size: this.props.size, block: block, disabled: disabled, active: opened, onClick: this.onToggle },
            inner,
            ' ',
            React.createElement(Button.Caret, null)
        );
    },


    /**
     * Render button to open select
     */
    renderSearch: function renderSearch() {
        var query = this.state.query;


        return React.createElement(
            'div',
            { className: 'SelectSearch' },
            React.createElement(Input, { ref: 'searchInput',
                value: query,
                onChange: this.onSearchChanged,
                placeholder: this.props.placeholder
            })
        );
    },


    /**
     * Render the options selector
     */
    renderGroup: function renderGroup(group, index) {
        var query = this.state.query;
        var filter = this.props.filter;

        var Component = this.props.component;
        var count = 0;

        var options = group.options.map(function (item, i) {
            var _this = this;

            if (!filter(query, item, i)) {
                return '';
            }

            count++;

            return React.createElement(
                'div',
                {
                    key: i,
                    className: classNames('SelectOption', { active: this.hasValue(item) }),
                    onClick: function onClick(e) {
                        return _this.onToggleOption(item);
                    }
                },
                React.createElement(Component, { option: item, index: i })
            );
        }, this);

        // Don't display empty groups (when filtered)
        if (count === 0) {
            return '';
        }

        return React.createElement(
            'div',
            { key: index, className: 'SelectOptGroup' },
            group.label ? React.createElement(
                'div',
                { className: 'GroupLabel' },
                group.label
            ) : '',
            React.createElement(
                'div',
                { className: 'GroupOptions' },
                options
            )
        );
    },


    /**
     * Render the groups
     */
    renderGroups: function renderGroups() {
        var _state3 = this.state,
            opened = _state3.opened,
            groups = _state3.groups;
        var search = this.props.search;


        var className = classNames('SelectContainer', {
            'open': opened
        });

        return React.createElement(
            'div',
            { className: className },
            search ? this.renderSearch() : '',
            React.createElement(
                'div',
                { className: 'SelectGroups' },
                groups.map(this.renderGroup)
            )
        );
    },
    render: function render() {
        var _props2 = this.props,
            name = _props2.name,
            block = _props2.block;
        var opened = this.state.opened;


        var className = classNames('SelectFormControl', {
            block: block
        });

        return React.createElement(
            'div',
            { className: className, onClick: function onClick(e) {
                    return e.stopPropagation();
                } },
            React.createElement('input', { type: 'hidden', name: name, value: this.getStringValue() }),
            this.renderButton(),
            opened ? React.createElement(
                Backdrop,
                { onClose: this.close },
                this.renderGroups()
            ) : ''
        );
    }
});

module.exports = Select;
//# sourceMappingURL=data:application/json;base64,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