'use strict';

var React = require('react');
var classNames = require('classnames');

var SIZES = require('./SIZES');
var Button = require('./Button');
var Input = require('./Input');
var Backdrop = require('./Backdrop');

var DEFAULT_SEARCH_PLACEHOLDER = 'Search';

var itemShape = React.PropTypes.oneOfType([React.PropTypes.string, React.PropTypes.object]);

var groupShape = React.PropTypes.shape({
    label: React.PropTypes.string,
    options: React.PropTypes.arrayOf(itemShape)
});

/**
 * Default filter for select
 */
function defaultFilter(query, item, i) {
    return true;
}

/**
 * Default render for options
 */
function defaultComponent(_ref) {
    var option = _ref.option;

    return React.createElement(
        'span',
        null,
        option
    );
}
defaultComponent.propTypes = {
    option: itemShape
};

/**
 * Default render to string for input
 */
function defaultRenderToString(item, i) {
    return String(item);
}

/**
 * Interractive select for forms
 *
 * It renders as a normal select on server and has a custom UI on browser (with search, images support).
 *
 * <Select name="test" options=[] />
 */
var Select = React.createClass({
    displayName: 'Select',

    propTypes: {
        // Current value of the select
        value: React.PropTypes.oneOfType([itemShape, React.PropTypes.arrayOf(itemShape)]),

        // List of items to display
        groups: React.PropTypes.arrayOf(groupShape),
        options: React.PropTypes.arrayOf(itemShape),

        // Function to render the option to a string or element
        component: React.PropTypes.func,

        // Function to render the selected option in the button
        // Defaults to "renderOption"
        componentSelection: React.PropTypes.func,

        // Function to output an option as a string
        // Defaults to a string representation, you have to provide your own value
        // when using a custom option renderer
        renderToString: React.PropTypes.func,

        // Function to filter an element
        filter: React.PropTypes.func,

        // Optional callback when value changed
        onChange: React.PropTypes.func,

        // Name when using server posting
        name: React.PropTypes.string,

        // Text to display when no value is set
        placeholder: React.PropTypes.string,
        searchPlaceholder: React.PropTypes.string,

        // Delimiter for multiple values
        delimiter: React.PropTypes.string,

        // Prevent selection
        disabled: React.PropTypes.bool,

        // Display the search filter?
        search: React.PropTypes.bool,

        // Accept multiple values
        multiple: React.PropTypes.bool,

        // Size of the select to display
        size: React.PropTypes.oneOf(SIZES),

        // Take the whole width
        block: React.PropTypes.bool,

        // Should an option be marked as disabled
        isOptionDisabled: React.PropTypes.func
    },

    getDefaultProps: function getDefaultProps() {
        return {
            disabled: false,
            search: true,
            delimiter: ',',
            size: SIZES[0],
            multiple: false,
            block: false,
            filter: defaultFilter,
            component: defaultComponent,
            renderToString: defaultRenderToString,
            searchPlaceholder: DEFAULT_SEARCH_PLACEHOLDER,
            placeholder: 'Select',
            isOptionDisabled: function isOptionDisabled() {
                return false;
            }
        };
    },
    getInitialState: function getInitialState() {
        return {
            value: this.props.value,
            query: '',
            opened: false,
            groups: this.propsToGroups(this.props)
        };
    },
    componentWillReceiveProps: function componentWillReceiveProps(newProps) {
        this.setState({
            value: newProps.value,
            groups: this.propsToGroups(newProps),
            opened: newProps.disabled ? false : this.state.opened
        });
    },


    /**
     * Create list of groups from props
     * @param {Object} props
     * @return {Array<groupShape>}
     */
    propsToGroups: function propsToGroups(props) {
        var options = props.options,
            groups = props.groups;


        if (groups) {
            return groups;
        }

        return [{ options: options }];
    },


    /**
     * Search query changed
     */
    onSearchChanged: function onSearchChanged(e) {
        this.setState({
            query: e.target.value
        });
    },


    /**
     * Toggle (close/open) the select
     */
    onToggle: function onToggle() {
        this.setState({
            opened: !this.state.opened
        });
    },


    /**
     * Close the select
     */
    close: function close() {
        this.setState({
            opened: false
        });
    },


    /**
     * Open the select
     */
    open: function open() {
        this.setState({
            opened: false
        });
    },


    /**
     * Focus the search if open
     */
    focusOnOpen: function focusOnOpen() {
        if (this.state.opened) {
            this.focusSearch();
        }
    },
    componentDidUpdate: function componentDidUpdate() {
        this.focusOnOpen();
    },
    componentDidMount: function componentDidMount() {
        this.focusOnOpen();
    },


    /**
     * Toggle an option
     */
    onToggleOption: function onToggleOption(addValue, e) {
        if (e) {
            e.preventDefault();
        }

        var _state = this.state,
            value = _state.value,
            multiple = _state.multiple;
        var onChange = this.props.onChange;

        var newState = void 0,
            newValue = void 0;

        if (multiple) {
            newValue = value;

            // Add to selection if not yet selected
            if (!this.hasValue(addValue)) {
                newValue = value.concat([addValue]);
            } else if (value.length > 1) {
                // Unselect if many options are selected
                newValue.splice(newValue.indexOf(addValue), 1);
            }

            newState = {
                value: newValue
            };
        } else {
            newValue = addValue;

            newState = {
                value: addValue,
                opened: false
            };
        }

        this.setState(newState, function () {
            if (onChange) {
                onChange(newValue);
            }
        });
    },


    /**
     * Get current value as a string (for hidden input)
     * @return {String}
     */
    getStringValue: function getStringValue() {
        var renderToString = this.props.renderToString;
        var value = this.state.value;


        if (!value) {
            return '';
        }

        if (!this.props.multiple) {
            return renderToString(value);
        } else {
            return value.map(renderToString).join(this.props.delimiter);
        }
    },


    /**
     * Check if a value is selected
     * @param {String} value
     * @return {Boolean}
     */
    hasValue: function hasValue(value) {
        var currentValue = this.state.value;

        if (!this.props.multiple) {
            return currentValue === value;
        } else {
            return currentValue.indexOf(value) >= 0;
        }
    },


    /**
     * Focus the search input
     */
    focusSearch: function focusSearch() {
        var searchInput = this.refs.searchInput;

        if (!searchInput) {
            return;
        }

        searchInput.focus();
    },


    /**
     * Render button to open select
     */
    renderButton: function renderButton() {
        var _props = this.props,
            disabled = _props.disabled,
            block = _props.block,
            multiple = _props.multiple,
            placeholder = _props.placeholder;
        var _state2 = this.state,
            value = _state2.value,
            opened = _state2.opened;

        var ComponentSelection = this.props.componentSelection || this.props.component;

        var inner = void 0;

        if (value) {
            var values = multiple ? value : [value];
            inner = React.createElement(
                'span',
                { className: 'SelectSelections' },
                values.map(function (val, i) {
                    return React.createElement(
                        'span',
                        { key: i, className: 'SelectSelection' },
                        React.createElement(ComponentSelection, { option: val, index: i })
                    );
                })
            );
        } else {
            inner = React.createElement(
                'span',
                { className: 'SelectPlaceholder' },
                placeholder
            );
        }

        return React.createElement(
            Button,
            { size: this.props.size, block: block, disabled: disabled, active: opened, onClick: this.onToggle },
            inner,
            ' ',
            React.createElement(Button.Caret, null)
        );
    },


    /**
     * Render button to open select
     */
    renderSearch: function renderSearch() {
        var query = this.state.query;


        return React.createElement(
            'div',
            { className: 'SelectSearch' },
            React.createElement(Input, { ref: 'searchInput',
                value: query,
                onChange: this.onSearchChanged,
                placeholder: this.props.placeholder
            })
        );
    },


    /**
     * Render the options selector
     */
    renderGroup: function renderGroup(group, index) {
        var _this = this;

        var query = this.state.query;
        var _props2 = this.props,
            filter = _props2.filter,
            isOptionDisabled = _props2.isOptionDisabled;

        var Component = this.props.component;
        var count = 0;

        var options = group.options.map(function (item, i) {
            if (!filter(query, item, i)) {
                return '';
            }

            count++;

            // Check if item should be displayed but marked as disabled
            var isDisabled = isOptionDisabled(item);

            var className = classNames('SelectOption', {
                active: _this.hasValue(item),
                disabled: isDisabled
            });

            return React.createElement(
                'div',
                {
                    key: i,
                    className: className,
                    onClick: function onClick(e) {
                        if (!isDisabled) {
                            _this.onToggleOption(item);
                        }
                    }
                },
                React.createElement(Component, { option: item, index: i })
            );
        });

        // Don't display empty groups (when filtered)
        if (count === 0) {
            return '';
        }

        return React.createElement(
            'div',
            { key: index, className: 'SelectOptGroup' },
            group.label ? React.createElement(
                'div',
                { className: 'GroupLabel' },
                group.label
            ) : '',
            React.createElement(
                'div',
                { className: 'GroupOptions' },
                options
            )
        );
    },


    /**
     * Render the groups
     */
    renderGroups: function renderGroups() {
        var _state3 = this.state,
            opened = _state3.opened,
            groups = _state3.groups;
        var search = this.props.search;


        var className = classNames('SelectContainer', {
            'open': opened
        });

        return React.createElement(
            'div',
            { className: className },
            search ? this.renderSearch() : '',
            React.createElement(
                'div',
                { className: 'SelectGroups' },
                groups.map(this.renderGroup)
            )
        );
    },
    render: function render() {
        var _props3 = this.props,
            name = _props3.name,
            block = _props3.block;
        var opened = this.state.opened;


        var className = classNames('SelectFormControl', {
            block: block
        });

        return React.createElement(
            'div',
            { className: className, onClick: function onClick(e) {
                    return e.stopPropagation();
                } },
            React.createElement('input', { type: 'hidden', name: name, value: this.getStringValue() }),
            this.renderButton(),
            opened ? React.createElement(
                Backdrop,
                { onClose: this.close },
                this.renderGroups()
            ) : ''
        );
    }
});

module.exports = Select;
//# sourceMappingURL=data:application/json;base64,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