'use strict';

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

function _objectWithoutProperties(obj, keys) { var target = {}; for (var i in obj) { if (keys.indexOf(i) >= 0) continue; if (!Object.prototype.hasOwnProperty.call(obj, i)) continue; target[i] = obj[i]; } return target; }

var React = require('react');
var classNames = require('classnames');

var Button = require('./Button');
var Icon = require('./Icon');
var Backdrop = require('./Backdrop');
var warning = require('./utils/warning');

/**
 * Dropdown (or up). Automatically bound to child Button.
 * See ButtonDropdown implementation to wrap something else than a button.
 *
 * <Dropdown>
 *
 *     <Button ... />
 *
 *     <Dropdown.Item header>Category 1</Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *
 *     <Dropdown.Item divider />
 *     <Dropdown.Item header>Category 2</Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *
 *     <Dropdown.Item>
 *         A submenu
 *         <Dropdown.Menu open={false}> // CSS takes care of display on hover
 *             <Dropdown.Item href={...}> Subitem </Dropdown.Item>
 *         </Dropdown.Menu>
 *     </Dropdown.Item>
 *
 * </Dropdown>
 */

var ButtonDropdown = React.createClass({
    displayName: 'ButtonDropdown',

    propTypes: {
        className: React.PropTypes.string,
        children: React.PropTypes.node,
        up: React.PropTypes.bool,
        width: React.PropTypes.string
    },

    getInitialState: function getInitialState() {
        return {
            open: false
        };
    },


    /**
     * Toggle the dopdown
     * @param  {Event} e?
     */
    toggle: function toggle(e) {
        if (e) {
            e.stopPropagation();
        }

        this.setState({
            open: !this.state.open
        });
    },


    /**
     * Close the dropdown
     */
    close: function close() {
        this.setState({
            open: false
        });
    },
    render: function render() {
        var that = this;
        var inner = [];
        var items = [];

        var _props = this.props,
            className = _props.className,
            children = _props.children,
            up = _props.up,
            width = _props.width,
            otherProps = _objectWithoutProperties(_props, ['className', 'children', 'up', 'width']);

        var open = this.state.open;


        className = classNames('dropdown', className, {
            'dropup': up
        });

        inner = React.Children.map(children, function (child) {
            // If the Button is connected through Redux.connect, it is
            // renamed to "Connect(Button...)"
            if (child && child.type && child.type.displayName && child.type.displayName.indexOf('Button') !== -1) {
                if (!child.props.onClick && !child.props.href) {
                    return React.cloneElement(child, {
                        onClick: that.toggle,
                        dropdownToggle: true
                    });
                }
                return child;
            }

            return null;
        });

        items = React.Children.map(children, function (child) {
            if (child && child.type && (child.type.displayName == 'DropdownItem' || child.type.displayName == 'DropdownDivider')) {
                return React.cloneElement(child, {
                    onClick: function onClick() {
                        if (child.props.onClick) {
                            child.props.onClick();
                        }
                        that.close();
                    }
                });
            }
            return null;
        });

        var content = React.createElement(
            Button.Group,
            _extends({}, otherProps, { className: className }),
            inner
        );

        // Wrap in a backdrop when open
        if (open) {
            return React.createElement(
                Backdrop,
                { wrapper: content, onClose: this.close },
                React.createElement(
                    DropdownMenu,
                    { width: width },
                    items
                )
            );
        } else {
            return content;
        }
    }
});

var DropdownItem = React.createClass({
    displayName: 'DropdownItem',

    propTypes: {
        children: React.PropTypes.node,
        onClick: React.PropTypes.func,
        href: React.PropTypes.string,
        disabled: React.PropTypes.bool,
        divider: React.PropTypes.bool,
        header: React.PropTypes.bool,
        checked: React.PropTypes.bool
    },

    onClick: function onClick(e) {
        if (!this.props.href) {
            e.preventDefault();
            e.stopPropagation();

            if (this.props.onClick) this.props.onClick();
        }
    },
    isInner: function isInner(child) {
        return !child || !child.type || child.type.displayName !== 'DropdownMenu';
    },
    render: function render() {
        var _props2 = this.props,
            divider = _props2.divider,
            header = _props2.header,
            checked = _props2.checked;


        if (divider) {
            warning('Prop "divider" on Dropdown.Item is deprecated, use Dropdown.Divider instead');
            return React.createElement(DropdownDivider, null);
        }
        if (header) {
            return React.createElement(
                'li',
                { className: 'dropdown-header' },
                this.props.children
            );
        }

        var inner = [],
            outer = [];

        inner = React.Children.map(this.props.children, function (child) {
            if (this.isInner(child)) return child;
            return null;
        }, this);

        outer = React.Children.map(this.props.children, function (child) {
            if (!this.isInner(child)) return child;
            return null;
        }, this);

        return React.createElement(
            'li',
            { className: this.props.disabled ? 'disabled' : '' },
            React.createElement(
                'a',
                _extends({}, this.props, { href: this.props.href || '#', onClick: this.props.disabled ? null : this.onClick }),
                checked ? React.createElement(
                    'div',
                    { className: 'dropdown-icon pull-left' },
                    React.createElement(Icon, { id: 'check' })
                ) : '',
                inner
            ),
            outer
        );
    }
});

var DropdownDivider = React.createClass({
    displayName: 'DropdownDivider',

    propTypes: {
        children: React.PropTypes.node
    },

    render: function render() {
        return React.createElement('li', { className: 'divider' });
    }
});

var DropdownMenu = React.createClass({
    displayName: 'DropdownMenu',

    propTypes: {
        className: React.PropTypes.string,
        children: React.PropTypes.node,
        open: React.PropTypes.bool,
        width: React.PropTypes.string
    },

    getDefaultProps: function getDefaultProps() {
        return {
            open: true,
            width: null
        };
    },
    render: function render() {
        var width = this.props.width;

        var className = classNames('dropdown-menu', width ? 'dropdown-' + width : '', {
            open: this.props.open
        });

        return React.createElement(
            'ul',
            { className: className },
            this.props.children
        );
    }
});

var ItemHeader = React.createClass({
    displayName: 'ItemHeader',

    propTypes: {
        children: React.PropTypes.node
    },

    render: function render() {
        return React.createElement(
            'div',
            { className: 'dropdown-itemheader' },
            this.props.children
        );
    }
});

var ItemDesc = React.createClass({
    displayName: 'ItemDesc',

    propTypes: {
        children: React.PropTypes.node
    },

    render: function render() {
        return React.createElement(
            'div',
            { className: 'dropdown-itemdesc' },
            this.props.children
        );
    }
});

module.exports = ButtonDropdown;
module.exports.Divider = DropdownDivider;
module.exports.Item = DropdownItem;
module.exports.Item.Header = ItemHeader;
module.exports.Item.Desc = ItemDesc;
module.exports.Menu = DropdownMenu;
//# sourceMappingURL=data:application/json;base64,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