'use strict';

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

function _objectWithoutProperties(obj, keys) { var target = {}; for (var i in obj) { if (keys.indexOf(i) >= 0) continue; if (!Object.prototype.hasOwnProperty.call(obj, i)) continue; target[i] = obj[i]; } return target; }

var React = require('react');
var classNames = require('classnames');

var SIZES = require('./SIZES');
var Button = require('./Button');
var Input = require('./Input');
var Backdrop = require('./Backdrop');

var DEFAULT_SEARCH_PLACEHOLDER = 'Search';

var itemShape = React.PropTypes.oneOfType([React.PropTypes.string, React.PropTypes.object]);

var groupShape = React.PropTypes.shape({
    label: React.PropTypes.string,
    options: React.PropTypes.arrayOf(itemShape)
});

/**
 * Default filter for select
 */
function defaultFilter(query, item, i) {
    return true;
}

/**
 * Default render to string for input
 */
function defaultRenderToString(item, i) {
    return String(item);
}

/**
 * Default render for options
 * @type {ReactClass}
 */
var DefaultComponent = React.createClass({
    displayName: 'DefaultComponent',

    propTypes: {
        option: itemShape
    },

    render: function render() {
        var option = this.props.option;

        return React.createElement(
            'span',
            null,
            option
        );
    }
});

/**
 * Component to render a Selection option
 * @type {ReactClass}
 */
var SelectOption = React.createClass({
    displayName: 'SelectOption',

    propTypes: {
        item: itemShape.isRequired,
        index: React.PropTypes.number.isRequired,
        // Function to render the option to a string or element
        component: React.PropTypes.func.isRequired,
        // Function to check if option is in selected values
        hasValue: React.PropTypes.func.isRequired,
        // Toggle an option in main Select state
        onToggleOption: React.PropTypes.func.isRequired,
        // Should an option be marked as disabled
        isOptionDisabled: React.PropTypes.func
    },

    getDefaultProps: function getDefaultProps() {
        return {
            isOptionDisabled: function isOptionDisabled() {
                return false;
            }
        };
    },
    render: function render() {
        var _props = this.props,
            item = _props.item,
            index = _props.index,
            isOptionDisabled = _props.isOptionDisabled,
            hasValue = _props.hasValue,
            onToggleOption = _props.onToggleOption,
            Component = _props.component;

        // Check if item should be displayed but marked as disabled

        var isDisabled = isOptionDisabled(item);

        var className = classNames('SelectOption', {
            active: hasValue(item),
            disabled: isDisabled
        });

        return React.createElement(
            'div',
            { className: className,
                onClick: function onClick(e) {
                    if (!isDisabled) {
                        onToggleOption(item);
                    }
                } },
            React.createElement(Component, { option: item, index: index })
        );
    }
});

/**
 * Component to render a Select option group
 * @type {ReactClass}
 */
var SelectOptGroup = React.createClass({
    displayName: 'SelectOptGroup',

    propTypes: {
        group: groupShape
    },

    render: function render() {
        var _props2 = this.props,
            group = _props2.group,
            props = _objectWithoutProperties(_props2, ['group']);

        return React.createElement(
            'div',
            { className: 'SelectOptGroup' },
            group.label ? React.createElement(
                'div',
                { className: 'GroupLabel' },
                group.label
            ) : null,
            React.createElement(
                'div',
                { className: 'GroupOptions' },
                group.options.map(function (item, i) {
                    return React.createElement(SelectOption, _extends({}, props, { key: i, item: item, index: i }));
                })
            )
        );
    }
});

/**
 * Interractive select for forms
 *
 * It renders as a normal select on server and has a custom UI on browser (with search, images support).
 *
 * <Select name="test" options=[] />
 */
var Select = React.createClass({
    displayName: 'Select',

    propTypes: {
        // Current value of the select
        value: React.PropTypes.oneOfType([itemShape, React.PropTypes.arrayOf(itemShape)]),

        // List of items to display
        groups: React.PropTypes.arrayOf(groupShape),
        options: React.PropTypes.arrayOf(itemShape),

        // Function to render the option to a string or element
        component: React.PropTypes.func,

        // Function to render a message when search did not return any results
        searchEmptyComponent: React.PropTypes.func,

        // Function to render the selected option in the button
        // Defaults to "renderOption"
        componentSelection: React.PropTypes.func,

        // Function to output an option as a string
        // Defaults to a string representation, you have to provide your own value
        // when using a custom option renderer
        renderToString: React.PropTypes.func,

        // Function to filter an element
        filter: React.PropTypes.func,

        // Optional callback when value changed
        onChange: React.PropTypes.func,

        // Name when using server posting
        name: React.PropTypes.string,

        // Text to display when no value is set
        placeholder: React.PropTypes.string,
        searchPlaceholder: React.PropTypes.string,

        // Delimiter for multiple values
        delimiter: React.PropTypes.string,

        // Prevent selection
        disabled: React.PropTypes.bool,

        // Display the search filter?
        search: React.PropTypes.bool,

        // Accept multiple values
        multiple: React.PropTypes.bool,

        // Size of the select to display
        size: React.PropTypes.oneOf(SIZES),

        // Take the whole width
        block: React.PropTypes.bool,

        // Should an option be marked as disabled
        isOptionDisabled: React.PropTypes.func
    },

    getDefaultProps: function getDefaultProps() {
        return {
            disabled: false,
            search: true,
            delimiter: ',',
            size: SIZES[0],
            multiple: false,
            block: false,
            filter: defaultFilter,
            component: DefaultComponent,
            renderToString: defaultRenderToString,
            searchPlaceholder: DEFAULT_SEARCH_PLACEHOLDER,
            placeholder: 'Select',
            searchEmptyComponent: null
        };
    },
    getInitialState: function getInitialState() {
        return {
            value: this.props.value,
            query: '',
            opened: false,
            groups: this.propsToGroups(this.props)
        };
    },
    componentWillReceiveProps: function componentWillReceiveProps(newProps) {
        this.setState({
            value: newProps.value,
            groups: this.propsToGroups(newProps),
            opened: newProps.disabled ? false : this.state.opened
        });
    },


    /**
     * Create list of groups from props
     * @param {Object} props
     * @return {Array<groupShape>}
     */
    propsToGroups: function propsToGroups(props) {
        var options = props.options,
            groups = props.groups;


        if (groups) {
            return groups;
        }

        return [{ options: options }];
    },


    /**
     * Search query changed
     */
    onSearchChanged: function onSearchChanged(e) {
        this.setState({
            query: e.target.value
        });
    },


    /**
     * Toggle (close/open) the select
     */
    onToggle: function onToggle() {
        this.setState({
            opened: !this.state.opened
        });
    },


    /**
     * Close the select
     */
    close: function close() {
        this.setState({
            opened: false
        });
    },


    /**
     * Open the select
     */
    open: function open() {
        this.setState({
            opened: false
        });
    },


    /**
     * Focus the search if open
     */
    focusOnOpen: function focusOnOpen() {
        if (this.state.opened) {
            this.focusSearch();
        }
    },
    componentDidUpdate: function componentDidUpdate() {
        this.focusOnOpen();
    },
    componentDidMount: function componentDidMount() {
        this.focusOnOpen();
    },


    /**
     * Toggle an option
     */
    onToggleOption: function onToggleOption(addValue, e) {
        if (e) {
            e.preventDefault();
        }

        var _state = this.state,
            value = _state.value,
            multiple = _state.multiple;
        var onChange = this.props.onChange;

        var newState = void 0,
            newValue = void 0;

        if (multiple) {
            newValue = value;

            // Add to selection if not yet selected
            if (!this.hasValue(addValue)) {
                newValue = value.concat([addValue]);
            } else if (value.length > 1) {
                // Unselect if many options are selected
                newValue.splice(newValue.indexOf(addValue), 1);
            }

            newState = {
                value: newValue
            };
        } else {
            newValue = addValue;

            newState = {
                value: addValue,
                opened: false
            };
        }

        this.setState(newState, function () {
            if (onChange) {
                onChange(newValue);
            }
        });
    },


    /**
     * Get current value as a string (for hidden input)
     * @return {String}
     */
    getStringValue: function getStringValue() {
        var renderToString = this.props.renderToString;
        var value = this.state.value;


        if (!value) {
            return '';
        }

        if (!this.props.multiple) {
            return renderToString(value);
        } else {
            return value.map(renderToString).join(this.props.delimiter);
        }
    },


    /**
     * Check if a value is selected
     * @param {String} value
     * @return {Boolean}
     */
    hasValue: function hasValue(value) {
        var currentValue = this.state.value;

        if (!this.props.multiple) {
            return currentValue === value;
        } else {
            return currentValue.indexOf(value) >= 0;
        }
    },


    /**
     * Focus the search input
     */
    focusSearch: function focusSearch() {
        var searchInput = this.refs.searchInput;

        if (!searchInput) {
            return;
        }

        searchInput.focus();
    },


    /**
     * Render button to open select
     */
    renderButton: function renderButton() {
        var _props3 = this.props,
            disabled = _props3.disabled,
            block = _props3.block,
            multiple = _props3.multiple,
            placeholder = _props3.placeholder;
        var _state2 = this.state,
            value = _state2.value,
            opened = _state2.opened;

        var ComponentSelection = this.props.componentSelection || this.props.component;

        var inner = void 0;

        if (value) {
            var values = multiple ? value : [value];
            inner = React.createElement(
                'span',
                { className: 'SelectSelections' },
                values.map(function (val, i) {
                    return React.createElement(
                        'span',
                        { key: i, className: 'SelectSelection' },
                        React.createElement(ComponentSelection, { option: val, index: i })
                    );
                })
            );
        } else {
            inner = React.createElement(
                'span',
                { className: 'SelectPlaceholder' },
                placeholder
            );
        }

        return React.createElement(
            Button,
            { size: this.props.size, block: block, disabled: disabled, active: opened, onClick: this.onToggle },
            inner,
            ' ',
            React.createElement(Button.Caret, null)
        );
    },


    /**
     * Render the groups
     */
    renderGroups: function renderGroups() {
        var _this = this;

        var _state3 = this.state,
            opened = _state3.opened,
            groups = _state3.groups,
            query = _state3.query;

        var _props4 = this.props,
            search = _props4.search,
            filter = _props4.filter,
            SearchEmptyComponent = _props4.searchEmptyComponent,
            props = _objectWithoutProperties(_props4, ['search', 'filter', 'searchEmptyComponent']);

        var className = classNames('SelectContainer', {
            'open': opened
        });

        // Filter empty groups based on search query
        var filteredGroups = groups.map(function (group) {
            return _extends({}, group, {
                options: group.options.filter(function (item, i) {
                    return filter(query, item, i);
                })
            });
        }).filter(function (group) {
            return group.options.length > 0;
        });

        return React.createElement(
            'div',
            { className: className },
            search ? React.createElement(
                'div',
                { className: 'SelectSearch' },
                React.createElement(Input, { ref: 'searchInput',
                    value: query,
                    onChange: this.onSearchChanged,
                    placeholder: this.props.placeholder
                })
            ) : null,
            Boolean(filteredGroups.length) ? React.createElement(
                'div',
                { className: 'SelectGroups' },
                filteredGroups.map(function (group, i) {
                    return React.createElement(SelectOptGroup, _extends({}, props, { key: i, group: group, hasValue: _this.hasValue, onToggleOption: _this.onToggleOption }));
                })
            ) : null,
            search && !filteredGroups.length ? React.createElement(
                'div',
                { className: 'SearchEmpty' },
                React.createElement(SearchEmptyComponent, { query: query })
            ) : null
        );
    },
    render: function render() {
        var _props5 = this.props,
            name = _props5.name,
            block = _props5.block;
        var opened = this.state.opened;


        var className = classNames('SelectFormControl', {
            block: block
        });

        return React.createElement(
            'div',
            { className: className, onClick: function onClick(e) {
                    return e.stopPropagation();
                } },
            React.createElement('input', { type: 'hidden', name: name, value: this.getStringValue() }),
            this.renderButton(),
            opened ? React.createElement(
                Backdrop,
                { onClose: this.close },
                this.renderGroups()
            ) : null
        );
    }
});

module.exports = Select;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uL3NyYy9TZWxlY3QuanMiXSwibmFtZXMiOlsiUmVhY3QiLCJyZXF1aXJlIiwiY2xhc3NOYW1lcyIsIlNJWkVTIiwiQnV0dG9uIiwiSW5wdXQiLCJCYWNrZHJvcCIsIkRFRkFVTFRfU0VBUkNIX1BMQUNFSE9MREVSIiwiaXRlbVNoYXBlIiwiUHJvcFR5cGVzIiwib25lT2ZUeXBlIiwic3RyaW5nIiwib2JqZWN0IiwiZ3JvdXBTaGFwZSIsInNoYXBlIiwibGFiZWwiLCJvcHRpb25zIiwiYXJyYXlPZiIsImRlZmF1bHRGaWx0ZXIiLCJxdWVyeSIsIml0ZW0iLCJpIiwiZGVmYXVsdFJlbmRlclRvU3RyaW5nIiwiU3RyaW5nIiwiRGVmYXVsdENvbXBvbmVudCIsImNyZWF0ZUNsYXNzIiwicHJvcFR5cGVzIiwib3B0aW9uIiwicmVuZGVyIiwicHJvcHMiLCJTZWxlY3RPcHRpb24iLCJpc1JlcXVpcmVkIiwiaW5kZXgiLCJudW1iZXIiLCJjb21wb25lbnQiLCJmdW5jIiwiaGFzVmFsdWUiLCJvblRvZ2dsZU9wdGlvbiIsImlzT3B0aW9uRGlzYWJsZWQiLCJnZXREZWZhdWx0UHJvcHMiLCJDb21wb25lbnQiLCJpc0Rpc2FibGVkIiwiY2xhc3NOYW1lIiwiYWN0aXZlIiwiZGlzYWJsZWQiLCJlIiwiU2VsZWN0T3B0R3JvdXAiLCJncm91cCIsIm1hcCIsIlNlbGVjdCIsInZhbHVlIiwiZ3JvdXBzIiwic2VhcmNoRW1wdHlDb21wb25lbnQiLCJjb21wb25lbnRTZWxlY3Rpb24iLCJyZW5kZXJUb1N0cmluZyIsImZpbHRlciIsIm9uQ2hhbmdlIiwibmFtZSIsInBsYWNlaG9sZGVyIiwic2VhcmNoUGxhY2Vob2xkZXIiLCJkZWxpbWl0ZXIiLCJib29sIiwic2VhcmNoIiwibXVsdGlwbGUiLCJzaXplIiwib25lT2YiLCJibG9jayIsImdldEluaXRpYWxTdGF0ZSIsIm9wZW5lZCIsInByb3BzVG9Hcm91cHMiLCJjb21wb25lbnRXaWxsUmVjZWl2ZVByb3BzIiwibmV3UHJvcHMiLCJzZXRTdGF0ZSIsInN0YXRlIiwib25TZWFyY2hDaGFuZ2VkIiwidGFyZ2V0Iiwib25Ub2dnbGUiLCJjbG9zZSIsIm9wZW4iLCJmb2N1c09uT3BlbiIsImZvY3VzU2VhcmNoIiwiY29tcG9uZW50RGlkVXBkYXRlIiwiY29tcG9uZW50RGlkTW91bnQiLCJhZGRWYWx1ZSIsInByZXZlbnREZWZhdWx0IiwibmV3U3RhdGUiLCJuZXdWYWx1ZSIsImNvbmNhdCIsImxlbmd0aCIsInNwbGljZSIsImluZGV4T2YiLCJnZXRTdHJpbmdWYWx1ZSIsImpvaW4iLCJjdXJyZW50VmFsdWUiLCJzZWFyY2hJbnB1dCIsInJlZnMiLCJmb2N1cyIsInJlbmRlckJ1dHRvbiIsIkNvbXBvbmVudFNlbGVjdGlvbiIsImlubmVyIiwidmFsdWVzIiwidmFsIiwicmVuZGVyR3JvdXBzIiwiU2VhcmNoRW1wdHlDb21wb25lbnQiLCJmaWx0ZXJlZEdyb3VwcyIsIkJvb2xlYW4iLCJzdG9wUHJvcGFnYXRpb24iLCJtb2R1bGUiLCJleHBvcnRzIl0sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSxJQUFNQSxRQUFRQyxRQUFRLE9BQVIsQ0FBZDtBQUNBLElBQU1DLGFBQWFELFFBQVEsWUFBUixDQUFuQjs7QUFFQSxJQUFNRSxRQUFRRixRQUFRLFNBQVIsQ0FBZDtBQUNBLElBQU1HLFNBQVNILFFBQVEsVUFBUixDQUFmO0FBQ0EsSUFBTUksUUFBUUosUUFBUSxTQUFSLENBQWQ7QUFDQSxJQUFNSyxXQUFXTCxRQUFRLFlBQVIsQ0FBakI7O0FBRUEsSUFBTU0sNkJBQTZCLFFBQW5DOztBQUVBLElBQU1DLFlBQVlSLE1BQU1TLFNBQU4sQ0FBZ0JDLFNBQWhCLENBQTBCLENBQ3hDVixNQUFNUyxTQUFOLENBQWdCRSxNQUR3QixFQUV4Q1gsTUFBTVMsU0FBTixDQUFnQkcsTUFGd0IsQ0FBMUIsQ0FBbEI7O0FBS0EsSUFBTUMsYUFBYWIsTUFBTVMsU0FBTixDQUFnQkssS0FBaEIsQ0FBc0I7QUFDckNDLFdBQU9mLE1BQU1TLFNBQU4sQ0FBZ0JFLE1BRGM7QUFFckNLLGFBQVNoQixNQUFNUyxTQUFOLENBQWdCUSxPQUFoQixDQUF3QlQsU0FBeEI7QUFGNEIsQ0FBdEIsQ0FBbkI7O0FBS0E7OztBQUdBLFNBQVNVLGFBQVQsQ0FBdUJDLEtBQXZCLEVBQThCQyxJQUE5QixFQUFvQ0MsQ0FBcEMsRUFBdUM7QUFDbkMsV0FBTyxJQUFQO0FBQ0g7O0FBRUQ7OztBQUdBLFNBQVNDLHFCQUFULENBQStCRixJQUEvQixFQUFxQ0MsQ0FBckMsRUFBd0M7QUFDcEMsV0FBT0UsT0FBT0gsSUFBUCxDQUFQO0FBQ0g7O0FBRUQ7Ozs7QUFJQSxJQUFNSSxtQkFBbUJ4QixNQUFNeUIsV0FBTixDQUFrQjtBQUFBOztBQUN2Q0MsZUFBVztBQUNQQyxnQkFBUW5CO0FBREQsS0FENEI7O0FBS3ZDb0IsVUFMdUMsb0JBSzlCO0FBQUEsWUFDR0QsTUFESCxHQUNjLEtBQUtFLEtBRG5CLENBQ0dGLE1BREg7O0FBRUwsZUFDSTtBQUFBO0FBQUE7QUFBT0E7QUFBUCxTQURKO0FBR0g7QUFWc0MsQ0FBbEIsQ0FBekI7O0FBYUE7Ozs7QUFJQSxJQUFNRyxlQUFlOUIsTUFBTXlCLFdBQU4sQ0FBa0I7QUFBQTs7QUFDbkNDLGVBQVc7QUFDUE4sY0FBTVosVUFBVXVCLFVBRFQ7QUFFUEMsZUFBT2hDLE1BQU1TLFNBQU4sQ0FBZ0J3QixNQUFoQixDQUF1QkYsVUFGdkI7QUFHUDtBQUNBRyxtQkFBV2xDLE1BQU1TLFNBQU4sQ0FBZ0IwQixJQUFoQixDQUFxQkosVUFKekI7QUFLUDtBQUNBSyxrQkFBVXBDLE1BQU1TLFNBQU4sQ0FBZ0IwQixJQUFoQixDQUFxQkosVUFOeEI7QUFPUDtBQUNBTSx3QkFBZ0JyQyxNQUFNUyxTQUFOLENBQWdCMEIsSUFBaEIsQ0FBcUJKLFVBUjlCO0FBU1A7QUFDQU8sMEJBQWtCdEMsTUFBTVMsU0FBTixDQUFnQjBCO0FBVjNCLEtBRHdCOztBQWNuQ0ksbUJBZG1DLDZCQWNqQjtBQUNkLGVBQU87QUFDSEQsOEJBQWtCO0FBQUEsdUJBQU0sS0FBTjtBQUFBO0FBRGYsU0FBUDtBQUdILEtBbEJrQztBQW9CbkNWLFVBcEJtQyxvQkFvQjFCO0FBQUEscUJBUUQsS0FBS0MsS0FSSjtBQUFBLFlBRURULElBRkMsVUFFREEsSUFGQztBQUFBLFlBR0RZLEtBSEMsVUFHREEsS0FIQztBQUFBLFlBSURNLGdCQUpDLFVBSURBLGdCQUpDO0FBQUEsWUFLREYsUUFMQyxVQUtEQSxRQUxDO0FBQUEsWUFNREMsY0FOQyxVQU1EQSxjQU5DO0FBQUEsWUFPVUcsU0FQVixVQU9ETixTQVBDOztBQVVMOztBQUNBLFlBQU1PLGFBQWFILGlCQUFpQmxCLElBQWpCLENBQW5COztBQUVBLFlBQU1zQixZQUFZeEMsV0FBVyxjQUFYLEVBQTJCO0FBQ3pDeUMsb0JBQVFQLFNBQVNoQixJQUFULENBRGlDO0FBRXpDd0Isc0JBQVVIO0FBRitCLFNBQTNCLENBQWxCOztBQUtBLGVBQ0k7QUFBQTtBQUFBLGNBQUssV0FBV0MsU0FBaEI7QUFDSSx5QkFBUyxpQkFBQ0csQ0FBRCxFQUFPO0FBQ1osd0JBQUksQ0FBQ0osVUFBTCxFQUFpQjtBQUNiSix1Q0FBZWpCLElBQWY7QUFDSDtBQUNKLGlCQUxMO0FBTUksZ0NBQUMsU0FBRCxJQUFXLFFBQVFBLElBQW5CLEVBQXlCLE9BQU9ZLEtBQWhDO0FBTkosU0FESjtBQVVIO0FBaERrQyxDQUFsQixDQUFyQjs7QUFtREE7Ozs7QUFJQSxJQUFNYyxpQkFBaUI5QyxNQUFNeUIsV0FBTixDQUFrQjtBQUFBOztBQUNyQ0MsZUFBVztBQUNQcUIsZUFBT2xDO0FBREEsS0FEMEI7O0FBS3JDZSxVQUxxQyxvQkFLNUI7QUFBQSxzQkFJRCxLQUFLQyxLQUpKO0FBQUEsWUFFRGtCLEtBRkMsV0FFREEsS0FGQztBQUFBLFlBR0VsQixLQUhGOztBQU1MLGVBQ0k7QUFBQTtBQUFBLGNBQUssV0FBVSxnQkFBZjtBQUNDa0Isa0JBQU1oQyxLQUFOLEdBQ0c7QUFBQTtBQUFBLGtCQUFLLFdBQVUsWUFBZjtBQUE2QmdDLHNCQUFNaEM7QUFBbkMsYUFESCxHQUVLLElBSE47QUFLSTtBQUFBO0FBQUEsa0JBQUssV0FBVSxjQUFmO0FBQ0NnQyxzQkFBTS9CLE9BQU4sQ0FBY2dDLEdBQWQsQ0FBa0IsVUFBQzVCLElBQUQsRUFBT0MsQ0FBUDtBQUFBLDJCQUNmLG9CQUFDLFlBQUQsZUFBa0JRLEtBQWxCLElBQXlCLEtBQUtSLENBQTlCLEVBQWlDLE1BQU1ELElBQXZDLEVBQTZDLE9BQU9DLENBQXBELElBRGU7QUFBQSxpQkFBbEI7QUFERDtBQUxKLFNBREo7QUFhSDtBQXhCb0MsQ0FBbEIsQ0FBdkI7O0FBMkJBOzs7Ozs7O0FBT0EsSUFBTTRCLFNBQVNqRCxNQUFNeUIsV0FBTixDQUFrQjtBQUFBOztBQUM3QkMsZUFBVztBQUNQO0FBQ0F3QixlQUFPbEQsTUFBTVMsU0FBTixDQUFnQkMsU0FBaEIsQ0FBMEIsQ0FDN0JGLFNBRDZCLEVBRTdCUixNQUFNUyxTQUFOLENBQWdCUSxPQUFoQixDQUF3QlQsU0FBeEIsQ0FGNkIsQ0FBMUIsQ0FGQTs7QUFPUDtBQUNBMkMsZ0JBQVFuRCxNQUFNUyxTQUFOLENBQWdCUSxPQUFoQixDQUF3QkosVUFBeEIsQ0FSRDtBQVNQRyxpQkFBU2hCLE1BQU1TLFNBQU4sQ0FBZ0JRLE9BQWhCLENBQXdCVCxTQUF4QixDQVRGOztBQVdQO0FBQ0EwQixtQkFBV2xDLE1BQU1TLFNBQU4sQ0FBZ0IwQixJQVpwQjs7QUFjUDtBQUNBaUIsOEJBQXNCcEQsTUFBTVMsU0FBTixDQUFnQjBCLElBZi9COztBQWlCUDtBQUNBO0FBQ0FrQiw0QkFBb0JyRCxNQUFNUyxTQUFOLENBQWdCMEIsSUFuQjdCOztBQXFCUDtBQUNBO0FBQ0E7QUFDQW1CLHdCQUFnQnRELE1BQU1TLFNBQU4sQ0FBZ0IwQixJQXhCekI7O0FBMEJQO0FBQ0FvQixnQkFBUXZELE1BQU1TLFNBQU4sQ0FBZ0IwQixJQTNCakI7O0FBNkJQO0FBQ0FxQixrQkFBVXhELE1BQU1TLFNBQU4sQ0FBZ0IwQixJQTlCbkI7O0FBZ0NQO0FBQ0FzQixjQUFNekQsTUFBTVMsU0FBTixDQUFnQkUsTUFqQ2Y7O0FBbUNQO0FBQ0ErQyxxQkFBYTFELE1BQU1TLFNBQU4sQ0FBZ0JFLE1BcEN0QjtBQXFDUGdELDJCQUFtQjNELE1BQU1TLFNBQU4sQ0FBZ0JFLE1BckM1Qjs7QUF1Q1A7QUFDQWlELG1CQUFXNUQsTUFBTVMsU0FBTixDQUFnQkUsTUF4Q3BCOztBQTBDUDtBQUNBaUMsa0JBQVU1QyxNQUFNUyxTQUFOLENBQWdCb0QsSUEzQ25COztBQTZDUDtBQUNBQyxnQkFBUTlELE1BQU1TLFNBQU4sQ0FBZ0JvRCxJQTlDakI7O0FBZ0RQO0FBQ0FFLGtCQUFVL0QsTUFBTVMsU0FBTixDQUFnQm9ELElBakRuQjs7QUFtRFA7QUFDQUcsY0FBTWhFLE1BQU1TLFNBQU4sQ0FBZ0J3RCxLQUFoQixDQUFzQjlELEtBQXRCLENBcERDOztBQXNEUDtBQUNBK0QsZUFBT2xFLE1BQU1TLFNBQU4sQ0FBZ0JvRCxJQXZEaEI7O0FBeURQO0FBQ0F2QiwwQkFBa0J0QyxNQUFNUyxTQUFOLENBQWdCMEI7QUExRDNCLEtBRGtCOztBQThEN0JJLG1CQTlENkIsNkJBOERYO0FBQ2QsZUFBTztBQUNISyxzQkFBVSxLQURQO0FBRUhrQixvQkFBUSxJQUZMO0FBR0hGLHVCQUFXLEdBSFI7QUFJSEksa0JBQU03RCxNQUFNLENBQU4sQ0FKSDtBQUtINEQsc0JBQVUsS0FMUDtBQU1IRyxtQkFBTyxLQU5KO0FBT0hYLG9CQUFRckMsYUFQTDtBQVFIZ0IsdUJBQVdWLGdCQVJSO0FBU0g4Qiw0QkFBZ0JoQyxxQkFUYjtBQVVIcUMsK0JBQW1CcEQsMEJBVmhCO0FBV0htRCx5QkFBYSxRQVhWO0FBWUhOLGtDQUFzQjtBQVpuQixTQUFQO0FBY0gsS0E3RTRCO0FBK0U3QmUsbUJBL0U2Qiw2QkErRVg7QUFDZCxlQUFPO0FBQ0hqQixtQkFBTyxLQUFLckIsS0FBTCxDQUFXcUIsS0FEZjtBQUVIL0IsbUJBQU8sRUFGSjtBQUdIaUQsb0JBQVEsS0FITDtBQUlIakIsb0JBQVEsS0FBS2tCLGFBQUwsQ0FBbUIsS0FBS3hDLEtBQXhCO0FBSkwsU0FBUDtBQU1ILEtBdEY0QjtBQXdGN0J5Qyw2QkF4RjZCLHFDQXdGSEMsUUF4RkcsRUF3Rk87QUFDaEMsYUFBS0MsUUFBTCxDQUFjO0FBQ1Z0QixtQkFBT3FCLFNBQVNyQixLQUROO0FBRVZDLG9CQUFRLEtBQUtrQixhQUFMLENBQW1CRSxRQUFuQixDQUZFO0FBR1ZILG9CQUFRRyxTQUFTM0IsUUFBVCxHQUFvQixLQUFwQixHQUE0QixLQUFLNkIsS0FBTCxDQUFXTDtBQUhyQyxTQUFkO0FBS0gsS0E5RjRCOzs7QUFnRzdCOzs7OztBQUtBQyxpQkFyRzZCLHlCQXFHZnhDLEtBckdlLEVBcUdSO0FBQUEsWUFDVGIsT0FEUyxHQUNXYSxLQURYLENBQ1RiLE9BRFM7QUFBQSxZQUNBbUMsTUFEQSxHQUNXdEIsS0FEWCxDQUNBc0IsTUFEQTs7O0FBR2pCLFlBQUlBLE1BQUosRUFBWTtBQUNSLG1CQUFPQSxNQUFQO0FBQ0g7O0FBRUQsZUFBTyxDQUNILEVBQUVuQyxnQkFBRixFQURHLENBQVA7QUFHSCxLQS9HNEI7OztBQWlIN0I7OztBQUdBMEQsbUJBcEg2QiwyQkFvSGI3QixDQXBIYSxFQW9IVjtBQUNmLGFBQUsyQixRQUFMLENBQWM7QUFDVnJELG1CQUFPMEIsRUFBRThCLE1BQUYsQ0FBU3pCO0FBRE4sU0FBZDtBQUdILEtBeEg0Qjs7O0FBMEg3Qjs7O0FBR0EwQixZQTdINkIsc0JBNkhsQjtBQUNQLGFBQUtKLFFBQUwsQ0FBYztBQUNWSixvQkFBUSxDQUFDLEtBQUtLLEtBQUwsQ0FBV0w7QUFEVixTQUFkO0FBR0gsS0FqSTRCOzs7QUFtSTdCOzs7QUFHQVMsU0F0STZCLG1CQXNJckI7QUFDSixhQUFLTCxRQUFMLENBQWM7QUFDVkosb0JBQVE7QUFERSxTQUFkO0FBR0gsS0ExSTRCOzs7QUE0STdCOzs7QUFHQVUsUUEvSTZCLGtCQStJdEI7QUFDSCxhQUFLTixRQUFMLENBQWM7QUFDVkosb0JBQVE7QUFERSxTQUFkO0FBR0gsS0FuSjRCOzs7QUFxSjdCOzs7QUFHQVcsZUF4SjZCLHlCQXdKZjtBQUNWLFlBQUksS0FBS04sS0FBTCxDQUFXTCxNQUFmLEVBQXVCO0FBQ25CLGlCQUFLWSxXQUFMO0FBQ0g7QUFDSixLQTVKNEI7QUE4SjdCQyxzQkE5SjZCLGdDQThKUjtBQUNqQixhQUFLRixXQUFMO0FBQ0gsS0FoSzRCO0FBa0s3QkcscUJBbEs2QiwrQkFrS1Q7QUFDaEIsYUFBS0gsV0FBTDtBQUNILEtBcEs0Qjs7O0FBc0s3Qjs7O0FBR0ExQyxrQkF6SzZCLDBCQXlLZDhDLFFBektjLEVBeUtKdEMsQ0F6S0ksRUF5S0Q7QUFDeEIsWUFBSUEsQ0FBSixFQUFPO0FBQ0hBLGNBQUV1QyxjQUFGO0FBQ0g7O0FBSHVCLHFCQUtJLEtBQUtYLEtBTFQ7QUFBQSxZQUtoQnZCLEtBTGdCLFVBS2hCQSxLQUxnQjtBQUFBLFlBS1RhLFFBTFMsVUFLVEEsUUFMUztBQUFBLFlBTWhCUCxRQU5nQixHQU1ILEtBQUszQixLQU5GLENBTWhCMkIsUUFOZ0I7O0FBT3hCLFlBQUk2QixpQkFBSjtBQUFBLFlBQWNDLGlCQUFkOztBQUVBLFlBQUl2QixRQUFKLEVBQWM7QUFDVnVCLHVCQUFXcEMsS0FBWDs7QUFFQTtBQUNBLGdCQUFJLENBQUMsS0FBS2QsUUFBTCxDQUFjK0MsUUFBZCxDQUFMLEVBQThCO0FBQzFCRywyQkFBV3BDLE1BQU1xQyxNQUFOLENBQWEsQ0FBQ0osUUFBRCxDQUFiLENBQVg7QUFDSCxhQUZELE1BRU8sSUFBSWpDLE1BQU1zQyxNQUFOLEdBQWUsQ0FBbkIsRUFBc0I7QUFDekI7QUFDQUYseUJBQVNHLE1BQVQsQ0FDSUgsU0FBU0ksT0FBVCxDQUFpQlAsUUFBakIsQ0FESixFQUVJLENBRko7QUFJSDs7QUFFREUsdUJBQVc7QUFDUG5DLHVCQUFPb0M7QUFEQSxhQUFYO0FBR0gsU0FqQkQsTUFpQk87QUFDSEEsdUJBQVdILFFBQVg7O0FBRUFFLHVCQUFXO0FBQ1BuQyx1QkFBUWlDLFFBREQ7QUFFUGYsd0JBQVE7QUFGRCxhQUFYO0FBSUg7O0FBRUQsYUFBS0ksUUFBTCxDQUNJYSxRQURKLEVBRUksWUFBTTtBQUNGLGdCQUFJN0IsUUFBSixFQUFjO0FBQ1ZBLHlCQUFTOEIsUUFBVDtBQUNIO0FBQ0osU0FOTDtBQVFILEtBcE40Qjs7O0FBc043Qjs7OztBQUlBSyxrQkExTjZCLDRCQTBOWjtBQUFBLFlBQ0xyQyxjQURLLEdBQ2MsS0FBS3pCLEtBRG5CLENBQ0x5QixjQURLO0FBQUEsWUFFTEosS0FGSyxHQUVLLEtBQUt1QixLQUZWLENBRUx2QixLQUZLOzs7QUFJYixZQUFJLENBQUNBLEtBQUwsRUFBWTtBQUNSLG1CQUFPLEVBQVA7QUFDSDs7QUFFRCxZQUFJLENBQUMsS0FBS3JCLEtBQUwsQ0FBV2tDLFFBQWhCLEVBQTBCO0FBQ3RCLG1CQUFPVCxlQUFlSixLQUFmLENBQVA7QUFDSCxTQUZELE1BRU87QUFDSCxtQkFBT0EsTUFDTkYsR0FETSxDQUNGTSxjQURFLEVBRU5zQyxJQUZNLENBRUQsS0FBSy9ELEtBQUwsQ0FBVytCLFNBRlYsQ0FBUDtBQUdIO0FBQ0osS0F6TzRCOzs7QUEyTzdCOzs7OztBQUtBeEIsWUFoUDZCLG9CQWdQcEJjLEtBaFBvQixFQWdQYjtBQUNaLFlBQU0yQyxlQUFlLEtBQUtwQixLQUFMLENBQVd2QixLQUFoQzs7QUFFQSxZQUFJLENBQUMsS0FBS3JCLEtBQUwsQ0FBV2tDLFFBQWhCLEVBQTBCO0FBQ3RCLG1CQUFROEIsaUJBQWlCM0MsS0FBekI7QUFDSCxTQUZELE1BRU87QUFDSCxtQkFBUTJDLGFBQWFILE9BQWIsQ0FBcUJ4QyxLQUFyQixLQUErQixDQUF2QztBQUNIO0FBQ0osS0F4UDRCOzs7QUEwUDdCOzs7QUFHQThCLGVBN1A2Qix5QkE2UGY7QUFBQSxZQUNGYyxXQURFLEdBQ2MsS0FBS0MsSUFEbkIsQ0FDRkQsV0FERTs7QUFFVixZQUFJLENBQUNBLFdBQUwsRUFBa0I7QUFDZDtBQUNIOztBQUVEQSxvQkFBWUUsS0FBWjtBQUNILEtBcFE0Qjs7O0FBc1E3Qjs7O0FBR0FDLGdCQXpRNkIsMEJBeVFkO0FBQUEsc0JBQ3NDLEtBQUtwRSxLQUQzQztBQUFBLFlBQ0xlLFFBREssV0FDTEEsUUFESztBQUFBLFlBQ0tzQixLQURMLFdBQ0tBLEtBREw7QUFBQSxZQUNZSCxRQURaLFdBQ1lBLFFBRFo7QUFBQSxZQUNzQkwsV0FEdEIsV0FDc0JBLFdBRHRCO0FBQUEsc0JBRWEsS0FBS2UsS0FGbEI7QUFBQSxZQUVMdkIsS0FGSyxXQUVMQSxLQUZLO0FBQUEsWUFFRWtCLE1BRkYsV0FFRUEsTUFGRjs7QUFHWCxZQUFJOEIscUJBQXFCLEtBQUtyRSxLQUFMLENBQVd3QixrQkFBWCxJQUFpQyxLQUFLeEIsS0FBTCxDQUFXSyxTQUFyRTs7QUFFQSxZQUFJaUUsY0FBSjs7QUFFQSxZQUFJakQsS0FBSixFQUFXO0FBQ1AsZ0JBQU1rRCxTQUFTckMsV0FBV2IsS0FBWCxHQUFtQixDQUFDQSxLQUFELENBQWxDO0FBQ0FpRCxvQkFDSTtBQUFBO0FBQUEsa0JBQU0sV0FBVSxrQkFBaEI7QUFDQ0MsdUJBQU9wRCxHQUFQLENBQVcsVUFBQ3FELEdBQUQsRUFBTWhGLENBQU47QUFBQSwyQkFDUjtBQUFBO0FBQUEsMEJBQU0sS0FBS0EsQ0FBWCxFQUFjLFdBQVUsaUJBQXhCO0FBQ0ksNENBQUMsa0JBQUQsSUFBb0IsUUFBUWdGLEdBQTVCLEVBQWlDLE9BQU9oRixDQUF4QztBQURKLHFCQURRO0FBQUEsaUJBQVg7QUFERCxhQURKO0FBU0gsU0FYRCxNQVdPO0FBQ0g4RSxvQkFBUTtBQUFBO0FBQUEsa0JBQU0sV0FBVSxtQkFBaEI7QUFBcUN6QztBQUFyQyxhQUFSO0FBQ0g7O0FBRUQsZUFDSTtBQUFDLGtCQUFEO0FBQUEsY0FBUSxNQUFNLEtBQUs3QixLQUFMLENBQVdtQyxJQUF6QixFQUErQixPQUFPRSxLQUF0QyxFQUE2QyxVQUFVdEIsUUFBdkQsRUFBaUUsUUFBUXdCLE1BQXpFLEVBQWlGLFNBQVMsS0FBS1EsUUFBL0Y7QUFDS3VCLGlCQURMO0FBQUE7QUFDWSxnQ0FBQyxNQUFELENBQVEsS0FBUjtBQURaLFNBREo7QUFLSCxLQXBTNEI7OztBQXNTN0I7OztBQUdBRyxnQkF6UzZCLDBCQXlTZDtBQUFBOztBQUFBLHNCQUN1QixLQUFLN0IsS0FENUI7QUFBQSxZQUNITCxNQURHLFdBQ0hBLE1BREc7QUFBQSxZQUNLakIsTUFETCxXQUNLQSxNQURMO0FBQUEsWUFDYWhDLEtBRGIsV0FDYUEsS0FEYjs7QUFBQSxzQkFPTixLQUFLVSxLQVBDO0FBQUEsWUFHUGlDLE1BSE8sV0FHUEEsTUFITztBQUFBLFlBSVBQLE1BSk8sV0FJUEEsTUFKTztBQUFBLFlBS2VnRCxvQkFMZixXQUtQbkQsb0JBTE87QUFBQSxZQU1KdkIsS0FOSTs7QUFTWCxZQUFNYSxZQUFZeEMsV0FBVyxpQkFBWCxFQUE4QjtBQUM1QyxvQkFBUWtFO0FBRG9DLFNBQTlCLENBQWxCOztBQUlBO0FBQ0EsWUFBTW9DLGlCQUFpQnJELE9BQU9ILEdBQVAsQ0FBVyxVQUFDRCxLQUFEO0FBQUEsZ0NBQzNCQSxLQUQyQjtBQUU5Qi9CLHlCQUFTK0IsTUFBTS9CLE9BQU4sQ0FBY3VDLE1BQWQsQ0FBcUIsVUFBQ25DLElBQUQsRUFBT0MsQ0FBUDtBQUFBLDJCQUFha0MsT0FBT3BDLEtBQVAsRUFBY0MsSUFBZCxFQUFvQkMsQ0FBcEIsQ0FBYjtBQUFBLGlCQUFyQjtBQUZxQjtBQUFBLFNBQVgsRUFJdEJrQyxNQUpzQixDQUlmLFVBQUNSLEtBQUQ7QUFBQSxtQkFBV0EsTUFBTS9CLE9BQU4sQ0FBY3dFLE1BQWQsR0FBdUIsQ0FBbEM7QUFBQSxTQUplLENBQXZCOztBQU1BLGVBQ0k7QUFBQTtBQUFBLGNBQUssV0FBVzlDLFNBQWhCO0FBQ0NvQixxQkFDRztBQUFBO0FBQUEsa0JBQUssV0FBVSxjQUFmO0FBQ0ksb0NBQUMsS0FBRCxJQUFPLEtBQUksYUFBWDtBQUNJLDJCQUFPM0MsS0FEWDtBQUVJLDhCQUFVLEtBQUt1RCxlQUZuQjtBQUdJLGlDQUFhLEtBQUs3QyxLQUFMLENBQVc2QjtBQUg1QjtBQURKLGFBREgsR0FRSyxJQVROO0FBWUMrQyxvQkFBUUQsZUFBZWhCLE1BQXZCLElBQ0c7QUFBQTtBQUFBLGtCQUFLLFdBQVUsY0FBZjtBQUNDZ0IsK0JBQWV4RCxHQUFmLENBQW1CLFVBQUNELEtBQUQsRUFBUTFCLENBQVI7QUFBQSwyQkFDaEIsb0JBQUMsY0FBRCxlQUFvQlEsS0FBcEIsSUFBMkIsS0FBS1IsQ0FBaEMsRUFBbUMsT0FBTzBCLEtBQTFDLEVBQWlELFVBQVUsTUFBS1gsUUFBaEUsRUFBMEUsZ0JBQWdCLE1BQUtDLGNBQS9GLElBRGdCO0FBQUEsaUJBQW5CO0FBREQsYUFESCxHQU1LLElBbEJOO0FBcUJDeUIsc0JBQVUsQ0FBQzBDLGVBQWVoQixNQUExQixHQUNHO0FBQUE7QUFBQSxrQkFBSyxXQUFVLGFBQWY7QUFDSSxvQ0FBQyxvQkFBRCxJQUFzQixPQUFPckUsS0FBN0I7QUFESixhQURILEdBSUs7QUF6Qk4sU0FESjtBQThCSCxLQTNWNEI7QUE2VjdCUyxVQTdWNkIsb0JBNlZwQjtBQUFBLHNCQUNpQixLQUFLQyxLQUR0QjtBQUFBLFlBQ0M0QixJQURELFdBQ0NBLElBREQ7QUFBQSxZQUNPUyxLQURQLFdBQ09BLEtBRFA7QUFBQSxZQUVHRSxNQUZILEdBRWMsS0FBS0ssS0FGbkIsQ0FFR0wsTUFGSDs7O0FBSUwsWUFBSTFCLFlBQVl4QyxXQUFXLG1CQUFYLEVBQWdDO0FBQzVDZ0U7QUFENEMsU0FBaEMsQ0FBaEI7O0FBSUEsZUFDSTtBQUFBO0FBQUEsY0FBSyxXQUFXeEIsU0FBaEIsRUFBMkIsU0FBUyxpQkFBQ0csQ0FBRDtBQUFBLDJCQUFPQSxFQUFFNkQsZUFBRixFQUFQO0FBQUEsaUJBQXBDO0FBQ0ksMkNBQU8sTUFBSyxRQUFaLEVBQXFCLE1BQU1qRCxJQUEzQixFQUFpQyxPQUFPLEtBQUtrQyxjQUFMLEVBQXhDLEdBREo7QUFFSyxpQkFBS00sWUFBTCxFQUZMO0FBR0M3QixxQkFDRztBQUFDLHdCQUFEO0FBQUEsa0JBQVUsU0FBUyxLQUFLUyxLQUF4QjtBQUNLLHFCQUFLeUIsWUFBTDtBQURMLGFBREgsR0FJSztBQVBOLFNBREo7QUFZSDtBQWpYNEIsQ0FBbEIsQ0FBZjs7QUFvWEFLLE9BQU9DLE9BQVAsR0FBaUIzRCxNQUFqQiIsImZpbGUiOiJTZWxlY3QuanMiLCJzb3VyY2VzQ29udGVudCI6WyJjb25zdCBSZWFjdCA9IHJlcXVpcmUoJ3JlYWN0Jyk7XG5jb25zdCBjbGFzc05hbWVzID0gcmVxdWlyZSgnY2xhc3NuYW1lcycpO1xuXG5jb25zdCBTSVpFUyA9IHJlcXVpcmUoJy4vU0laRVMnKTtcbmNvbnN0IEJ1dHRvbiA9IHJlcXVpcmUoJy4vQnV0dG9uJyk7XG5jb25zdCBJbnB1dCA9IHJlcXVpcmUoJy4vSW5wdXQnKTtcbmNvbnN0IEJhY2tkcm9wID0gcmVxdWlyZSgnLi9CYWNrZHJvcCcpO1xuXG5jb25zdCBERUZBVUxUX1NFQVJDSF9QTEFDRUhPTERFUiA9ICdTZWFyY2gnO1xuXG5jb25zdCBpdGVtU2hhcGUgPSBSZWFjdC5Qcm9wVHlwZXMub25lT2ZUeXBlKFtcbiAgICBSZWFjdC5Qcm9wVHlwZXMuc3RyaW5nLFxuICAgIFJlYWN0LlByb3BUeXBlcy5vYmplY3Rcbl0pO1xuXG5jb25zdCBncm91cFNoYXBlID0gUmVhY3QuUHJvcFR5cGVzLnNoYXBlKHtcbiAgICBsYWJlbDogUmVhY3QuUHJvcFR5cGVzLnN0cmluZyxcbiAgICBvcHRpb25zOiBSZWFjdC5Qcm9wVHlwZXMuYXJyYXlPZihpdGVtU2hhcGUpXG59KTtcblxuLyoqXG4gKiBEZWZhdWx0IGZpbHRlciBmb3Igc2VsZWN0XG4gKi9cbmZ1bmN0aW9uIGRlZmF1bHRGaWx0ZXIocXVlcnksIGl0ZW0sIGkpIHtcbiAgICByZXR1cm4gdHJ1ZTtcbn1cblxuLyoqXG4gKiBEZWZhdWx0IHJlbmRlciB0byBzdHJpbmcgZm9yIGlucHV0XG4gKi9cbmZ1bmN0aW9uIGRlZmF1bHRSZW5kZXJUb1N0cmluZyhpdGVtLCBpKSB7XG4gICAgcmV0dXJuIFN0cmluZyhpdGVtKTtcbn1cblxuLyoqXG4gKiBEZWZhdWx0IHJlbmRlciBmb3Igb3B0aW9uc1xuICogQHR5cGUge1JlYWN0Q2xhc3N9XG4gKi9cbmNvbnN0IERlZmF1bHRDb21wb25lbnQgPSBSZWFjdC5jcmVhdGVDbGFzcyh7XG4gICAgcHJvcFR5cGVzOiB7XG4gICAgICAgIG9wdGlvbjogaXRlbVNoYXBlXG4gICAgfSxcblxuICAgIHJlbmRlcigpIHtcbiAgICAgICAgY29uc3QgeyBvcHRpb24gfSA9IHRoaXMucHJvcHM7XG4gICAgICAgIHJldHVybiAoXG4gICAgICAgICAgICA8c3Bhbj57b3B0aW9ufTwvc3Bhbj5cbiAgICAgICAgKTtcbiAgICB9XG59KTtcblxuLyoqXG4gKiBDb21wb25lbnQgdG8gcmVuZGVyIGEgU2VsZWN0aW9uIG9wdGlvblxuICogQHR5cGUge1JlYWN0Q2xhc3N9XG4gKi9cbmNvbnN0IFNlbGVjdE9wdGlvbiA9IFJlYWN0LmNyZWF0ZUNsYXNzKHtcbiAgICBwcm9wVHlwZXM6IHtcbiAgICAgICAgaXRlbTogaXRlbVNoYXBlLmlzUmVxdWlyZWQsXG4gICAgICAgIGluZGV4OiBSZWFjdC5Qcm9wVHlwZXMubnVtYmVyLmlzUmVxdWlyZWQsXG4gICAgICAgIC8vIEZ1bmN0aW9uIHRvIHJlbmRlciB0aGUgb3B0aW9uIHRvIGEgc3RyaW5nIG9yIGVsZW1lbnRcbiAgICAgICAgY29tcG9uZW50OiBSZWFjdC5Qcm9wVHlwZXMuZnVuYy5pc1JlcXVpcmVkLFxuICAgICAgICAvLyBGdW5jdGlvbiB0byBjaGVjayBpZiBvcHRpb24gaXMgaW4gc2VsZWN0ZWQgdmFsdWVzXG4gICAgICAgIGhhc1ZhbHVlOiBSZWFjdC5Qcm9wVHlwZXMuZnVuYy5pc1JlcXVpcmVkLFxuICAgICAgICAvLyBUb2dnbGUgYW4gb3B0aW9uIGluIG1haW4gU2VsZWN0IHN0YXRlXG4gICAgICAgIG9uVG9nZ2xlT3B0aW9uOiBSZWFjdC5Qcm9wVHlwZXMuZnVuYy5pc1JlcXVpcmVkLFxuICAgICAgICAvLyBTaG91bGQgYW4gb3B0aW9uIGJlIG1hcmtlZCBhcyBkaXNhYmxlZFxuICAgICAgICBpc09wdGlvbkRpc2FibGVkOiBSZWFjdC5Qcm9wVHlwZXMuZnVuY1xuICAgIH0sXG5cbiAgICBnZXREZWZhdWx0UHJvcHMoKSB7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBpc09wdGlvbkRpc2FibGVkOiAoKSA9PiBmYWxzZVxuICAgICAgICB9O1xuICAgIH0sXG5cbiAgICByZW5kZXIoKSB7XG4gICAgICAgIGNvbnN0IHtcbiAgICAgICAgICAgIGl0ZW0sXG4gICAgICAgICAgICBpbmRleCxcbiAgICAgICAgICAgIGlzT3B0aW9uRGlzYWJsZWQsXG4gICAgICAgICAgICBoYXNWYWx1ZSxcbiAgICAgICAgICAgIG9uVG9nZ2xlT3B0aW9uLFxuICAgICAgICAgICAgY29tcG9uZW50OiBDb21wb25lbnRcbiAgICAgICAgfSA9IHRoaXMucHJvcHM7XG5cbiAgICAgICAgLy8gQ2hlY2sgaWYgaXRlbSBzaG91bGQgYmUgZGlzcGxheWVkIGJ1dCBtYXJrZWQgYXMgZGlzYWJsZWRcbiAgICAgICAgY29uc3QgaXNEaXNhYmxlZCA9IGlzT3B0aW9uRGlzYWJsZWQoaXRlbSk7XG5cbiAgICAgICAgY29uc3QgY2xhc3NOYW1lID0gY2xhc3NOYW1lcygnU2VsZWN0T3B0aW9uJywge1xuICAgICAgICAgICAgYWN0aXZlOiBoYXNWYWx1ZShpdGVtKSxcbiAgICAgICAgICAgIGRpc2FibGVkOiBpc0Rpc2FibGVkXG4gICAgICAgIH0pO1xuXG4gICAgICAgIHJldHVybiAoXG4gICAgICAgICAgICA8ZGl2IGNsYXNzTmFtZT17Y2xhc3NOYW1lfVxuICAgICAgICAgICAgICAgIG9uQ2xpY2s9eyhlKSA9PiB7XG4gICAgICAgICAgICAgICAgICAgIGlmICghaXNEaXNhYmxlZCkge1xuICAgICAgICAgICAgICAgICAgICAgICAgb25Ub2dnbGVPcHRpb24oaXRlbSk7XG4gICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICB9fT5cbiAgICAgICAgICAgICAgICA8Q29tcG9uZW50IG9wdGlvbj17aXRlbX0gaW5kZXg9e2luZGV4fSAvPlxuICAgICAgICAgICAgPC9kaXY+XG4gICAgICAgICk7XG4gICAgfVxufSk7XG5cbi8qKlxuICogQ29tcG9uZW50IHRvIHJlbmRlciBhIFNlbGVjdCBvcHRpb24gZ3JvdXBcbiAqIEB0eXBlIHtSZWFjdENsYXNzfVxuICovXG5jb25zdCBTZWxlY3RPcHRHcm91cCA9IFJlYWN0LmNyZWF0ZUNsYXNzKHtcbiAgICBwcm9wVHlwZXM6IHtcbiAgICAgICAgZ3JvdXA6IGdyb3VwU2hhcGVcbiAgICB9LFxuXG4gICAgcmVuZGVyKCkge1xuICAgICAgICBjb25zdCB7XG4gICAgICAgICAgICBncm91cCxcbiAgICAgICAgICAgIC4uLnByb3BzXG4gICAgICAgIH0gPSB0aGlzLnByb3BzO1xuXG4gICAgICAgIHJldHVybiAoXG4gICAgICAgICAgICA8ZGl2IGNsYXNzTmFtZT1cIlNlbGVjdE9wdEdyb3VwXCI+XG4gICAgICAgICAgICB7Z3JvdXAubGFiZWwgP1xuICAgICAgICAgICAgICAgIDxkaXYgY2xhc3NOYW1lPVwiR3JvdXBMYWJlbFwiPntncm91cC5sYWJlbH08L2Rpdj5cbiAgICAgICAgICAgICAgICA6IG51bGxcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICA8ZGl2IGNsYXNzTmFtZT1cIkdyb3VwT3B0aW9uc1wiPlxuICAgICAgICAgICAgICAgIHtncm91cC5vcHRpb25zLm1hcCgoaXRlbSwgaSkgPT4gKFxuICAgICAgICAgICAgICAgICAgICA8U2VsZWN0T3B0aW9uIHsuLi5wcm9wc30ga2V5PXtpfSBpdGVtPXtpdGVtfSBpbmRleD17aX0gLz5cbiAgICAgICAgICAgICAgICApKX1cbiAgICAgICAgICAgICAgICA8L2Rpdj5cbiAgICAgICAgICAgIDwvZGl2PlxuICAgICAgICApO1xuICAgIH1cbn0pO1xuXG4vKipcbiAqIEludGVycmFjdGl2ZSBzZWxlY3QgZm9yIGZvcm1zXG4gKlxuICogSXQgcmVuZGVycyBhcyBhIG5vcm1hbCBzZWxlY3Qgb24gc2VydmVyIGFuZCBoYXMgYSBjdXN0b20gVUkgb24gYnJvd3NlciAod2l0aCBzZWFyY2gsIGltYWdlcyBzdXBwb3J0KS5cbiAqXG4gKiA8U2VsZWN0IG5hbWU9XCJ0ZXN0XCIgb3B0aW9ucz1bXSAvPlxuICovXG5jb25zdCBTZWxlY3QgPSBSZWFjdC5jcmVhdGVDbGFzcyh7XG4gICAgcHJvcFR5cGVzOiB7XG4gICAgICAgIC8vIEN1cnJlbnQgdmFsdWUgb2YgdGhlIHNlbGVjdFxuICAgICAgICB2YWx1ZTogUmVhY3QuUHJvcFR5cGVzLm9uZU9mVHlwZShbXG4gICAgICAgICAgICBpdGVtU2hhcGUsXG4gICAgICAgICAgICBSZWFjdC5Qcm9wVHlwZXMuYXJyYXlPZihpdGVtU2hhcGUpXG4gICAgICAgIF0pLFxuXG4gICAgICAgIC8vIExpc3Qgb2YgaXRlbXMgdG8gZGlzcGxheVxuICAgICAgICBncm91cHM6IFJlYWN0LlByb3BUeXBlcy5hcnJheU9mKGdyb3VwU2hhcGUpLFxuICAgICAgICBvcHRpb25zOiBSZWFjdC5Qcm9wVHlwZXMuYXJyYXlPZihpdGVtU2hhcGUpLFxuXG4gICAgICAgIC8vIEZ1bmN0aW9uIHRvIHJlbmRlciB0aGUgb3B0aW9uIHRvIGEgc3RyaW5nIG9yIGVsZW1lbnRcbiAgICAgICAgY29tcG9uZW50OiBSZWFjdC5Qcm9wVHlwZXMuZnVuYyxcblxuICAgICAgICAvLyBGdW5jdGlvbiB0byByZW5kZXIgYSBtZXNzYWdlIHdoZW4gc2VhcmNoIGRpZCBub3QgcmV0dXJuIGFueSByZXN1bHRzXG4gICAgICAgIHNlYXJjaEVtcHR5Q29tcG9uZW50OiBSZWFjdC5Qcm9wVHlwZXMuZnVuYyxcblxuICAgICAgICAvLyBGdW5jdGlvbiB0byByZW5kZXIgdGhlIHNlbGVjdGVkIG9wdGlvbiBpbiB0aGUgYnV0dG9uXG4gICAgICAgIC8vIERlZmF1bHRzIHRvIFwicmVuZGVyT3B0aW9uXCJcbiAgICAgICAgY29tcG9uZW50U2VsZWN0aW9uOiBSZWFjdC5Qcm9wVHlwZXMuZnVuYyxcblxuICAgICAgICAvLyBGdW5jdGlvbiB0byBvdXRwdXQgYW4gb3B0aW9uIGFzIGEgc3RyaW5nXG4gICAgICAgIC8vIERlZmF1bHRzIHRvIGEgc3RyaW5nIHJlcHJlc2VudGF0aW9uLCB5b3UgaGF2ZSB0byBwcm92aWRlIHlvdXIgb3duIHZhbHVlXG4gICAgICAgIC8vIHdoZW4gdXNpbmcgYSBjdXN0b20gb3B0aW9uIHJlbmRlcmVyXG4gICAgICAgIHJlbmRlclRvU3RyaW5nOiBSZWFjdC5Qcm9wVHlwZXMuZnVuYyxcblxuICAgICAgICAvLyBGdW5jdGlvbiB0byBmaWx0ZXIgYW4gZWxlbWVudFxuICAgICAgICBmaWx0ZXI6IFJlYWN0LlByb3BUeXBlcy5mdW5jLFxuXG4gICAgICAgIC8vIE9wdGlvbmFsIGNhbGxiYWNrIHdoZW4gdmFsdWUgY2hhbmdlZFxuICAgICAgICBvbkNoYW5nZTogUmVhY3QuUHJvcFR5cGVzLmZ1bmMsXG5cbiAgICAgICAgLy8gTmFtZSB3aGVuIHVzaW5nIHNlcnZlciBwb3N0aW5nXG4gICAgICAgIG5hbWU6IFJlYWN0LlByb3BUeXBlcy5zdHJpbmcsXG5cbiAgICAgICAgLy8gVGV4dCB0byBkaXNwbGF5IHdoZW4gbm8gdmFsdWUgaXMgc2V0XG4gICAgICAgIHBsYWNlaG9sZGVyOiBSZWFjdC5Qcm9wVHlwZXMuc3RyaW5nLFxuICAgICAgICBzZWFyY2hQbGFjZWhvbGRlcjogUmVhY3QuUHJvcFR5cGVzLnN0cmluZyxcblxuICAgICAgICAvLyBEZWxpbWl0ZXIgZm9yIG11bHRpcGxlIHZhbHVlc1xuICAgICAgICBkZWxpbWl0ZXI6IFJlYWN0LlByb3BUeXBlcy5zdHJpbmcsXG5cbiAgICAgICAgLy8gUHJldmVudCBzZWxlY3Rpb25cbiAgICAgICAgZGlzYWJsZWQ6IFJlYWN0LlByb3BUeXBlcy5ib29sLFxuXG4gICAgICAgIC8vIERpc3BsYXkgdGhlIHNlYXJjaCBmaWx0ZXI/XG4gICAgICAgIHNlYXJjaDogUmVhY3QuUHJvcFR5cGVzLmJvb2wsXG5cbiAgICAgICAgLy8gQWNjZXB0IG11bHRpcGxlIHZhbHVlc1xuICAgICAgICBtdWx0aXBsZTogUmVhY3QuUHJvcFR5cGVzLmJvb2wsXG5cbiAgICAgICAgLy8gU2l6ZSBvZiB0aGUgc2VsZWN0IHRvIGRpc3BsYXlcbiAgICAgICAgc2l6ZTogUmVhY3QuUHJvcFR5cGVzLm9uZU9mKFNJWkVTKSxcblxuICAgICAgICAvLyBUYWtlIHRoZSB3aG9sZSB3aWR0aFxuICAgICAgICBibG9jazogUmVhY3QuUHJvcFR5cGVzLmJvb2wsXG5cbiAgICAgICAgLy8gU2hvdWxkIGFuIG9wdGlvbiBiZSBtYXJrZWQgYXMgZGlzYWJsZWRcbiAgICAgICAgaXNPcHRpb25EaXNhYmxlZDogUmVhY3QuUHJvcFR5cGVzLmZ1bmNcbiAgICB9LFxuXG4gICAgZ2V0RGVmYXVsdFByb3BzKCkge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgZGlzYWJsZWQ6IGZhbHNlLFxuICAgICAgICAgICAgc2VhcmNoOiB0cnVlLFxuICAgICAgICAgICAgZGVsaW1pdGVyOiAnLCcsXG4gICAgICAgICAgICBzaXplOiBTSVpFU1swXSxcbiAgICAgICAgICAgIG11bHRpcGxlOiBmYWxzZSxcbiAgICAgICAgICAgIGJsb2NrOiBmYWxzZSxcbiAgICAgICAgICAgIGZpbHRlcjogZGVmYXVsdEZpbHRlcixcbiAgICAgICAgICAgIGNvbXBvbmVudDogRGVmYXVsdENvbXBvbmVudCxcbiAgICAgICAgICAgIHJlbmRlclRvU3RyaW5nOiBkZWZhdWx0UmVuZGVyVG9TdHJpbmcsXG4gICAgICAgICAgICBzZWFyY2hQbGFjZWhvbGRlcjogREVGQVVMVF9TRUFSQ0hfUExBQ0VIT0xERVIsXG4gICAgICAgICAgICBwbGFjZWhvbGRlcjogJ1NlbGVjdCcsXG4gICAgICAgICAgICBzZWFyY2hFbXB0eUNvbXBvbmVudDogbnVsbFxuICAgICAgICB9O1xuICAgIH0sXG5cbiAgICBnZXRJbml0aWFsU3RhdGUoKSB7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICB2YWx1ZTogdGhpcy5wcm9wcy52YWx1ZSxcbiAgICAgICAgICAgIHF1ZXJ5OiAnJyxcbiAgICAgICAgICAgIG9wZW5lZDogZmFsc2UsXG4gICAgICAgICAgICBncm91cHM6IHRoaXMucHJvcHNUb0dyb3Vwcyh0aGlzLnByb3BzKVxuICAgICAgICB9O1xuICAgIH0sXG5cbiAgICBjb21wb25lbnRXaWxsUmVjZWl2ZVByb3BzKG5ld1Byb3BzKSB7XG4gICAgICAgIHRoaXMuc2V0U3RhdGUoe1xuICAgICAgICAgICAgdmFsdWU6IG5ld1Byb3BzLnZhbHVlLFxuICAgICAgICAgICAgZ3JvdXBzOiB0aGlzLnByb3BzVG9Hcm91cHMobmV3UHJvcHMpLFxuICAgICAgICAgICAgb3BlbmVkOiBuZXdQcm9wcy5kaXNhYmxlZCA/IGZhbHNlIDogdGhpcy5zdGF0ZS5vcGVuZWRcbiAgICAgICAgfSk7XG4gICAgfSxcblxuICAgIC8qKlxuICAgICAqIENyZWF0ZSBsaXN0IG9mIGdyb3VwcyBmcm9tIHByb3BzXG4gICAgICogQHBhcmFtIHtPYmplY3R9IHByb3BzXG4gICAgICogQHJldHVybiB7QXJyYXk8Z3JvdXBTaGFwZT59XG4gICAgICovXG4gICAgcHJvcHNUb0dyb3Vwcyhwcm9wcykge1xuICAgICAgICBjb25zdCB7IG9wdGlvbnMsIGdyb3VwcyB9ID0gcHJvcHM7XG5cbiAgICAgICAgaWYgKGdyb3Vwcykge1xuICAgICAgICAgICAgcmV0dXJuIGdyb3VwcztcbiAgICAgICAgfVxuXG4gICAgICAgIHJldHVybiBbXG4gICAgICAgICAgICB7IG9wdGlvbnMgfVxuICAgICAgICBdO1xuICAgIH0sXG5cbiAgICAvKipcbiAgICAgKiBTZWFyY2ggcXVlcnkgY2hhbmdlZFxuICAgICAqL1xuICAgIG9uU2VhcmNoQ2hhbmdlZChlKSB7XG4gICAgICAgIHRoaXMuc2V0U3RhdGUoe1xuICAgICAgICAgICAgcXVlcnk6IGUudGFyZ2V0LnZhbHVlXG4gICAgICAgIH0pO1xuICAgIH0sXG5cbiAgICAvKipcbiAgICAgKiBUb2dnbGUgKGNsb3NlL29wZW4pIHRoZSBzZWxlY3RcbiAgICAgKi9cbiAgICBvblRvZ2dsZSgpIHtcbiAgICAgICAgdGhpcy5zZXRTdGF0ZSh7XG4gICAgICAgICAgICBvcGVuZWQ6ICF0aGlzLnN0YXRlLm9wZW5lZFxuICAgICAgICB9KTtcbiAgICB9LFxuXG4gICAgLyoqXG4gICAgICogQ2xvc2UgdGhlIHNlbGVjdFxuICAgICAqL1xuICAgIGNsb3NlKCkge1xuICAgICAgICB0aGlzLnNldFN0YXRlKHtcbiAgICAgICAgICAgIG9wZW5lZDogZmFsc2VcbiAgICAgICAgfSk7XG4gICAgfSxcblxuICAgIC8qKlxuICAgICAqIE9wZW4gdGhlIHNlbGVjdFxuICAgICAqL1xuICAgIG9wZW4oKSB7XG4gICAgICAgIHRoaXMuc2V0U3RhdGUoe1xuICAgICAgICAgICAgb3BlbmVkOiBmYWxzZVxuICAgICAgICB9KTtcbiAgICB9LFxuXG4gICAgLyoqXG4gICAgICogRm9jdXMgdGhlIHNlYXJjaCBpZiBvcGVuXG4gICAgICovXG4gICAgZm9jdXNPbk9wZW4oKSB7XG4gICAgICAgIGlmICh0aGlzLnN0YXRlLm9wZW5lZCkge1xuICAgICAgICAgICAgdGhpcy5mb2N1c1NlYXJjaCgpO1xuICAgICAgICB9XG4gICAgfSxcblxuICAgIGNvbXBvbmVudERpZFVwZGF0ZSgpIHtcbiAgICAgICAgdGhpcy5mb2N1c09uT3BlbigpO1xuICAgIH0sXG5cbiAgICBjb21wb25lbnREaWRNb3VudCgpIHtcbiAgICAgICAgdGhpcy5mb2N1c09uT3BlbigpO1xuICAgIH0sXG5cbiAgICAvKipcbiAgICAgKiBUb2dnbGUgYW4gb3B0aW9uXG4gICAgICovXG4gICAgb25Ub2dnbGVPcHRpb24oYWRkVmFsdWUsIGUpIHtcbiAgICAgICAgaWYgKGUpIHtcbiAgICAgICAgICAgIGUucHJldmVudERlZmF1bHQoKTtcbiAgICAgICAgfVxuXG4gICAgICAgIGNvbnN0IHsgdmFsdWUsIG11bHRpcGxlIH0gPSB0aGlzLnN0YXRlO1xuICAgICAgICBjb25zdCB7IG9uQ2hhbmdlIH0gPSB0aGlzLnByb3BzO1xuICAgICAgICBsZXQgbmV3U3RhdGUsIG5ld1ZhbHVlO1xuXG4gICAgICAgIGlmIChtdWx0aXBsZSkge1xuICAgICAgICAgICAgbmV3VmFsdWUgPSB2YWx1ZTtcblxuICAgICAgICAgICAgLy8gQWRkIHRvIHNlbGVjdGlvbiBpZiBub3QgeWV0IHNlbGVjdGVkXG4gICAgICAgICAgICBpZiAoIXRoaXMuaGFzVmFsdWUoYWRkVmFsdWUpKSB7XG4gICAgICAgICAgICAgICAgbmV3VmFsdWUgPSB2YWx1ZS5jb25jYXQoW2FkZFZhbHVlXSk7XG4gICAgICAgICAgICB9IGVsc2UgaWYgKHZhbHVlLmxlbmd0aCA+IDEpIHtcbiAgICAgICAgICAgICAgICAvLyBVbnNlbGVjdCBpZiBtYW55IG9wdGlvbnMgYXJlIHNlbGVjdGVkXG4gICAgICAgICAgICAgICAgbmV3VmFsdWUuc3BsaWNlKFxuICAgICAgICAgICAgICAgICAgICBuZXdWYWx1ZS5pbmRleE9mKGFkZFZhbHVlKSxcbiAgICAgICAgICAgICAgICAgICAgMVxuICAgICAgICAgICAgICAgICk7XG4gICAgICAgICAgICB9XG5cbiAgICAgICAgICAgIG5ld1N0YXRlID0ge1xuICAgICAgICAgICAgICAgIHZhbHVlOiBuZXdWYWx1ZVxuICAgICAgICAgICAgfTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIG5ld1ZhbHVlID0gYWRkVmFsdWU7XG5cbiAgICAgICAgICAgIG5ld1N0YXRlID0ge1xuICAgICAgICAgICAgICAgIHZhbHVlOiAgYWRkVmFsdWUsXG4gICAgICAgICAgICAgICAgb3BlbmVkOiBmYWxzZVxuICAgICAgICAgICAgfTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMuc2V0U3RhdGUoXG4gICAgICAgICAgICBuZXdTdGF0ZSxcbiAgICAgICAgICAgICgpID0+IHtcbiAgICAgICAgICAgICAgICBpZiAob25DaGFuZ2UpIHtcbiAgICAgICAgICAgICAgICAgICAgb25DaGFuZ2UobmV3VmFsdWUpO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgKTtcbiAgICB9LFxuXG4gICAgLyoqXG4gICAgICogR2V0IGN1cnJlbnQgdmFsdWUgYXMgYSBzdHJpbmcgKGZvciBoaWRkZW4gaW5wdXQpXG4gICAgICogQHJldHVybiB7U3RyaW5nfVxuICAgICAqL1xuICAgIGdldFN0cmluZ1ZhbHVlKCkge1xuICAgICAgICBjb25zdCB7IHJlbmRlclRvU3RyaW5nIH0gPSB0aGlzLnByb3BzO1xuICAgICAgICBjb25zdCB7IHZhbHVlIH0gPSB0aGlzLnN0YXRlO1xuXG4gICAgICAgIGlmICghdmFsdWUpIHtcbiAgICAgICAgICAgIHJldHVybiAnJztcbiAgICAgICAgfVxuXG4gICAgICAgIGlmICghdGhpcy5wcm9wcy5tdWx0aXBsZSkge1xuICAgICAgICAgICAgcmV0dXJuIHJlbmRlclRvU3RyaW5nKHZhbHVlKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIHJldHVybiB2YWx1ZVxuICAgICAgICAgICAgLm1hcChyZW5kZXJUb1N0cmluZylcbiAgICAgICAgICAgIC5qb2luKHRoaXMucHJvcHMuZGVsaW1pdGVyKTtcbiAgICAgICAgfVxuICAgIH0sXG5cbiAgICAvKipcbiAgICAgKiBDaGVjayBpZiBhIHZhbHVlIGlzIHNlbGVjdGVkXG4gICAgICogQHBhcmFtIHtTdHJpbmd9IHZhbHVlXG4gICAgICogQHJldHVybiB7Qm9vbGVhbn1cbiAgICAgKi9cbiAgICBoYXNWYWx1ZSh2YWx1ZSkge1xuICAgICAgICBjb25zdCBjdXJyZW50VmFsdWUgPSB0aGlzLnN0YXRlLnZhbHVlO1xuXG4gICAgICAgIGlmICghdGhpcy5wcm9wcy5tdWx0aXBsZSkge1xuICAgICAgICAgICAgcmV0dXJuIChjdXJyZW50VmFsdWUgPT09IHZhbHVlKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIHJldHVybiAoY3VycmVudFZhbHVlLmluZGV4T2YodmFsdWUpID49IDApO1xuICAgICAgICB9XG4gICAgfSxcblxuICAgIC8qKlxuICAgICAqIEZvY3VzIHRoZSBzZWFyY2ggaW5wdXRcbiAgICAgKi9cbiAgICBmb2N1c1NlYXJjaCgpIHtcbiAgICAgICAgY29uc3QgeyBzZWFyY2hJbnB1dCB9ID0gdGhpcy5yZWZzO1xuICAgICAgICBpZiAoIXNlYXJjaElucHV0KSB7XG4gICAgICAgICAgICByZXR1cm47XG4gICAgICAgIH1cblxuICAgICAgICBzZWFyY2hJbnB1dC5mb2N1cygpO1xuICAgIH0sXG5cbiAgICAvKipcbiAgICAgKiBSZW5kZXIgYnV0dG9uIHRvIG9wZW4gc2VsZWN0XG4gICAgICovXG4gICAgcmVuZGVyQnV0dG9uKCkge1xuICAgICAgICBsZXQgeyBkaXNhYmxlZCwgYmxvY2ssIG11bHRpcGxlLCBwbGFjZWhvbGRlciB9ID0gdGhpcy5wcm9wcztcbiAgICAgICAgbGV0IHsgdmFsdWUsIG9wZW5lZCB9ID0gdGhpcy5zdGF0ZTtcbiAgICAgICAgbGV0IENvbXBvbmVudFNlbGVjdGlvbiA9IHRoaXMucHJvcHMuY29tcG9uZW50U2VsZWN0aW9uIHx8IHRoaXMucHJvcHMuY29tcG9uZW50O1xuXG4gICAgICAgIGxldCBpbm5lcjtcblxuICAgICAgICBpZiAodmFsdWUpIHtcbiAgICAgICAgICAgIGNvbnN0IHZhbHVlcyA9IG11bHRpcGxlID8gdmFsdWUgOiBbdmFsdWVdO1xuICAgICAgICAgICAgaW5uZXIgPSAoXG4gICAgICAgICAgICAgICAgPHNwYW4gY2xhc3NOYW1lPVwiU2VsZWN0U2VsZWN0aW9uc1wiPlxuICAgICAgICAgICAgICAgIHt2YWx1ZXMubWFwKCh2YWwsIGkpID0+IChcbiAgICAgICAgICAgICAgICAgICAgPHNwYW4ga2V5PXtpfSBjbGFzc05hbWU9XCJTZWxlY3RTZWxlY3Rpb25cIj5cbiAgICAgICAgICAgICAgICAgICAgICAgIDxDb21wb25lbnRTZWxlY3Rpb24gb3B0aW9uPXt2YWx9IGluZGV4PXtpfSAvPlxuICAgICAgICAgICAgICAgICAgICA8L3NwYW4+XG4gICAgICAgICAgICAgICAgKSl9XG4gICAgICAgICAgICAgICAgPC9zcGFuPlxuICAgICAgICAgICAgKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIGlubmVyID0gPHNwYW4gY2xhc3NOYW1lPVwiU2VsZWN0UGxhY2Vob2xkZXJcIj57cGxhY2Vob2xkZXJ9PC9zcGFuPjtcbiAgICAgICAgfVxuXG4gICAgICAgIHJldHVybiAoXG4gICAgICAgICAgICA8QnV0dG9uIHNpemU9e3RoaXMucHJvcHMuc2l6ZX0gYmxvY2s9e2Jsb2NrfSBkaXNhYmxlZD17ZGlzYWJsZWR9IGFjdGl2ZT17b3BlbmVkfSBvbkNsaWNrPXt0aGlzLm9uVG9nZ2xlfT5cbiAgICAgICAgICAgICAgICB7aW5uZXJ9IDxCdXR0b24uQ2FyZXQgLz5cbiAgICAgICAgICAgIDwvQnV0dG9uPlxuICAgICAgICApO1xuICAgIH0sXG5cbiAgICAvKipcbiAgICAgKiBSZW5kZXIgdGhlIGdyb3Vwc1xuICAgICAqL1xuICAgIHJlbmRlckdyb3VwcygpIHtcbiAgICAgICAgY29uc3QgeyBvcGVuZWQsIGdyb3VwcywgcXVlcnkgfSA9IHRoaXMuc3RhdGU7XG4gICAgICAgIGNvbnN0IHtcbiAgICAgICAgICAgIHNlYXJjaCxcbiAgICAgICAgICAgIGZpbHRlcixcbiAgICAgICAgICAgIHNlYXJjaEVtcHR5Q29tcG9uZW50OiBTZWFyY2hFbXB0eUNvbXBvbmVudCxcbiAgICAgICAgICAgIC4uLnByb3BzXG4gICAgICAgICB9ID0gdGhpcy5wcm9wcztcblxuICAgICAgICBjb25zdCBjbGFzc05hbWUgPSBjbGFzc05hbWVzKCdTZWxlY3RDb250YWluZXInLCB7XG4gICAgICAgICAgICAnb3Blbic6IG9wZW5lZFxuICAgICAgICB9KTtcblxuICAgICAgICAvLyBGaWx0ZXIgZW1wdHkgZ3JvdXBzIGJhc2VkIG9uIHNlYXJjaCBxdWVyeVxuICAgICAgICBjb25zdCBmaWx0ZXJlZEdyb3VwcyA9IGdyb3Vwcy5tYXAoKGdyb3VwKSA9PiAoe1xuICAgICAgICAgICAgLi4uZ3JvdXAsXG4gICAgICAgICAgICBvcHRpb25zOiBncm91cC5vcHRpb25zLmZpbHRlcigoaXRlbSwgaSkgPT4gZmlsdGVyKHF1ZXJ5LCBpdGVtLCBpKSlcbiAgICAgICAgfSkpXG4gICAgICAgIC5maWx0ZXIoKGdyb3VwKSA9PiBncm91cC5vcHRpb25zLmxlbmd0aCA+IDApO1xuXG4gICAgICAgIHJldHVybiAoXG4gICAgICAgICAgICA8ZGl2IGNsYXNzTmFtZT17Y2xhc3NOYW1lfT5cbiAgICAgICAgICAgIHtzZWFyY2ggP1xuICAgICAgICAgICAgICAgIDxkaXYgY2xhc3NOYW1lPVwiU2VsZWN0U2VhcmNoXCI+XG4gICAgICAgICAgICAgICAgICAgIDxJbnB1dCByZWY9XCJzZWFyY2hJbnB1dFwiXG4gICAgICAgICAgICAgICAgICAgICAgICB2YWx1ZT17cXVlcnl9XG4gICAgICAgICAgICAgICAgICAgICAgICBvbkNoYW5nZT17dGhpcy5vblNlYXJjaENoYW5nZWR9XG4gICAgICAgICAgICAgICAgICAgICAgICBwbGFjZWhvbGRlcj17dGhpcy5wcm9wcy5wbGFjZWhvbGRlcn1cbiAgICAgICAgICAgICAgICAgICAgLz5cbiAgICAgICAgICAgICAgICA8L2Rpdj5cbiAgICAgICAgICAgICAgICA6IG51bGxcbiAgICAgICAgICAgIH1cblxuICAgICAgICAgICAge0Jvb2xlYW4oZmlsdGVyZWRHcm91cHMubGVuZ3RoKSA/XG4gICAgICAgICAgICAgICAgPGRpdiBjbGFzc05hbWU9XCJTZWxlY3RHcm91cHNcIj5cbiAgICAgICAgICAgICAgICB7ZmlsdGVyZWRHcm91cHMubWFwKChncm91cCwgaSkgPT4gKFxuICAgICAgICAgICAgICAgICAgICA8U2VsZWN0T3B0R3JvdXAgey4uLnByb3BzfSBrZXk9e2l9IGdyb3VwPXtncm91cH0gaGFzVmFsdWU9e3RoaXMuaGFzVmFsdWV9IG9uVG9nZ2xlT3B0aW9uPXt0aGlzLm9uVG9nZ2xlT3B0aW9ufSAvPlxuICAgICAgICAgICAgICAgICkpfVxuICAgICAgICAgICAgICAgIDwvZGl2PlxuICAgICAgICAgICAgICAgIDogbnVsbFxuICAgICAgICAgICAgfVxuXG4gICAgICAgICAgICB7c2VhcmNoICYmICFmaWx0ZXJlZEdyb3Vwcy5sZW5ndGggP1xuICAgICAgICAgICAgICAgIDxkaXYgY2xhc3NOYW1lPVwiU2VhcmNoRW1wdHlcIj5cbiAgICAgICAgICAgICAgICAgICAgPFNlYXJjaEVtcHR5Q29tcG9uZW50IHF1ZXJ5PXtxdWVyeX0gLz5cbiAgICAgICAgICAgICAgICA8L2Rpdj5cbiAgICAgICAgICAgICAgICA6IG51bGxcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIDwvZGl2PlxuICAgICAgICApO1xuICAgIH0sXG5cbiAgICByZW5kZXIoKSB7XG4gICAgICAgIGxldCB7IG5hbWUsIGJsb2NrIH0gPSB0aGlzLnByb3BzO1xuICAgICAgICBjb25zdCB7IG9wZW5lZCB9ID0gdGhpcy5zdGF0ZTtcblxuICAgICAgICBsZXQgY2xhc3NOYW1lID0gY2xhc3NOYW1lcygnU2VsZWN0Rm9ybUNvbnRyb2wnLCB7XG4gICAgICAgICAgICBibG9ja1xuICAgICAgICB9KTtcblxuICAgICAgICByZXR1cm4gKFxuICAgICAgICAgICAgPGRpdiBjbGFzc05hbWU9e2NsYXNzTmFtZX0gb25DbGljaz17KGUpID0+IGUuc3RvcFByb3BhZ2F0aW9uKCl9PlxuICAgICAgICAgICAgICAgIDxpbnB1dCB0eXBlPVwiaGlkZGVuXCIgbmFtZT17bmFtZX0gdmFsdWU9e3RoaXMuZ2V0U3RyaW5nVmFsdWUoKX0gLz5cbiAgICAgICAgICAgICAgICB7dGhpcy5yZW5kZXJCdXR0b24oKX1cbiAgICAgICAgICAgIHtvcGVuZWQgP1xuICAgICAgICAgICAgICAgIDxCYWNrZHJvcCBvbkNsb3NlPXt0aGlzLmNsb3NlfT5cbiAgICAgICAgICAgICAgICAgICAge3RoaXMucmVuZGVyR3JvdXBzKCl9XG4gICAgICAgICAgICAgICAgPC9CYWNrZHJvcD5cbiAgICAgICAgICAgICAgICA6IG51bGxcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIDwvZGl2PlxuICAgICAgICApO1xuICAgIH1cbn0pO1xuXG5tb2R1bGUuZXhwb3J0cyA9IFNlbGVjdDtcbiJdfQ==