'use strict';

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

function _objectWithoutProperties(obj, keys) { var target = {}; for (var i in obj) { if (keys.indexOf(i) >= 0) continue; if (!Object.prototype.hasOwnProperty.call(obj, i)) continue; target[i] = obj[i]; } return target; }

var React = require('react');
var classNames = require('classnames');

var Button = require('./Button');
var Icon = require('./Icon');
var Backdrop = require('./Backdrop');
var warning = require('./utils/warning');

/**
 * Dropdown (or up). Automatically bound to child Button.
 * See ButtonDropdown implementation to wrap something else than a button.
 *
 * <Dropdown>
 *
 *     <Button ... />
 *
 *     <Dropdown.Item header>Category 1</Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *
 *     <Dropdown.Item divider />
 *     <Dropdown.Item header>Category 2</Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *
 * </Dropdown>
 */

var ButtonDropdown = React.createClass({
    displayName: 'ButtonDropdown',

    propTypes: {
        className: React.PropTypes.string,
        children: React.PropTypes.node,
        up: React.PropTypes.bool,
        size: React.PropTypes.string,
        scroll: React.PropTypes.bool
    },

    getDefaultProps: function getDefaultProps() {
        return {
            scroll: false
        };
    },
    getInitialState: function getInitialState() {
        return {
            open: false
        };
    },


    /**
     * Toggle the dopdown
     * @param  {Event} e?
     */
    toggle: function toggle(e) {
        if (e) {
            e.stopPropagation();
        }

        this.setState({
            open: !this.state.open
        });
    },


    /**
     * Close the dropdown
     */
    close: function close() {
        this.setState({
            open: false
        });
    },
    render: function render() {
        var that = this;
        var inner = [];
        var items = [];

        var _props = this.props,
            className = _props.className,
            children = _props.children,
            up = _props.up,
            size = _props.size,
            scroll = _props.scroll,
            otherProps = _objectWithoutProperties(_props, ['className', 'children', 'up', 'size', 'scroll']);

        var open = this.state.open;


        className = classNames('dropdown', className, {
            'dropup': up
        });

        inner = React.Children.map(children, function (child) {
            // If the Button is connected through Redux.connect, it is
            // renamed to "Connect(Button...)"
            if (child && child.type && child.type.displayName && child.type.displayName.indexOf('Button') !== -1) {
                if (!child.props.onClick && !child.props.href) {
                    return React.cloneElement(child, {
                        onClick: that.toggle,
                        dropdownToggle: true
                    });
                }
                return child;
            }

            return null;
        });

        items = React.Children.map(children, function (child) {
            if (child && child.type && (child.type.displayName == 'DropdownItem' || child.type.displayName == 'DropdownDivider')) {
                return React.cloneElement(child, {
                    onClick: function onClick() {
                        if (child.props.onClick) {
                            child.props.onClick();
                        }
                        that.close();
                    }
                });
            }
            return null;
        });

        var content = React.createElement(
            Button.Group,
            _extends({}, otherProps, { className: className }),
            inner
        );

        // Wrap in a backdrop when open
        if (open) {
            return React.createElement(
                Backdrop,
                { wrapper: content, onClose: this.close, scroll: true },
                React.createElement(
                    DropdownMenu,
                    { size: size, scroll: scroll },
                    items
                )
            );
        } else {
            return content;
        }
    }
});

/**
 * An item in the dropdown.
 * @type {[type]}
 */
var DropdownItem = React.createClass({
    displayName: 'DropdownItem',

    propTypes: {
        children: React.PropTypes.node,
        onClick: React.PropTypes.func,
        href: React.PropTypes.string,
        disabled: React.PropTypes.bool,
        divider: React.PropTypes.bool,
        header: React.PropTypes.bool,
        checked: React.PropTypes.bool
    },

    onClick: function onClick(e) {
        if (!this.props.href) {
            e.preventDefault();
            e.stopPropagation();

            if (this.props.onClick) this.props.onClick();
        }
    },
    isInner: function isInner(child) {
        return !child || !child.type || child.type.displayName !== 'DropdownMenu';
    },
    render: function render() {
        var _props2 = this.props,
            children = _props2.children,
            divider = _props2.divider,
            disabled = _props2.disabled,
            header = _props2.header,
            checked = _props2.checked,
            href = _props2.href;


        if (divider) {
            warning('Prop "divider" on Dropdown.Item is deprecated, use Dropdown.Divider instead');
            return React.createElement(DropdownDivider, null);
        }
        if (header) {
            warning('Prop "header" on Dropdown.Item is deprecated, use Dropdown.Header instead');
            return React.createElement(
                DropdownHeader,
                null,
                children
            );
        }

        var inner = [],
            outer = [];

        inner = React.Children.map(children, function (child) {
            if (this.isInner(child)) return child;
            return null;
        }, this);

        outer = React.Children.map(children, function (child) {
            if (!this.isInner(child)) return child;
            return null;
        }, this);

        return React.createElement(
            'li',
            { className: disabled ? 'disabled' : '' },
            React.createElement(
                'a',
                _extends({}, this.props, { href: href || '#', onClick: disabled ? null : this.onClick }),
                checked ? React.createElement(
                    'div',
                    { className: 'dropdown-icon pull-left' },
                    React.createElement(Icon, { id: 'check' })
                ) : '',
                inner
            ),
            outer
        );
    }
});

/**
 * A divider in the dropdown items.
 * @type {ReactClass}
 */
var DropdownDivider = React.createClass({
    displayName: 'DropdownDivider',
    render: function render() {
        return React.createElement('li', { className: 'divider' });
    }
});

/**
 * An header in the dropdown items
 * @type {ReactClass}
 */
var DropdownHeader = React.createClass({
    displayName: 'DropdownHeader',

    propTypes: {
        children: React.PropTypes.node
    },

    render: function render() {
        var children = this.props.children;

        return React.createElement(
            'li',
            { className: 'dropdown-header' },
            children
        );
    }
});

/**
 * Container for the dropdown menu.
 * @type {ReactClass}
 */
var DropdownMenu = React.createClass({
    displayName: 'DropdownMenu',

    propTypes: {
        className: React.PropTypes.string,
        children: React.PropTypes.node,
        size: React.PropTypes.string,
        // Auto-scroll in the dropdown ?
        scroll: React.PropTypes.bool
    },

    getDefaultProps: function getDefaultProps() {
        return {
            size: 'sm',
            scroll: true
        };
    },
    getInitialState: function getInitialState() {
        return {
            maxHeight: null
        };
    },


    /**
     * Detect the max height for this dropdown according to position on screen.
     */
    detectMaxSize: function detectMaxSize() {
        var scroll = this.props.scroll;
        var container = this.refs.container;


        if (!scroll) {
            return;
        }

        var rect = container.getBoundingClientRect();
        var maxHeight = window.innerHeight - rect.top - 30;

        this.setState({
            maxHeight: maxHeight
        });
    },
    componentDidMount: function componentDidMount() {
        this.detectMaxSize();
    },
    componentDidUpdate: function componentDidUpdate(prevProps) {
        var hasChanged = prevProps.children != this.props.children;

        if (hasChanged) {
            this.detectMaxSize();
        }
    },
    render: function render() {
        var _props3 = this.props,
            children = _props3.children,
            size = _props3.size;
        var maxHeight = this.state.maxHeight;

        var className = classNames('dropdown-menu', 'dropdown-' + size);

        return React.createElement(
            'div',
            { ref: 'container', className: className },
            React.createElement(
                'ul',
                { style: { maxHeight: maxHeight } },
                children
            )
        );
    }
});

var ItemHeader = React.createClass({
    displayName: 'ItemHeader',

    propTypes: {
        children: React.PropTypes.node
    },

    render: function render() {
        return React.createElement(
            'div',
            { className: 'dropdown-itemheader' },
            this.props.children
        );
    }
});

var ItemDesc = React.createClass({
    displayName: 'ItemDesc',

    propTypes: {
        children: React.PropTypes.node
    },

    render: function render() {
        return React.createElement(
            'div',
            { className: 'dropdown-itemdesc' },
            this.props.children
        );
    }
});

module.exports = ButtonDropdown;
module.exports.Divider = DropdownDivider;
module.exports.Header = DropdownHeader;
module.exports.Item = DropdownItem;
module.exports.Item.Header = ItemHeader;
module.exports.Item.Desc = ItemDesc;
module.exports.Menu = DropdownMenu;
//# sourceMappingURL=data:application/json;base64,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