'use strict';

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

function _objectWithoutProperties(obj, keys) { var target = {}; for (var i in obj) { if (keys.indexOf(i) >= 0) continue; if (!Object.prototype.hasOwnProperty.call(obj, i)) continue; target[i] = obj[i]; } return target; }

var React = require('react');
var classNames = require('classnames');

var Button = require('./Button');
var Icon = require('./Icon');
var Backdrop = require('./Backdrop');
var warning = require('./utils/warning');

/**
 * Dropdown (or up). Automatically bound to child Button.
 * See ButtonDropdown implementation to wrap something else than a button.
 *
 * <Dropdown>
 *
 *     <Button ... />
 *
 *     <Dropdown.Item header>Category 1</Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *
 *     <Dropdown.Item divider />
 *     <Dropdown.Item header>Category 2</Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *
 * </Dropdown>
 */

var ButtonDropdown = React.createClass({
    displayName: 'ButtonDropdown',

    propTypes: {
        className: React.PropTypes.string,
        children: React.PropTypes.node,
        up: React.PropTypes.bool,
        size: React.PropTypes.string,
        scroll: React.PropTypes.bool
    },

    getDefaultProps: function getDefaultProps() {
        return {
            scroll: false
        };
    },
    getInitialState: function getInitialState() {
        return {
            open: false
        };
    },


    /**
     * Toggle the dopdown
     * @param  {Event} e?
     */
    toggle: function toggle(e) {
        if (e) {
            e.stopPropagation();
        }

        this.setState({
            open: !this.state.open
        });
    },


    /**
     * Close the dropdown
     */
    close: function close() {
        this.setState({
            open: false
        });
    },
    render: function render() {
        var that = this;
        var inner = [];
        var items = [];

        var _props = this.props,
            className = _props.className,
            children = _props.children,
            up = _props.up,
            size = _props.size,
            scroll = _props.scroll,
            otherProps = _objectWithoutProperties(_props, ['className', 'children', 'up', 'size', 'scroll']);

        var open = this.state.open;


        className = classNames('dropdown', className, {
            'dropup': up
        });

        inner = React.Children.map(children, function (child) {
            // If the Button is connected through Redux.connect, it is
            // renamed to "Connect(Button...)"
            if (child && child.type && child.type.displayName && child.type.displayName.indexOf('Button') !== -1) {
                if (!child.props.onClick && !child.props.href) {
                    return React.cloneElement(child, {
                        onClick: that.toggle,
                        dropdownToggle: true
                    });
                }
                return child;
            }

            return null;
        });

        items = React.Children.map(children, function (child) {
            if (child && child.type && (child.type.displayName == 'DropdownItem' || child.type.displayName == 'DropdownDivider')) {
                return React.cloneElement(child, {
                    onClick: function onClick() {
                        if (child.props.onClick) {
                            child.props.onClick();
                        }
                        that.close();
                    }
                });
            }
            return null;
        });

        var content = React.createElement(
            Button.Group,
            _extends({}, otherProps, { className: className }),
            inner
        );

        // Wrap in a backdrop when open
        if (open) {
            return React.createElement(
                Backdrop,
                { wrapper: content, onClose: this.close, scroll: true },
                React.createElement(
                    DropdownMenu,
                    { size: size, scroll: scroll },
                    items
                )
            );
        } else {
            return content;
        }
    }
});

/**
 * An item in the dropdown.
 * @type {[type]}
 */
var DropdownItem = React.createClass({
    displayName: 'DropdownItem',

    propTypes: {
        children: React.PropTypes.node,
        onClick: React.PropTypes.func,
        href: React.PropTypes.string,
        disabled: React.PropTypes.bool,
        divider: React.PropTypes.bool,
        header: React.PropTypes.bool,
        checked: React.PropTypes.bool
    },

    onClick: function onClick(event) {
        var _props2 = this.props,
            href = _props2.href,
            onClick = _props2.onClick;


        if (href) {
            return;
        }

        event.preventDefault();
        event.stopPropagation();

        if (onClick) {
            onClick();
        }
    },
    isInner: function isInner(child) {
        return !child || !child.type || child.type.displayName !== 'DropdownMenu';
    },
    render: function render() {
        var _props3 = this.props,
            children = _props3.children,
            divider = _props3.divider,
            disabled = _props3.disabled,
            header = _props3.header,
            checked = _props3.checked,
            href = _props3.href;


        if (divider) {
            warning('Prop "divider" on Dropdown.Item is deprecated, use Dropdown.Divider instead');
            return React.createElement(DropdownDivider, null);
        }
        if (header) {
            warning('Prop "header" on Dropdown.Item is deprecated, use Dropdown.Header instead');
            return React.createElement(
                DropdownHeader,
                null,
                children
            );
        }

        var inner = [],
            outer = [];

        inner = React.Children.map(children, function (child) {
            if (this.isInner(child)) return child;
            return null;
        }, this);

        outer = React.Children.map(children, function (child) {
            if (!this.isInner(child)) return child;
            return null;
        }, this);

        return React.createElement(
            'li',
            { className: disabled ? 'disabled' : '' },
            React.createElement(
                'a',
                _extends({}, this.props, { href: href || '#', onClick: disabled ? null : this.onClick }),
                checked ? React.createElement(
                    'div',
                    { className: 'dropdown-icon pull-left' },
                    React.createElement(Icon, { id: 'check' })
                ) : '',
                inner
            ),
            outer
        );
    }
});

/**
 * A divider in the dropdown items.
 * @type {ReactClass}
 */
var DropdownDivider = React.createClass({
    displayName: 'DropdownDivider',
    render: function render() {
        return React.createElement('li', { className: 'divider' });
    }
});

/**
 * An header in the dropdown items
 * @type {ReactClass}
 */
var DropdownHeader = React.createClass({
    displayName: 'DropdownHeader',

    propTypes: {
        children: React.PropTypes.node
    },

    render: function render() {
        var children = this.props.children;

        return React.createElement(
            'li',
            { className: 'dropdown-header' },
            children
        );
    }
});

/**
 * Container for the dropdown menu.
 * @type {ReactClass}
 */
var DropdownMenu = React.createClass({
    displayName: 'DropdownMenu',

    propTypes: {
        className: React.PropTypes.string,
        children: React.PropTypes.node,
        size: React.PropTypes.string,
        // Auto-scroll in the dropdown ?
        scroll: React.PropTypes.bool
    },

    getDefaultProps: function getDefaultProps() {
        return {
            size: 'sm',
            scroll: true
        };
    },
    getInitialState: function getInitialState() {
        return {
            maxHeight: null
        };
    },


    /**
     * Detect the max height for this dropdown according to position on screen.
     */
    detectMaxSize: function detectMaxSize() {
        var scroll = this.props.scroll;
        var container = this.refs.container;


        if (!scroll) {
            return;
        }

        var rect = container.getBoundingClientRect();
        var maxHeight = window.innerHeight - rect.top - 30;

        this.setState({
            maxHeight: maxHeight
        });
    },
    componentDidMount: function componentDidMount() {
        this.detectMaxSize();
    },
    componentDidUpdate: function componentDidUpdate(prevProps) {
        var hasChanged = prevProps.children != this.props.children;

        if (hasChanged) {
            this.detectMaxSize();
        }
    },
    render: function render() {
        var _props4 = this.props,
            children = _props4.children,
            size = _props4.size;
        var maxHeight = this.state.maxHeight;

        var className = classNames('dropdown-menu', 'dropdown-' + size);

        return React.createElement(
            'div',
            { ref: 'container', className: className },
            React.createElement(
                'ul',
                { style: { maxHeight: maxHeight } },
                children
            )
        );
    }
});

var ItemHeader = React.createClass({
    displayName: 'ItemHeader',

    propTypes: {
        children: React.PropTypes.node
    },

    render: function render() {
        return React.createElement(
            'div',
            { className: 'dropdown-itemheader' },
            this.props.children
        );
    }
});

var ItemDesc = React.createClass({
    displayName: 'ItemDesc',

    propTypes: {
        children: React.PropTypes.node
    },

    render: function render() {
        return React.createElement(
            'div',
            { className: 'dropdown-itemdesc' },
            this.props.children
        );
    }
});

module.exports = ButtonDropdown;
module.exports.Divider = DropdownDivider;
module.exports.Header = DropdownHeader;
module.exports.Item = DropdownItem;
module.exports.Item.Header = ItemHeader;
module.exports.Item.Desc = ItemDesc;
module.exports.Menu = DropdownMenu;
//# sourceMappingURL=data:application/json;base64,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