'use strict';

var React = require('react');
var classNames = require('classnames');

var Input = require('./Input');
var Spinner = require('./Spinner');

var KEYCODE_ENTER = 13;
var KEYCODE_UP = 38;
var KEYCODE_DOWN = 40;

var Autocomplete = React.createClass({
    displayName: 'Autocomplete',

    propTypes: {
        onFetch: React.PropTypes.func.isRequired,
        renderResult: React.PropTypes.func.isRequired,
        // Called when onEnter on the input (no result selected)
        onEnter: React.PropTypes.func.isRequired,
        onPaste: React.PropTypes.func,
        // Focus events
        onFocus: React.PropTypes.func,
        onBlur: React.PropTypes.func,
        // Control the event
        value: React.PropTypes.string,
        // Called when typing
        onChange: React.PropTypes.func.isRequired,
        // Called when selecting an entry
        onSelect: React.PropTypes.func.isRequired,
        // Render options
        placeholder: React.PropTypes.string,
        size: React.PropTypes.string
    },

    getInitialState: function getInitialState() {
        var _props = this.props,
            value = _props.value,
            onChange = _props.onChange;

        if (typeof value == 'string' && !onChange) {
            throw new Error('onChange should be passed to Autocomplete when value is passed');
        }

        return {
            value: value || '',
            cursor: null,
            loading: false,
            focused: false,
            results: []
        };
    },
    componentWillReceiveProps: function componentWillReceiveProps(nextProps) {
        var value = nextProps.value;

        this.updateValue(value);
    },

    /**
     * Update the value.
     */
    updateValue: function updateValue(value) {
        var _this = this;

        var prevValue = this.state.value;
        var onFetch = this.props.onFetch;

        if (prevValue == value) {
            return;
        }

        this.setState({
            value: value,
            loading: true
        });

        onFetch(value, function (results) {
            _this.setState({
                loading: false,
                results: results
            });
        });
    },

    /**
     * Typed value changed, we fetch the new autocomplete result
     */
    onInputChanged: function onInputChanged(e) {
        var onChange = this.props.onChange;
        var value = e.target.value;

        this.updateValue(value);

        if (onChange) {
            onChange(value);
        }
    },

    /**
     * User is focusing/blur the input
     */
    onFocusChanged: function onFocusChanged(isFocused) {
        var _props2 = this.props,
            onFocus = _props2.onFocus,
            onBlur = _props2.onBlur;

        if (isFocused && onFocus) {
            onFocus();
        } else if (onBlur) {
            onBlur();
        }

        this.setState({
            focused: isFocused
        });
    },

    /**
     * Submit value at cursor
     */
    onEnter: function onEnter() {
        var _state = this.state,
            cursor = _state.cursor,
            value = _state.value;
        var onEnter = this.props.onEnter;

        if (cursor >= 0) {
            this.onSelect(cursor);
        } else if (onEnter) {
            onEnter(value);
            this.setState({
                focused: false,
                cursor: null,
                results: [],
                value: ''
            });
        }
    },

    /**
     * Submit a value
     */
    onSelect: function onSelect(index) {
        var onSelect = this.props.onSelect;
        var _state2 = this.state,
            value = _state2.value,
            results = _state2.results;

        var selected = results[index];

        this.setState({
            cursor: null,
            results: [],
            value: ''
        });

        onSelect(value, selected);
    },

    /**
     * User pressed a key in text input
     */
    onKeyDown: function onKeyDown(e) {
        var _state3 = this.state,
            cursor = _state3.cursor,
            results = _state3.results;

        if (e.keyCode === KEYCODE_ENTER) {
            e.preventDefault();
            this.onEnter();
        } else if (e.keyCode === KEYCODE_DOWN) {
            e.preventDefault();
            cursor++;
        } else if (e.keyCode === KEYCODE_UP) {
            e.preventDefault();
            cursor--;
        }

        if (cursor >= results.length) {
            cursor = results.length - 1;
        }
        if (cursor < -1) {
            cursor = -1;
        }

        this.setState({
            cursor: cursor
        });
    },

    /**
     * Render the suggestions
     */
    renderResults: function renderResults() {
        var _this2 = this;

        var _state4 = this.state,
            results = _state4.results,
            value = _state4.value,
            cursor = _state4.cursor;

        var ResultComponent = this.props.renderResult;

        return React.createElement('div', { className: 'AutocompleteResults' }, results.map(function (result, i) {
            var isActive = i === cursor;

            return React.createElement(AutocompleteResult, { key: value + '-' + i, active: isActive,
                onClick: function onClick(e) {
                    return _this2.onSelect(i);
                } }, React.createElement(ResultComponent, { result: result, index: i, active: isActive }));
        }));
    },

    /**
     * Focus or blur the autocomplete
     */
    focus: function focus() {
        var input = this.refs.input;

        input.focus();
    },
    blur: function blur() {
        var input = this.refs.input;

        input.blur();
    },
    render: function render() {
        var _this3 = this;

        var _props3 = this.props,
            onPaste = _props3.onPaste,
            size = _props3.size,
            placeholder = _props3.placeholder;
        var _state5 = this.state,
            value = _state5.value,
            focused = _state5.focused,
            loading = _state5.loading,
            results = _state5.results;

        return React.createElement('div', { className: 'Autocomplete' }, React.createElement(Input, {
            ref: 'input',
            value: value,
            placeholder: placeholder,
            size: size,
            onChange: this.onInputChanged,
            onFocus: function onFocus(e) {
                return _this3.onFocusChanged(true);
            },
            onBlur: function onBlur(e) {
                return _this3.onFocusChanged(false);
            },
            onPaste: onPaste,
            onKeyDown: this.onKeyDown
        }), loading ? React.createElement(Spinner, { size: 'sm', centered: false }) : '', focused && results.length > 0 ? this.renderResults() : '');
    }
});

/**
 * Container for the results.
 * @type {ReactClass}
 */
var AutocompleteResult = React.createClass({
    displayName: 'AutocompleteResult',

    propTypes: {
        active: React.PropTypes.bool,
        onClick: React.PropTypes.func,
        children: React.PropTypes.node
    },

    render: function render() {
        var _props4 = this.props,
            active = _props4.active,
            children = _props4.children,
            onClick = _props4.onClick;

        return React.createElement('div', { className: classNames('AutocompleteResult', { active: active }),
            onMouseDown: onClick }, children);
    }
});

module.exports = Autocomplete;