'use strict';

var _extends = Object.assign || function (target) {
    for (var i = 1; i < arguments.length; i++) {
        var source = arguments[i];for (var key in source) {
            if (Object.prototype.hasOwnProperty.call(source, key)) {
                target[key] = source[key];
            }
        }
    }return target;
};

function _objectWithoutProperties(obj, keys) {
    var target = {};for (var i in obj) {
        if (keys.indexOf(i) >= 0) continue;if (!Object.prototype.hasOwnProperty.call(obj, i)) continue;target[i] = obj[i];
    }return target;
}

var React = require('react');
var ReactDOM = require('react-dom');
var classNames = require('classnames');

var Backdrop = require('./Backdrop');
var Dropdown = require('./Dropdown');

var MENU_RIGHT_SPACING = 300;
var MENU_BOTTOM_SPACING = 160;

/**
 * Helper to display a context menu.
 *
 *  <ContextMenu component={() => <ContextMenu.Menu></ContextMenu.Menu>}>
 *    Right click here !
 *  </ContextMenu>
 *
 * @type {ReactClass}
 */
var ContextMenu = React.createClass({
    displayName: 'ContextMenu',

    propTypes: {
        // Delay before opening the menu
        // It should be use when integrating the context menu with a textarea and a selection event
        delay: React.PropTypes.number,
        // Should it prevent the default menu ?
        preventDefault: React.PropTypes.bool,
        // Should it be propaged to parent ?
        stopPropagation: React.PropTypes.bool,
        // Component to render for the menu
        component: React.PropTypes.func.isRequired,
        children: React.PropTypes.node.isRequired
    },

    getDefaultProps: function getDefaultProps() {
        return {
            delay: 0,
            preventDefault: true,
            stopPropagation: true
        };
    },
    getInitialState: function getInitialState() {
        return {
            open: false,
            x: 0,
            y: 0,
            directionH: 'e',
            directionV: 's'
        };
    },

    /**
     * When user is opening context menu.
     */
    onOpen: function onOpen(event) {
        var _this = this;

        var _props = this.props,
            delay = _props.delay,
            preventDefault = _props.preventDefault,
            stopPropagation = _props.stopPropagation;

        if (preventDefault) {
            event.preventDefault();
        }
        if (stopPropagation) {
            event.stopPropagation();
        }

        var x = event.clientX;
        var y = event.clientY;

        if (delay) {
            setTimeout(function () {
                return _this.open(x, y);
            }, delay);
        } else {
            this.open(x, y);
        }
    },

    /**
     * Open the menu at position x,y
     */
    open: function open(x, y) {
        var width = window.innerWidth;
        var height = window.innerHeight;

        this.setState({
            open: true,
            x: x,
            y: y,
            directionH: x > width - MENU_RIGHT_SPACING ? 'w' : 'e',
            directionV: y > height - MENU_BOTTOM_SPACING ? 'n' : 's'
        });
    },

    /**
     * When user is closing the menu.
     */
    onClose: function onClose() {
        this.setState({
            open: false
        });
    },

    /**
     * Close the menu after any click on the window.
     * We wait next event loop to avoid hiding the menu before the click got propaged to it.
     */
    onWindowClick: function onWindowClick() {
        var _this2 = this;

        setTimeout(function () {
            return _this2.onClose();
        }, 1);
    },
    bindEvents: function bindEvents() {
        var el = ReactDOM.findDOMNode(this);
        el.addEventListener('contextmenu', this.onOpen, false);
        el.addEventListener('click', this.onWindowClick, false);
    },
    unbindEvents: function unbindEvents() {
        var el = ReactDOM.findDOMNode(this);
        el.removeEventListener('contextmenu', this.onOpen, false);
        el.removeEventListener('click', this.onWindowClick, false);
    },
    componentDidMount: function componentDidMount() {
        this.bindEvents();
    },
    componentWillUnmount: function componentWillUnmount() {
        this.unbindEvents();
    },
    componentWillUpdate: function componentWillUpdate() {
        this.unbindEvents();
    },
    componentDidUpdate: function componentDidUpdate() {
        this.bindEvents();
    },
    render: function render() {
        var _props2 = this.props,
            children = _props2.children,
            component = _props2.component,
            otherProps = _objectWithoutProperties(_props2, ['children', 'component']);

        var _state = this.state,
            open = _state.open,
            x = _state.x,
            y = _state.y,
            directionH = _state.directionH,
            directionV = _state.directionV;

        var inner = React.Children.only(children);

        if (!open) {
            return inner;
        }

        var menu = React.createElement(component, _extends({
            // Pass the current position if we use a specific menu component.
            mouseX: x,
            mouseY: y
        }, otherProps));

        return React.createElement(Backdrop, { wrapper: inner, onClose: this.onClose }, React.createElement('div', {
            className: classNames('ContextMenu', 'direction-' + directionH, 'direction-' + directionV),
            style: { top: y, left: x },
            onClick: this.onClick }, menu));
    }
});

module.exports = ContextMenu;
module.exports.Divider = Dropdown.Divider;
module.exports.Header = Dropdown.Header;
module.exports.Item = Dropdown.Item;
module.exports.Menu = Dropdown.Menu;