'use strict';

var _extends = Object.assign || function (target) {
    for (var i = 1; i < arguments.length; i++) {
        var source = arguments[i];for (var key in source) {
            if (Object.prototype.hasOwnProperty.call(source, key)) {
                target[key] = source[key];
            }
        }
    }return target;
};

function _objectWithoutProperties(obj, keys) {
    var target = {};for (var i in obj) {
        if (keys.indexOf(i) >= 0) continue;if (!Object.prototype.hasOwnProperty.call(obj, i)) continue;target[i] = obj[i];
    }return target;
}

var React = require('react');
var classNames = require('classnames');

var Button = require('./Button');
var Icon = require('./Icon');
var Backdrop = require('./Backdrop');
var warning = require('./utils/warning');

/**
 * Dropdown (or up). Automatically bound to child Button.
 * See ButtonDropdown implementation to wrap something else than a button.
 *
 * <Dropdown>
 *
 *     <Button ... />
 *
 *     <Dropdown.Item header>Category 1</Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *
 *     <Dropdown.Item divider />
 *     <Dropdown.Item header>Category 2</Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *
 * </Dropdown>
 */

var ButtonDropdown = React.createClass({
    displayName: 'ButtonDropdown',

    propTypes: {
        className: React.PropTypes.string,
        children: React.PropTypes.node,
        up: React.PropTypes.bool,
        center: React.PropTypes.bool,
        size: React.PropTypes.string,
        scroll: React.PropTypes.bool
    },

    getDefaultProps: function getDefaultProps() {
        return {
            scroll: false,
            up: false,
            center: false
        };
    },
    getInitialState: function getInitialState() {
        return {
            open: false
        };
    },

    /**
     * Toggle the dopdown
     * @param  {Event} e?
     */
    toggle: function toggle(e) {
        if (e) {
            e.stopPropagation();
        }

        this.setState({
            open: !this.state.open
        });
    },

    /**
     * Close the dropdown
     */
    close: function close() {
        this.setState({
            open: false
        });
    },
    render: function render() {
        var that = this;
        var inner = [];
        var items = [];

        var _props = this.props,
            className = _props.className,
            children = _props.children,
            up = _props.up,
            center = _props.center,
            size = _props.size,
            scroll = _props.scroll,
            otherProps = _objectWithoutProperties(_props, ['className', 'children', 'up', 'center', 'size', 'scroll']);

        var open = this.state.open;

        className = classNames('dropdown', className, {
            'dropup': up,
            'dropcenter': center
        });

        inner = React.Children.map(children, function (child) {
            // If the Button is connected through Redux.connect, it is
            // renamed to "Connect(Button...)"
            if (child && child.type && child.type.displayName && child.type.displayName.indexOf('Button') !== -1) {
                if (!child.props.onClick && !child.props.href) {
                    return React.cloneElement(child, {
                        onClick: that.toggle,
                        dropdownToggle: true
                    });
                }
                return child;
            }

            return null;
        });

        items = React.Children.map(children, function (child) {
            if (child && child.type && (child.type.displayName == 'DropdownItem' || child.type.displayName == 'DropdownDivider')) {
                return React.cloneElement(child, {
                    onClick: function onClick() {
                        if (child.props.onClick) {
                            child.props.onClick();
                        }
                        that.close();
                    }
                });
            }
            return null;
        });

        var content = React.createElement(Button.Group, _extends({}, otherProps, { className: className }), inner);

        // Wrap in a backdrop when open
        if (open) {
            return React.createElement(Backdrop, { wrapper: content, onClose: this.close, scroll: true }, React.createElement(DropdownMenu, { size: size, scroll: scroll }, items));
        } else {
            return content;
        }
    }
});

/**
 * An item in the dropdown.
 * @type {[type]}
 */
var DropdownItem = React.createClass({
    displayName: 'DropdownItem',

    propTypes: {
        children: React.PropTypes.node,
        onClick: React.PropTypes.func,
        href: React.PropTypes.string,
        disabled: React.PropTypes.bool,
        divider: React.PropTypes.bool,
        header: React.PropTypes.bool,
        checked: React.PropTypes.bool
    },

    onClick: function onClick(event) {
        var _props2 = this.props,
            href = _props2.href,
            onClick = _props2.onClick;

        if (href) {
            return;
        }

        event.preventDefault();
        event.stopPropagation();

        if (onClick) {
            onClick();
        }
    },
    isInner: function isInner(child) {
        return !child || !child.type || child.type.displayName !== 'DropdownMenu';
    },
    render: function render() {
        var _props3 = this.props,
            children = _props3.children,
            divider = _props3.divider,
            disabled = _props3.disabled,
            header = _props3.header,
            checked = _props3.checked,
            href = _props3.href;

        if (divider) {
            warning('Prop "divider" on Dropdown.Item is deprecated, use Dropdown.Divider instead');
            return React.createElement(DropdownDivider, null);
        }
        if (header) {
            warning('Prop "header" on Dropdown.Item is deprecated, use Dropdown.Header instead');
            return React.createElement(DropdownHeader, null, children);
        }

        var inner = [],
            outer = [];

        inner = React.Children.map(children, function (child) {
            if (this.isInner(child)) return child;
            return null;
        }, this);

        outer = React.Children.map(children, function (child) {
            if (!this.isInner(child)) return child;
            return null;
        }, this);

        return React.createElement('li', { className: disabled ? 'disabled' : '' }, React.createElement('a', _extends({}, this.props, { href: href || '#', onClick: disabled ? null : this.onClick }), checked ? React.createElement('div', { className: 'dropdown-icon pull-left' }, React.createElement(Icon, { id: 'check' })) : '', inner), outer);
    }
});

/**
 * A divider in the dropdown items.
 * @type {ReactClass}
 */
var DropdownDivider = React.createClass({
    displayName: 'DropdownDivider',
    render: function render() {
        return React.createElement('li', { className: 'divider' });
    }
});

/**
 * An header in the dropdown items
 * @type {ReactClass}
 */
var DropdownHeader = React.createClass({
    displayName: 'DropdownHeader',

    propTypes: {
        children: React.PropTypes.node
    },

    render: function render() {
        var children = this.props.children;

        return React.createElement('li', { className: 'dropdown-header' }, children);
    }
});

/**
 * Container for the dropdown menu.
 * @type {ReactClass}
 */
var DropdownMenu = React.createClass({
    displayName: 'DropdownMenu',

    propTypes: {
        className: React.PropTypes.string,
        children: React.PropTypes.node,
        size: React.PropTypes.string,
        // Auto-scroll in the dropdown ?
        scroll: React.PropTypes.bool
    },

    getDefaultProps: function getDefaultProps() {
        return {
            size: 'sm',
            scroll: true
        };
    },
    getInitialState: function getInitialState() {
        return {
            maxHeight: null
        };
    },

    /**
     * Detect the max height for this dropdown according to position on screen.
     */
    detectMaxSize: function detectMaxSize() {
        var scroll = this.props.scroll;
        var container = this.refs.container;

        if (!scroll) {
            return;
        }

        var rect = container.getBoundingClientRect();
        var maxHeight = window.innerHeight - rect.top - 30;

        this.setState({
            maxHeight: maxHeight
        });
    },
    componentDidMount: function componentDidMount() {
        this.detectMaxSize();
    },
    componentDidUpdate: function componentDidUpdate(prevProps) {
        var hasChanged = prevProps.children != this.props.children;

        if (hasChanged) {
            this.detectMaxSize();
        }
    },
    render: function render() {
        var _props4 = this.props,
            children = _props4.children,
            size = _props4.size;
        var maxHeight = this.state.maxHeight;

        var className = classNames('dropdown-menu', 'dropdown-' + size);

        return React.createElement('div', { ref: 'container', className: className }, React.createElement('ul', { style: { maxHeight: maxHeight } }, children));
    }
});

var ItemHeader = React.createClass({
    displayName: 'ItemHeader',

    propTypes: {
        children: React.PropTypes.node
    },

    render: function render() {
        return React.createElement('div', { className: 'dropdown-itemheader' }, this.props.children);
    }
});

var ItemDesc = React.createClass({
    displayName: 'ItemDesc',

    propTypes: {
        children: React.PropTypes.node
    },

    render: function render() {
        return React.createElement('div', { className: 'dropdown-itemdesc' }, this.props.children);
    }
});

module.exports = ButtonDropdown;
module.exports.Divider = DropdownDivider;
module.exports.Header = DropdownHeader;
module.exports.Item = DropdownItem;
module.exports.Item.Header = ItemHeader;
module.exports.Item.Desc = ItemDesc;
module.exports.Menu = DropdownMenu;